// This file is part of CAF, the C++ Actor Framework. See the file LICENSE in
// the main distribution directory for license terms and copyright or visit
// https://github.com/actor-framework/actor-framework/blob/master/LICENSE.

#define CAF_SUITE io.unpublish

#include "caf/config.hpp"

#include "io-test.hpp"

#include <atomic>
#include <memory>
#include <new>
#include <thread>

#include "caf/all.hpp"
#include "caf/io/all.hpp"

using namespace caf;

namespace {

struct suite_state {
  long dtors_called = 0;
  suite_state() = default;
};

using suite_state_ptr = std::shared_ptr<suite_state>;

class dummy : public event_based_actor {
public:
  dummy(actor_config& cfg, suite_state_ptr ssp)
    : event_based_actor(cfg), ssp_(std::move(ssp)) {
    // nop
  }

  ~dummy() override {
    ssp_->dtors_called += 1;
  }

  behavior make_behavior() override {
    return {[] {
      // nop
    }};
  }

private:
  suite_state_ptr ssp_;
};

struct fixture : point_to_point_fixture<> {
  fixture() {
    prepare_connection(mars, earth, "mars", 8080);
    ssp = std::make_shared<suite_state>();
  }

  ~fixture() {
    run();
    CHECK_EQ(ssp->dtors_called, 2);
  }

  suite_state_ptr ssp;
};

} // namespace

BEGIN_FIXTURE_SCOPE(fixture)

CAF_TEST(actors can become unpublished) {
  auto testee = mars.sys.spawn<dummy>(ssp);
  auto guard = detail::make_scope_guard([&] {
    // The MM holds a reference to this actor publishing it, so we need to kill
    // it manually.
    anon_send_exit(testee, exit_reason::user_shutdown);
  });
  loop_after_next_enqueue(mars);
  auto port = unbox(mars.mm.publish(testee, 8080));
  CAF_REQUIRE_EQUAL(port, 8080);
  MESSAGE("the middleman ignores invalid unpublish() calls");
  auto testee2 = mars.sys.spawn<dummy>(ssp);
  loop_after_next_enqueue(mars);
  auto res = mars.mm.unpublish(testee2, 8080);
  CHECK(!res && res.error() == sec::no_actor_published_at_port);
  anon_send_exit(testee2, exit_reason::user_shutdown);
  MESSAGE("after unpublishing an actor, remotes can no longer connect");
  loop_after_next_enqueue(mars);
  CHECK(mars.mm.unpublish(testee, 8080));
  // TODO: ideally, we'd check that remote actors in fact can no longer connect.
  //       However, the test multiplexer does not support "closing" connections
  //       and the remote_actor blocks forever.
  // run();
  // loop_after_next_enqueue(earth);
  // CHECK(!earth.mm.remote_actor("mars", 8080));
}

END_FIXTURE_SCOPE()
