/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package org.apache.directory.shared.kerberos.codec.etypeInfoEntry.actions;


import org.apache.directory.api.asn1.DecoderException;
import org.apache.directory.api.asn1.ber.grammar.GrammarAction;
import org.apache.directory.api.asn1.ber.tlv.BerValue;
import org.apache.directory.api.asn1.ber.tlv.IntegerDecoder;
import org.apache.directory.api.asn1.ber.tlv.IntegerDecoderException;
import org.apache.directory.api.asn1.ber.tlv.TLV;
import org.apache.directory.api.i18n.I18n;
import org.apache.directory.api.util.Strings;
import org.apache.directory.shared.kerberos.codec.etypeInfoEntry.ETypeInfoEntryContainer;
import org.apache.directory.shared.kerberos.codec.types.EncryptionType;
import org.apache.directory.shared.kerberos.components.ETypeInfoEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * The action used to store the ETYPE-INFO-ENTRY etype
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class StoreEType extends GrammarAction<ETypeInfoEntryContainer>
{
    /** The logger */
    private static final Logger LOG = LoggerFactory.getLogger( StoreEType.class );

    /** Speedup for logs */
    private static final boolean IS_DEBUG = LOG.isDebugEnabled();


    /**
     * Instantiates a new StoreEType action.
     */
    public StoreEType()
    {
        super( "ETYPE-INFO-ENTRY etype" );
    }


    /**
     * {@inheritDoc}
     */
    public void action( ETypeInfoEntryContainer eTypeInfoEntryContainer ) throws DecoderException
    {
        TLV tlv = eTypeInfoEntryContainer.getCurrentTLV();

        // The Length should not be null
        if ( tlv.getLength() == 0 )
        {
            LOG.error( I18n.err( I18n.ERR_01308_ZERO_LENGTH_TLV ) );

            // This will generate a PROTOCOL_ERROR
            throw new DecoderException( I18n.err( I18n.ERR_01309_EMPTY_TLV ) );
        }

        // The etype is an integer
        BerValue value = tlv.getValue();

        EncryptionType etype = null;
        ETypeInfoEntry etypeInfoEntry = eTypeInfoEntryContainer.getETypeInfoEntry();

        try
        {
            int eType = IntegerDecoder.parse( value );
            etype = EncryptionType.getTypeByValue( eType );

            etypeInfoEntry.setEType( etype );

            if ( IS_DEBUG )
            {
                LOG.debug( "etype : {}", etype );
            }

            // The next tag is optional, we can end here
            eTypeInfoEntryContainer.setGrammarEndAllowed( true );
        }
        catch ( IntegerDecoderException ide )
        {
            LOG.error( I18n.err( I18n.ERR_01310_INTEGER_DECODING_ERROR, Strings.dumpBytes( value.getData() ) ) );

            // This will generate a PROTOCOL_ERROR
            throw new DecoderException( ide.getMessage() );
        }
    }
}
