#!/usr/bin/env perl

use v5.24;
use strict;
use warnings;
use feature qw( say );
use utf8;
use Encode;
use Unicode::Normalize;

use FindBin;
use lib $FindBin::RealBin;

use Biber::CodePage qw( :DEFAULT );

my $CS_system = get_CS_system();
my $sys_is_utf8 = ($CS_system eq 'UTF-8') || ($CS_system eq 'CP65001');


#----------------------
say "===================\n-----**Tests of non-ASCII names on command line** ----\n";
#----------------------

say "\n===================================================";

my @failures = ();

# Cases that should always succeed, if biber is bug free, given
#  System is macOS, Linux, or Windows with CP1252
my @main = (
    'NFC-café.tex',
    );

# Cases that only work on a UTF-8 system, and fail for Windows with system CP1252. 
my @utf8_only = (
    'Banana🍌.tex',
    'Carrot-🥕.tex',
    'NFD-café.tex',
    'mix-NFCé-café.tex',
    );


#------------------------------

foreach (@main) { one_tex($_); }

if ($sys_is_utf8) {
    say "Have UTF-8 system, so I'll do UTF-8 specific test(s).";
    foreach (@utf8_only) { one_tex($_); }
}

say "\n===================================================";

one_bcf( NFC("Outé.bbl"), NFC("Logé"), NFC("résultats/NFC-café") );
if ($sys_is_utf8) {
    one_bcf( NFD("Outé.bbl"), NFD("Logé"), NFC("résultats/NFC-café") );
    one_bcf( NFC("Outαβψδü").NFD("ñ").".bbl", NFC("Logγρωè").NFD("ü"), NFC("résultats/NFC-café") );
}

say "\n===================================================";

if (@failures) {
    say "Failures on the following:";
    foreach (@failures) { say "  $_"; }
}
else {
    say "SUCCESS";
}

exit 1 + $#failures;

#----------------------

sub one_tex {
    # Assumes there is a suitable latexmkrc file.
    systemU( "latexmk", "-C", "-bibtex", @_ );
    my $ret = systemU( "latexmk", "-interaction=batchmode", "-pdf", "-bibtex", @_ );
    if ($ret) { push @failures, join( ' ', @_ ); }
    return $ret;
}

#----------------------

sub one_bcf {
    my ( $out, $log, $bcf ) = @_;
    my @out_log_files = ( $out, "$log.blg" );
    say "Running biber with out and log arguments @out_log_files";
    unlinkU(@out_log_files);
    my $ret = systemU( "biber-jcc", "--output-file", $out, "--logfile", $log, $bcf );
    foreach (@out_log_files) {
        if (! file_exist($_) ) { $ret = 1; say "File '$_' not made"; }
    }
    if ($ret) { push @failures, join( ' ', @_ ); }
    
    return $ret;
}

#----------------------

sub file_exist {
    return -e encode_CS_system(shift);
}

#----------------------
sub systemU {
    my @arg_bytes = map { encode_CS_system($_) } @_;
    say "Trying to run '@_'";
    return system @arg_bytes;
}

#----------------------

sub unlinkU {
    my @arg_bytes = map { encode_CS_system($_) } @_;
    say "Trying to unlink '@_'";
    return unlink @arg_bytes;
}

#----------------------
