/*=============================================================================

    This file is part of FLINT.

    FLINT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    FLINT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with FLINT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

=============================================================================*/
/******************************************************************************

    Copyright (C) 2013 William Hart

******************************************************************************/

#include "e-antic/nf_elem.h"

void _nf_elem_mul_red(nf_elem_t a, const nf_elem_t b, 
                                     const nf_elem_t c, const nf_t nf, int red)
{
   if (nf->flag & NF_LINEAR)
   {
      const fmpz * const bnum = LNF_ELEM_NUMREF(b);
      const fmpz * const bden = LNF_ELEM_DENREF(b);
      const fmpz * const cnum = LNF_ELEM_NUMREF(c);
      const fmpz * const cden = LNF_ELEM_DENREF(c);
      fmpz * const anum = LNF_ELEM_NUMREF(a);
      fmpz * const aden = LNF_ELEM_DENREF(a);
      
      fmpz_mul(anum, bnum, cnum);
      
      fmpz_mul(aden, bden, cden);
   } else if (nf->flag & NF_QUADRATIC)
   {
      const fmpz * const bnum = QNF_ELEM_NUMREF(b);
      const fmpz * const bden = QNF_ELEM_DENREF(b);
      const fmpz * const cnum = QNF_ELEM_NUMREF(c);
      const fmpz * const cden = QNF_ELEM_DENREF(c);
      fmpz * const anum = QNF_ELEM_NUMREF(a);
      fmpz * const aden = QNF_ELEM_DENREF(a);
      
      fmpz_mul(anum, bnum, cnum);
      fmpz_mul(anum + 1, bnum, cnum + 1);
      fmpz_addmul(anum + 1, bnum + 1, cnum);
      fmpz_mul(anum + 2, bnum + 1, cnum + 1);

      fmpz_mul(aden, bden, cden);

      if (red && !fmpz_is_zero(anum + 2))
      {
         fmpz * pnum = fmpq_poly_numref(nf->pol);
           
         if (nf->flag & NF_MONIC)
         {
            fmpz_submul(anum + 1, anum + 2, pnum + 1);
            fmpz_submul(anum, anum + 2, pnum);
         } else
         {
            fmpz * prod = _fmpz_vec_init(3);

            _fmpq_poly_scalar_mul_fmpq(prod, prod + 2, 
                                  pnum, pnum + 2, 2, anum + 2, aden);
            _fmpq_poly_sub_can(anum, aden, anum, aden, 2, prod, prod + 2, 2, 0);

            _fmpz_vec_clear(prod, 3);
         }

         fmpz_zero(anum + 2);
      }
   } else /* generic nf_elem */
   {
      const slong len1 = NF_ELEM(b)->length;
      const slong len2 = NF_ELEM(c)->length;
      const slong len = nf->pol->length;
      slong plen = len1 + len2 - 1;
      
      if (len1 == 0 || len2 == 0)
      {
         nf_elem_zero(a, nf);

         return;
      }

      if (len1 >= len2)
      {
         _fmpz_poly_mul(NF_ELEM_NUMREF(a), NF_ELEM_NUMREF(b), len1,
            NF_ELEM_NUMREF(c), len2);
      }
      else
      {
          _fmpz_poly_mul(NF_ELEM_NUMREF(a), NF_ELEM_NUMREF(c), len2,
             NF_ELEM_NUMREF(b), len1);
      }

      fmpz_mul(fmpq_poly_denref(NF_ELEM(a)), fmpq_poly_denref(NF_ELEM(b)),
         fmpq_poly_denref(NF_ELEM(c)));

      _fmpq_poly_set_length(NF_ELEM(a), plen);

      if (red && plen >= len)
      {
         if (nf->flag & NF_MONIC)
         {
            if (len <= NF_POWERS_CUTOFF)
            {
               _fmpz_poly_rem_powers_precomp(NF_ELEM_NUMREF(a), plen,
                  fmpq_poly_numref(nf->pol), len, nf->powers.zz->powers);

               _fmpq_poly_set_length(NF_ELEM(a), len - 1);
               _fmpq_poly_normalise(NF_ELEM(a));
              
            } else
            {
               fmpz * q = _fmpz_vec_init(plen - len + 1);
               fmpz * r = _fmpz_vec_init(plen);
               slong i;
               
               _fmpz_vec_set(r, NF_ELEM_NUMREF(a), plen);

#if __FLINT_RELEASE >= 20600
               _fmpz_poly_divrem(q, NF_ELEM_NUMREF(a), r, plen, 
                  fmpq_poly_numref(nf->pol), len, 0);
#else
               _fmpz_poly_divrem(q, NF_ELEM_NUMREF(a), r, plen,
                  fmpq_poly_numref(nf->pol), len);
#endif

               _fmpz_vec_clear(r, plen);
               _fmpz_vec_clear(q, plen - len + 1);
          
               for (i = len - 2; i >= 0 && fmpz_is_zero(NF_ELEM_NUMREF(a) + i); i--);
               NF_ELEM(a)->length = i + 1;      
            }
         }
         else
         {
            fmpq_poly_t t;
        
            if (len <= NF_POWERS_CUTOFF)
            {
               _fmpq_poly_rem_powers_precomp(NF_ELEM_NUMREF(a), 
                  fmpq_poly_denref(NF_ELEM(a)), plen,
                  fmpq_poly_numref(nf->pol), fmpq_poly_denref(nf->pol), 
                  len, nf->powers.qq->powers);

               _fmpq_poly_set_length(NF_ELEM(a), len - 1);
               _fmpq_poly_normalise(NF_ELEM(a));
            } else
            {
               fmpq_poly_init2(t, 2*len - 3);
        
               _fmpq_poly_rem(t->coeffs, t->den,
                  NF_ELEM(a)->coeffs, NF_ELEM(a)->den, plen, 
                  nf->pol->coeffs, nf->pol->den, len, nf->pinv.qq); 
           
               _fmpq_poly_set_length(t, len - 1);
               _fmpq_poly_normalise(t);
        
               fmpq_poly_swap(t, NF_ELEM(a));
               fmpq_poly_clear(t);
            }
         }
      }
   }
}

void _nf_elem_mul(nf_elem_t a, const nf_elem_t b, 
                                     const nf_elem_t c, const nf_t nf)
{
   _nf_elem_mul_red(a, b, c, nf, 1);
}

void nf_elem_mul_red(nf_elem_t a, const nf_elem_t b, 
                                     const nf_elem_t c, const nf_t nf, int red)
{
   nf_elem_t t;
   
   if (nf->flag & NF_LINEAR)
      _fmpq_mul(LNF_ELEM_NUMREF(a), LNF_ELEM_DENREF(a), 
                LNF_ELEM_NUMREF(b), LNF_ELEM_DENREF(b),
                LNF_ELEM_NUMREF(c), LNF_ELEM_DENREF(c));
   else
   {
      if (a == b || a == c)
      {
         nf_elem_init(t, nf);

         _nf_elem_mul_red(t, b, c, nf, red);
         nf_elem_swap(t, a, nf);

         nf_elem_clear(t, nf);
      }
      else
         _nf_elem_mul_red(a, b, c, nf, red);

      nf_elem_canonicalise(a, nf);
   }
}

void nf_elem_mul(nf_elem_t a, const nf_elem_t b, 
                                              const nf_elem_t c, const nf_t nf)
{
   nf_elem_mul_red(a, b, c, nf, 1);
}
