// Copyright Martin Dosch.
// Use of this source code is governed by the BSD-2-clause
// license that can be found in the LICENSE file.

package main

import (
	"bytes"
	"encoding/xml"
	"fmt"
	"log/slog"
	"net/http"
	"net/url"
	"os"
	"path/filepath"
	"regexp"
	"strconv"
	"strings"
	"time"

	"github.com/ProtonMail/gopenpgp/v3/crypto" // MIT License
	"github.com/beevik/etree"                  // BSD-2-clause
	"github.com/gabriel-vasile/mimetype"       // MIT License
	"github.com/xmppo/go-xmpp"                 // BSD-3-Clause
)

func httpUpload(client *xmpp.Client, iqc chan xmpp.IQ, disc chan xmpp.DiscoItems, drc chan xmpp.DiscoResult,
	jserver string, filePaths []string, timeout time.Duration, oxPrivKey *crypto.Key, keyRing *crypto.KeyRing,
) (urls []string, err error) {
	var uploadComponent string
	var maxFileSize int64

	// Query server for disco#items
	slog.Info("http-upload: querying disco items", "server", jserver)
	discoItems, err := getDiscoItems(client, disc, jserver)
	if err != nil {
		return
	}

	// Check the services reported by disco#items for the http upload service
	for _, r := range discoItems.Items {
		slog.Info("http-upload: querying disco info", "disco-item", r.Jid)
		discoResult, err := getDiscoInfo(client, drc, r.Jid)
		if err != nil {
			return urls, err
		}
		slog.Info("http-upload: checking identities for type = \"file\" and category = \"store\"")
		for _, identity := range discoResult.Identities {
			if identity.Type == "file" &&
				identity.Category == "store" {
				uploadComponent = r.Jid
				slog.Info("http-upload:", "upload component", uploadComponent)
				for _, x := range discoResult.X {
					for i, field := range x.Field {
						if field.Var == "max-file-size" {
							if i > 0 {
								if x.Field[i-1].Value[0] == nsHTTPUpload {
									maxFileSize, err = strconv.ParseInt(field.Value[0], 10, 64)
									if err != nil {
										return urls, fmt.Errorf("http-upload: error while checking server maximum http upload file size")
									}
									slog.Info("http-upload:", "max file size", maxFileSize)
									break
								}
							}
						}
					}
				}
				break
			}
			if uploadComponent != "" {
				break
			}
		}
		if uploadComponent != "" {
			break
		}
	}
	if uploadComponent == "" {
		return urls, fmt.Errorf("http-upload: no http upload component found")
	}
	for _, filePath := range filePaths {
		// Read file
		buffer, err := readFile(filePath)
		if err != nil {
			return urls, err
		}
		fileSize := buffer.Len()
		slog.Info("http-upload: checking file size:", "file", filePath, "size", fileSize)
		// Get mime type
		mimeType := mimetype.Detect(buffer.Bytes()).String()
		slog.Info("http-upload:", "mime type", mimeType)
		var mimeTypeEscaped bytes.Buffer
		xml.Escape(&mimeTypeEscaped, []byte(mimeType))
		if oxPrivKey != nil {
			buffer, err = legacyPGPEncryptFile(oxPrivKey, keyRing, buffer)
			if err != nil {
				return urls, err
			}
			fileSize = buffer.Len()
			slog.Info("http-upload: updated file size after encrypting:", "file", filePath, "size", fileSize)
		}
		// Get file name
		fileName := filepath.Base(filePath)
		if oxPrivKey != nil {
			fileName = getID() + ".pgp"
		}
		// Just use alphanumerical and some special characters for now
		// to work around https://github.com/xmppo/go-xmpp/issues/132
		reg := regexp.MustCompile(`[^a-zA-Z0-9\+\-\_\.]+`)
		fileNameEscaped := reg.ReplaceAllString(fileName, "_")
		slog.Info("http-upload: escape file", "name", fileNameEscaped)

		// Check if the file size doesn't exceed the maximum file size of the http upload
		// component if a maximum file size is reported, if not just continue and hope for
		// the best.
		if maxFileSize != 0 {
			if int64(fileSize) > maxFileSize {
				return urls, fmt.Errorf("http-upload: file size %s MiB is larger than the maximum file size allowed (%s MiB)",
					strconv.FormatInt(int64(fileSize)/1024/1024, 10), strconv.FormatInt(maxFileSize/1024/1024, 10))
			}
		}

		request := etree.NewDocument()
		request.WriteSettings.AttrSingleQuote = true
		requestReq := request.CreateElement("request")
		requestReq.CreateAttr("xmlns", nsHTTPUpload)
		requestReq.CreateAttr("filename", fileNameEscaped)
		requestReq.CreateAttr("size", fmt.Sprint(fileSize))
		requestReq.CreateAttr("content-type", mimeType)
		r, err := request.WriteToString()
		if err != nil {
			return urls, err
		}

		// Request http upload slot
		slog.Info("http-upload: requesting upload slot")
		uploadSlot, err := sendIQ(client, iqc, uploadComponent, "get", r)
		if err != nil {
			return urls, err
		}
		if uploadSlot.Type != strResult {
			return urls, fmt.Errorf("http-upload: error while requesting upload slot")
		}
		iqHTTPUploadSlotXML := etree.NewDocument()
		err = iqHTTPUploadSlotXML.ReadFromBytes(uploadSlot.Query)
		if err != nil {
			return urls, err
		}
		iqHTTPUploadSlotXMLSlot := iqHTTPUploadSlotXML.SelectElement("slot")
		if iqHTTPUploadSlotXMLSlot == nil {
			return urls, fmt.Errorf("http-upload: no slot element")
		}
		iqHTTPUploadSlotXMLPut := iqHTTPUploadSlotXMLSlot.SelectElement("put")
		if iqHTTPUploadSlotXMLPut == nil {
			return urls, fmt.Errorf("http-upload: no put element")
		}
		iqHTTPUploadSlotXMLPutURL := iqHTTPUploadSlotXMLPut.SelectAttr("url")
		if iqHTTPUploadSlotXMLPutURL == nil {
			return urls, fmt.Errorf("http-upload: no url attribute")
		}
		if !strings.HasPrefix(iqHTTPUploadSlotXMLPutURL.Value, "https://") {
			return urls, fmt.Errorf("http-upload: upload slot does not provide https")
		}
		// Upload file
		slog.Info("http-upload: uploading file to", "slot", iqHTTPUploadSlotXMLPutURL.Value)
		httpTransport := &http.Transport{
			IdleConnTimeout:     timeout,
			TLSHandshakeTimeout: timeout,
		}
		proxyEnv := os.Getenv("HTTP_PROXY")
		slog.Info("http-upload: getting environment variable:", "HTTP_PROXY", proxyEnv)
		if proxyEnv != "" {
			proxyURL, err := url.Parse(proxyEnv)
			if err != nil {
				return urls, err
			}
			slog.Info("http-upload: using http transport", "proxy", proxyURL)
			httpTransport.Proxy = http.ProxyURL(proxyURL)
		}
		httpClient := &http.Client{Transport: httpTransport}
		req, err := http.NewRequest(http.MethodPut, iqHTTPUploadSlotXMLPutURL.Value,
			buffer)
		if err != nil {
			return urls, err
		}
		req.Header.Set("Content-Type", mimeTypeEscaped.String())
		iqHTTPUploadSlotXMLPutHeaders := iqHTTPUploadSlotXMLPut.SelectElements("header")
		for _, h := range iqHTTPUploadSlotXMLPutHeaders {
			name := h.SelectAttr("name")
			if name == nil {
				continue
			}
			switch name.Value {
			case "Authorization", "Cookie", "Expires":
				req.Header.Set(name.Value, h.Text())
			}
		}
		resp, err := httpClient.Do(req)
		if err != nil {
			return urls, err
		}
		slog.Info("http-upload: received http", "status", resp.StatusCode)
		// Test for http status code "200 OK" or "201 Created"
		if resp.StatusCode != 200 && resp.StatusCode != 201 {
			return urls, fmt.Errorf("http-upload: upload failed")
		}

		// Return http link
		iqHTTPUploadSlotXMLGet := iqHTTPUploadSlotXMLSlot.SelectElement("get")
		if iqHTTPUploadSlotXMLGet == nil {
			return urls, fmt.Errorf("http-upload: no get element")
		}
		iqHTTPUploadSlotXMLGetURL := iqHTTPUploadSlotXMLGet.SelectAttr("url")
		if iqHTTPUploadSlotXMLGetURL == nil {
			return urls, fmt.Errorf("http-upload: no url attribute")
		}
		slog.Info("http-upload: received get", "URL", iqHTTPUploadSlotXMLGetURL.Value)
		err = resp.Body.Close()
		if err != nil {
			fmt.Println("http-upload: error while closing http request body:", err)
		}
		urls = append(urls, iqHTTPUploadSlotXMLGetURL.Value)
	}
	return urls, nil
}
