/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */



/*!
    \page lcmaps_verify_proxy.mod Verification plugin plugin
 
    \section posixsynopsis SYNOPSIS
    \b lcmaps_verify_proxy.mod
 
    \section posixdesc DESCRIPTION
 
The Verify Proxy module will verify the validity of the proxy (with its chain) or any other X.509 certificate chain that has been used in the LCMAPS framework.
The module will report to the LCMAPS framework the validity of the chain.

 
*/


/*!
    \file   lcmaps_verify_proxy.c
    \brief  Plugin to verify the proxy (and its chain) on its validity.
    \author Oscar Koeroo for the EGEE.
*/





/*****************************************************************************
                            Include header files
******************************************************************************/

#include "lcmaps_proxylifetime.h"

#include "verify-lib/interface/verify_x509.h"


#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <time.h>
#include <sys/types.h>

#include <openssl/x509.h>
#include <openssl/asn1.h>

#include <lcmaps/lcmaps_log.h>




/******************************************************************************
                                Definitions
******************************************************************************/

#define	DN_BUFFER_LEN	    256
 
/******************************************************************************
                          Module specific prototypes
******************************************************************************/

/**
 * Returns the TTL for given level in the list p or 0 when no match is found.
 */
static time_t Search_TTL_By_Level (struct TProxyLevelTTL * p, int level);


/**
 * Returns 1 when min(t1,t2) < time < max(t1,t2), otherwise returns 0.
 */
static int timeIsInBetween (time_t this, time_t t1, time_t t2);

/******************************************************************************
                       Public functions
******************************************************************************/

/**
 * Verifies whether proxy lifetimes in given chain of length depth, comply with
 * the policy at each level, as given by plt. 
 * Returns: 1 on succes, 0 failure
 */
int lcmaps_lifetime_verifyProxyLifeTime (struct TProxyLevelTTL * plt, STACK_OF(X509) *chain, int depth)
{
#ifdef __func__
    const char *logstr=__func__;
#else
    const char *logstr="lcmaps_lifetime_verifyProxyLifeTime";
#endif
    int         i = 0;
/*    int         depth = sk_X509_num (chain); */
    int         amount_of_CAs = 0;
    time_t      delta_time = 0;
   /*
    int         days  = 0;
    int         hours = 0;
    int         minutes = 0;
    */
    X509 *      cert = NULL;
    char        dn_buffer[DN_BUFFER_LEN];
    int         proxyLevel = 0;
    int         proxyType = 0;
    time_t      levelTime = 0;
    time_t      notAfter  = 0;
    time_t      notBefore = 0;



    /* How many CA certs are there in the chain? */
    for (i = 0; i < depth; i++)
    {
        if (verify_x509IsCA(sk_X509_value(chain, i)))
            amount_of_CAs++;
    }

    /* 
     * Changed this value to start checking the proxy and such
     * ->  This means I start working after the CA(s) and user_cert
     *
     * Start at proxyLevel 0
     *
     */

    for (i = depth - (amount_of_CAs + 2); i >= 0; i--)
    {
        lcmaps_log_debug (1, "%s: checking proxy level: %d of depth %d\n", logstr, i, depth);

        /* Init cycle */
        proxyType = UNDEFINED_PROXY;

        /* Check for X509 certificate and point to it with 'cert' */
        if ((cert = sk_X509_value(chain, i)) != NULL)
        {
	    /* The chain descend result in first being confronted with a
	     * MYPROXY_PROXY (not necessarily a proxy created with myproxy) */
            if (proxyLevel == 0) 
                proxyType = MYPROXY_PROXY; 
            else 
                proxyType = INNER_PROXY;

            /* Overruling the proxyType when it is a LEAF_PROXY (ending the chain) */
            if (i == 0)
                proxyType = LEAF_PROXY;

            X509_NAME_oneline(X509_get_subject_name(cert), dn_buffer, DN_BUFFER_LEN);
            lcmaps_log_debug (2, "%s: Current cert: %s\n", logstr, dn_buffer);
            
	    notAfter  = verify_asn1TimeToTimeT (X509_get_notAfter(cert));
	    if (notAfter==0)  {
		lcmaps_log(LOG_ERR,"%s: Cannot convert notAfter ASN1 string\n", logstr);
		goto failure;
	    }
            notBefore = verify_asn1TimeToTimeT (X509_get_notBefore(cert));
	    if (notBefore==0)  {
		lcmaps_log(LOG_ERR,"%s: Cannot convert notBefore ASN1 string\n", logstr);
		goto failure;
	    }


            /* Delta Time is the amount of seconds between the two fixed time points notAfter and notBefore */
            delta_time = notAfter - notBefore;

            lcmaps_log_debug (2, "%s: Valid time period (Proxy LifeTime): %ld hours, %ld minutes en %ld seconds\n", 
                                logstr,
                                delta_time / 3600, 
                                (delta_time % 3600) / 60, 
                                (delta_time % 3600) % 60);

           
 
            /* 
	     * Retrieve the maximum proxy life time at the current level from
	     * the internal data structure (created a plugin init stage)
             * 
             * var i is counting backwards to zero.
	     * if i == 0 then we are at the top of the chain or actually the
	     * lower ends of it, which is *always* indicated as the LEAF proxy
             * If i == 0 then we can safely retrieve the LEAF_PROXY proxyLevel info
             *
	     * NOTE: The proxylevel which is being evaluated WILL BE overriden
	     * by the set LEAF_PROXY policy except when there is no LEAF_PROXY
	     * policy set.
             */
            if (i == 0) 
            {
                if ((levelTime = Search_TTL_By_Level(plt, LEAF_PROXY))) 
                {
                    lcmaps_log_debug (1, "%s: Overruling specific Proxy Level maximum TTL with leaf proxy policy. Leaf proxy found at Proxy Level %d\n", logstr, proxyLevel);
                }
                else
                {
                    lcmaps_log_debug (2, "%s: No policy for LEAF Proxy at Proxy Level %d. Trying policy for the current Proxy Level\n", logstr, proxyLevel);
                    
                    /* switching back from LEAF Proxy policy mode to current proxyLevel */
                    if ((levelTime = Search_TTL_By_Level(plt, proxyLevel)))
                    {
                        lcmaps_log_debug (5, "%s: Successfully found config for Proxy level %d\n", logstr, proxyLevel);
                    }
                    else
                    {
                        lcmaps_log_debug (5, "%s:     No policy for Proxy level %d\n", logstr, proxyLevel);
                    }
                }
            }
            else
            {
                /* Search for a registered policy at proxyLevel */
                if ((levelTime = Search_TTL_By_Level(plt, proxyLevel)))
                {
                    lcmaps_log_debug (2, "%s: Successfully found config for Proxy level %d\n", logstr, proxyLevel);
                }
                else
                {
                    lcmaps_log_debug (2, "%s: No policy for Proxy level %d\n", logstr, proxyLevel);
                }
            }


            /* Proxy LifeTime Level Check,
             * is zero, then no levelTime has been set in the initialization phase
             */
            if (levelTime != 0)
            {
                lcmaps_log_debug (2, "%s: Max Leveltime @ proxyLevel %d and proxytype %s: %ld hours, %ld minutes and %ld seconds\n", 
                                     logstr,
                                     proxyLevel,
                                     proxyType == LEAF_PROXY ? "LEAF" : proxyType == INNER_PROXY ? "INNER" : proxyType == MYPROXY_PROXY ? "MYPROXY/FIRST" : "unknown type",
                                     levelTime / 3600, 
                                     (levelTime % 3600) / 60, 
                                     (levelTime % 3600) % 60);


                /* The level time is the _max_ time at that proxy level */
                if (delta_time <= levelTime)
                {
                    lcmaps_log_debug (1, "%s:     Proxy Life Time policy check approved at Proxy Level %d.\n", logstr, proxyLevel);
                }
                else
                {
                    time_t dt,diff_dt_lt;
		    time_t days, hours, min, sec;
                    time_t diff_days, diff_hours, diff_min, diff_sec;

                    dt = delta_time;
                    days = dt / (3600 * 24);
                    dt = dt % (3600 * 24);
                    hours = dt / 3600;
                    dt = dt % 3600;
                    min = dt / 60;
                    dt = dt % 60;
                    sec = dt;

                    diff_dt_lt = delta_time - levelTime;
                    diff_days = diff_dt_lt / (3600 * 24);
                    diff_dt_lt = diff_dt_lt % (3600 * 24);
                    diff_hours = diff_dt_lt / 3600;
                    diff_dt_lt = diff_dt_lt % 3600;
                    diff_min = diff_dt_lt / 60;
                    diff_dt_lt = diff_dt_lt % 60;
                    diff_sec = diff_dt_lt;

                    /* Access/Policy Violation! */
                    lcmaps_log(LOG_ERR, "%s: Error: Proxy Life Time Violation. Proxy at level %d has a life time of '%ld day(s), %ld hour(s), %ld min(s), %ld sec(s)' which exceed the policy by '%ld day(s), %ld hour(s), %ld min(s), %ld sec(s)'.\n", 
                                    logstr,
                                    proxyLevel,
                                    (long)days,
                                    (long)hours,
                                    (long)min,
                                    (long)sec,
                                    (long)diff_days,
                                    (long)diff_hours,
                                    (long)diff_min,
                                    (long)diff_sec);
                    goto failure;
                }
            }
            else
            {
                /* No policy set */
                lcmaps_log_debug (5, "%s: No Proxy LifeTime check performed on this proxy level.\n", logstr);
            }

            /* kick proxy certificate level up by one and clean up */
            proxyLevel++;
            levelTime = 0;
        }
    }   


/* Success! */
    return 1;


failure:
    return 0;
}


/**
 * Verifies whether the VOMS AC lifetimes in given lcmaps_vomsdata comply with
 * the policy for the VOMS Attributes' Life Time (only one policy).
 * Returns 1 on success, 0 on failure
 */
int lcmaps_lifetime_verifyVOMSLifeTime (struct TProxyLevelTTL * plt,
					lcmaps_vomsdata_t * lcmaps_vomsdata)
{
#ifdef __func__
    const char *logstr = __func__;
#else
    const char *logstr = "lcmaps_lifetime_verifyVOMSLifeTime";
#endif
    int         i = 0;
    time_t      levelTime = 0;

    time_t      start;
    time_t      end;
    time_t      delta_time;

    time_t      now = time((time_t *)NULL);

    /* Do we have Any VOMS Attribs? */
    if (lcmaps_vomsdata)
    {
        for (i = 0; i < lcmaps_vomsdata->nvoms; i++)
        {
            start = verify_str_asn1TimeToTimeT (lcmaps_vomsdata->voms[i].date1);
	    if (start==0)  {
		lcmaps_log(LOG_ERR,"%s: Cannot convert `start' ASN1 string from voms data\n", logstr);
		goto failure;
	    }
            end   = verify_str_asn1TimeToTimeT (lcmaps_vomsdata->voms[i].date2);
	    if (end==0)  {
		lcmaps_log(LOG_ERR,"%s: Cannot convert `end' ASN1 string from voms data\n", logstr);
		goto failure;
	    }

	    /* First sanity checks to see if the VOMS Attributes are valid at
	     * this moment in time */
            if (timeIsInBetween (now, start, end))
            {
		/* Performing VOMS LifeTime (Policy) check -- level '0' means
		 * the one and only VOMS Life Time Policy*/
                if ((levelTime = Search_TTL_By_Level(plt, 0)))
                {
                    /* lcmaps_log_debug (5, "\t%s: VOMS Attributes for the VO '%s' will now be checked against the VOMS LifeTime policy.\n", logstr, lcmaps_vomsdata->voms[i].voname); */
                    
                 
		    /* Delta Time is the amount of seconds between the two fixed
		     * time points notAfter and notBefore */
                    delta_time = end - start;


		    /* Check if the lifetime of the VOMS Attribs is bigger then
		     * the set policy (a bad thing) */
                    if (delta_time > levelTime)
                    {
                        lcmaps_log (LOG_NOTICE, "%s:     Error: Proxy Life Time Violation. The VOMS Attributes for the VO '%s' exceed the set VOMS LifeTime policy of '%ld hours, %ld minutes en %ld seconds' by '%ld hours, %ld minutes en %ld seconds'\n",
                                        logstr,
                                        lcmaps_vomsdata->voms[i].voname,
                                        (long)levelTime / 3600,
                                        (long)(levelTime % 3600) / 60,
                                        (long)(levelTime % 3600) % 60,
                                        (long)(delta_time - levelTime) / 3600,
                                        (long)((delta_time -levelTime) % 3600) / 60,
                                        (long)((delta_time -levelTime) % 3600) % 60
                                   );
                        lcmaps_log_debug (5, "%s: Lifetime of the VOMS Attributes for the VO '%s': %ld hours, %ld minutes en %ld seconds\n",
                                          logstr,
                                          lcmaps_vomsdata->voms[i].voname,
                                          (long)delta_time / 3600,
                                          (long)(delta_time % 3600) / 60,
                                          (long)(delta_time % 3600) % 60);

                        goto failure;
                    }
                    else
                    {
                        lcmaps_log_debug (3, "%s:     Ok: Lifetime of the VOMS Attributes for the VO '%s': '%ld hours, %ld minutes en %ld seconds'. The set policy for the VOMS LifeTime: '%ld hours, %ld minutes en %ld seconds.'\n",
                                          logstr,
                                          lcmaps_vomsdata->voms[i].voname,
                                          (long) delta_time / 3600,
                                          (long)(delta_time % 3600) / 60,
                                          (long)(delta_time % 3600) % 60,
                                          (long) levelTime / 3600,
                                          (long)(levelTime % 3600) / 60,
                                          (long)(levelTime % 3600) % 60);
   
                    }
                }
                else
                {
                    lcmaps_log_debug (1, "%s: No VOMS Attribute Lifetime policy set to enforce, skipping VOMS Lifetime check.\n", logstr);
                }
            }
            else
            {
                if (now < start )
                    lcmaps_log (LOG_ERR, "%s: VOMS Attributes for the VO '%s' are not valid yet!\n", logstr, lcmaps_vomsdata->voms[i].voname);

                if (now > end )
                    lcmaps_log (LOG_ERR, "%s: VOMS Attributes for the VO '%s' are not valid anymore!\n", logstr, lcmaps_vomsdata->voms[i].voname);
 
                goto failure;
            }
        }
    }
    else
    {
        lcmaps_log_debug (3, "%s: No LCMAPS VOMS Data found, VOMS checks do not apply.\n", logstr);
    }


    return 1;

failure:
    return 0;

}



/**
 * Pushes a new lifetime struct element with given level and TTL to the list of
 * lifetime structs. Returns 0 on success, -1 on error.
 */
int lcmaps_lifetime_Push_New_Entry (struct TProxyLevelTTL ** p,
				     int level, time_t ttl)
{
#ifdef __func__
    const char *logstr=__func__;
#else
    const char *logstr="lcmaps_lifetime_Push_New_Entry";
#endif
    struct TProxyLevelTTL * lp = *p;
    struct TProxyLevelTTL *new = NULL, *hand = NULL;


    /* if first entry */
    if (lp == NULL)
    {
        /* Clean alloc and fill with char '\0' */
        lp = (struct TProxyLevelTTL *) calloc ((size_t)1, sizeof (struct TProxyLevelTTL));
	if ( lp==NULL )	{
	    lcmaps_log(LOG_ERR,"%s: out of memory\n", logstr);
	    return -1;
	}

        lp->level = level;
        lp->ttl   = ttl;
        lp->next  = NULL;
    }
    else
    {
        
        new = (struct TProxyLevelTTL *) calloc ((size_t)1, sizeof (struct TProxyLevelTTL));
	if ( new==NULL )	{
	    lcmaps_log(LOG_ERR,"%s: out of memory\n", logstr);
	    return -1;
	}

        new->level = level;
        new->ttl   = ttl;
        new->next  = NULL;

        hand = lp;
        lp = new;
        new->next = hand;
    }

    *p = lp;

    return 0;
}


/**
 * Print TTL By Level
 */
void lcmaps_lifetime_Print_TTL_By_Level (struct TProxyLevelTTL * p)
{
    struct TProxyLevelTTL * curr = NULL;

    curr = p;

    while (curr != NULL)
    {
        lcmaps_log_debug(5, "Print_TTL_By_Level  Max TTL @ Level:  %ld seconds @ %d\n", (long)(curr->ttl), curr->level);
        curr = curr->next;
    }
}


/**
 * Frees the linked list of lifetime structs.
 */
void lcmaps_lifetime_FreeProxyLevelTTL (struct TProxyLevelTTL ** p)
{
    struct TProxyLevelTTL * curr = NULL;
    struct TProxyLevelTTL * hulp = NULL;

    curr = *p;
    *p  = NULL;

    while (curr)
    {
        hulp = curr;
        curr = curr->next;
        free (hulp);
    }
}


/**
 * Converts a string of a special time format to a time_t
 * parameter datetime: string in the format:  2d-13:37 which means a time period
 * of 2 days, 13 hours and 37 minutes
 * Returns: number of seconds on success or -1 on failure.
 */
time_t lcmaps_lifetime_ttl_char2time_t (char * datetime)
{
    /*
     * Allowed notation:  [2m-][2d-]13:37
     */

#ifdef __func__
    const char *logstr=__func__;
#else
    const char *logstr="lcmaps_lifetime_ttl_char2time_t";
#endif

     /* 
      * I know it's totally not clean, but it will be in the near future 
      */


    size_t i = 0;
    int    do_days = 0;
    char * emitetad = NULL;
    int    minutes = 0;
    int    hours   = 0;
    int    days    = 0;
    /*int    months  = 0; */
    int    tot_sec = 0;
/*    int    years   = 0;  No you shouldn't want to wish to use this... */
    const size_t datetimelen=strlen(datetime);


    lcmaps_log_debug (2, "Proxy Time To Live parsing: %s\n", datetime);

    /*
     * extract time - which must be in a 24h notation!
     */
    if (datetimelen < 4)
    {
        lcmaps_log(LOG_ERR,
		"%s: parse error: implicit full 24h notation expected: range from 00:00 to 24:59, found: %s\n",
		logstr, datetime);
        return -1;
    }

    /*
     * Flip datetime
     */

    /* Create a buffer */
    emitetad = (char *) calloc (datetimelen + 1, sizeof (char));
    if (emitetad==NULL)	{
	lcmaps_log(LOG_ERR, "%s: Error: out of memory\n",logstr);
	return -1;
    }

    for (i = 0; i < datetimelen; i++)
        emitetad[i] = datetime[datetimelen - 1 - i];

    /*
     * Time extraction
     */

    for (i = 0; i < strlen(emitetad); i++)
    {
        /* Minutes */
        switch (i)
        {
            case 0:
            {
		if (!isdigit(emitetad[i]))
		    return -1;
                minutes = emitetad[i]-'0';
                break;
            }
            case 1:
            {
		if (!isdigit(emitetad[i]))
		    return -1;
                minutes+= 10 * (emitetad[i]-'0');
                break;
            }

            case 2: 
            {
                if (emitetad[i] == ':')
                    continue;
                else
                    return -1;
            }
            case 3:
            {
		if (!isdigit(emitetad[i]))
		    return -1;
                hours = emitetad[i]-'0';
                break;
            }
            case 4:
            {
		if (!isdigit(emitetad[i]))
		    return -1;
                hours+= 10 * (emitetad[i]-'0');
                break;
            }
 
            case 5:
            {
                if (emitetad[i] == '-')
                    continue;
                else
                    return -1;
            }

            case 6:
            {
                if ( (emitetad[i] == 'd') || (emitetad[i] == 'D') )
                {
                    do_days = 1;
                    continue;
                }
                else
                    return -1;  /* This is not always true, also a month should be selectable, without a day specification */
            }
           
            case 7:
            {
                /* I know it's totally not clean, but it will be in the near future */
                if (!do_days)
                    return -1;
                else
                {
		    if (!isdigit(emitetad[i]))
			return -1;
                    days = emitetad[i]-'0';
                }
                break;
            }
            case 8:
            {
                /* I know it's totally not clean, but it will be in the near future */
                if (!do_days)
                    return -1;
                else
                {
		    if (!isdigit(emitetad[i]))
			return -1;
                    days+= 10 * (emitetad[i]-'0');
                }
                break;
            }

            default:
                return -1;
        }
    }

    /* Free buffers */
    free(emitetad);

    /* Calc seconds: */
    tot_sec = minutes * 60 + hours * 3600 + days * 3600 * 24;

 
    lcmaps_log_debug (2, "Successfully finished Proxy Time To Live parsing: %d days, %d hours, %d minutes makes %d seconds.\n", days, hours, minutes, tot_sec);

    return tot_sec;
}


/******************************************************************************
 * Private functions
 ******************************************************************************/

/**
 * Returns the TTL for given level in the list p or 0 when no match is found.
 */
static time_t Search_TTL_By_Level (struct TProxyLevelTTL * p, int level)
{
    struct TProxyLevelTTL * curr = NULL;

    curr = p;

    while (curr != NULL)
    {
        if (curr->level == level)
        {
            return curr->ttl;   
        } 

        curr = curr->next;
    }

    return 0;
}


/**
 * Returns 1 when min(t1,t2) < time < max(t1,t2), otherwise returns 0.
 */
int timeIsInBetween (time_t this, time_t t1, time_t t2)
{
    time_t low, high;

    if (t1 > t2)
    {
        high = t1;
        low  = t2;
    }
    else
    {
        high = t2;
        low  = t1;
    }


    if (this < high && this > low)
        return 1;
    
    return 0;
}

