#include "srt_metrics.h"
#include "shared/metrics.h"

#include <srt/srt.h>

using namespace std;

void SRTMetrics::init(const vector<pair<string, string>> &labels)
{
	// Global measurements (counters).
	global_metrics.add("srt_uptime_seconds", labels, &metric_srt_uptime_seconds);
	global_metrics.add("srt_send_duration_seconds", labels, &metric_srt_send_duration_seconds);
	global_metrics.add("srt_sent_bytes", labels, &metric_srt_sent_bytes);
	global_metrics.add("srt_received_bytes", labels, &metric_srt_received_bytes);

	vector<pair<string, string>> packet_labels = labels;
	packet_labels.emplace_back("type", "normal");
	global_metrics.add("srt_sent_packets", packet_labels, &metric_srt_sent_packets_normal);
	global_metrics.add("srt_received_packets", packet_labels, &metric_srt_received_packets_normal);

	packet_labels.back().second = "lost";
	global_metrics.add("srt_sent_packets", packet_labels, &metric_srt_sent_packets_lost);
	global_metrics.add("srt_received_packets", packet_labels, &metric_srt_received_packets_lost);

	packet_labels.back().second = "retransmitted";
	global_metrics.add("srt_sent_packets", packet_labels, &metric_srt_sent_packets_retransmitted);
	global_metrics.add("srt_sent_bytes", packet_labels, &metric_srt_sent_bytes_retransmitted);

	packet_labels.back().second = "ack";
	global_metrics.add("srt_sent_packets", packet_labels, &metric_srt_sent_packets_ack);
	global_metrics.add("srt_received_packets", packet_labels, &metric_srt_received_packets_ack);

	packet_labels.back().second = "nak";
	global_metrics.add("srt_sent_packets", packet_labels, &metric_srt_sent_packets_nak);
	global_metrics.add("srt_received_packets", packet_labels, &metric_srt_received_packets_nak);

	packet_labels.back().second = "dropped";
	global_metrics.add("srt_sent_packets", packet_labels, &metric_srt_sent_packets_dropped);
	global_metrics.add("srt_received_packets", packet_labels, &metric_srt_received_packets_dropped);
	global_metrics.add("srt_sent_bytes", packet_labels, &metric_srt_sent_bytes_dropped);
	global_metrics.add("srt_received_bytes", packet_labels, &metric_srt_received_bytes_dropped);

	packet_labels.back().second = "undecryptable";
	global_metrics.add("srt_received_packets", packet_labels, &metric_srt_received_packets_undecryptable);
	global_metrics.add("srt_received_bytes", packet_labels, &metric_srt_received_bytes_undecryptable);

	global_metrics.add("srt_filter_sent_extra_packets", labels, &metric_srt_filter_sent_packets);
	global_metrics.add("srt_filter_received_extra_packets", labels, &metric_srt_filter_received_extra_packets);
	global_metrics.add("srt_filter_received_rebuilt_packets", labels, &metric_srt_filter_received_rebuilt_packets);
	global_metrics.add("srt_filter_received_lost_packets", labels, &metric_srt_filter_received_lost_packets);

	// Instant measurements (gauges).
	global_metrics.add("srt_packet_sending_period_seconds", labels, &metric_srt_packet_sending_period_seconds, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_flow_window_packets", labels, &metric_srt_flow_window_packets, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_congestion_window_packets", labels, &metric_srt_congestion_window_packets, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_flight_size_packets", labels, &metric_srt_flight_size_packets, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_rtt_seconds", labels, &metric_srt_rtt_seconds, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_estimated_bandwidth_bits_per_second", labels, &metric_srt_estimated_bandwidth_bits_per_second, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_bandwidth_ceiling_bits_per_second", labels, &metric_srt_bandwidth_ceiling_bits_per_second, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_send_buffer_available_bytes", labels, &metric_srt_send_buffer_available_bytes, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_receive_buffer_available_bytes", labels, &metric_srt_receive_buffer_available_bytes, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_mss_bytes", labels, &metric_srt_mss_bytes, Metrics::TYPE_GAUGE);

	global_metrics.add("srt_sender_unacked_packets", labels, &metric_srt_sender_unacked_packets, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_sender_unacked_bytes", labels, &metric_srt_sender_unacked_bytes, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_sender_unacked_timespan_seconds", labels, &metric_srt_sender_unacked_timespan_seconds, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_sender_delivery_delay_seconds", labels, &metric_srt_sender_delivery_delay_seconds, Metrics::TYPE_GAUGE);

	global_metrics.add("srt_receiver_unacked_packets", labels, &metric_srt_receiver_unacked_packets, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_receiver_unacked_bytes", labels, &metric_srt_receiver_unacked_bytes, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_receiver_unacked_timespan_seconds", labels, &metric_srt_receiver_unacked_timespan_seconds, Metrics::TYPE_GAUGE);
	global_metrics.add("srt_receiver_delivery_delay_seconds", labels, &metric_srt_receiver_delivery_delay_seconds, Metrics::TYPE_GAUGE);
}

void SRTMetrics::deinit(const vector<pair<string, string>> &labels)
{
	// Global measurements (counters).
	global_metrics.remove_if_exists("srt_uptime_seconds", labels);
	global_metrics.remove_if_exists("srt_send_duration_seconds", labels);
	global_metrics.remove_if_exists("srt_sent_bytes", labels);
	global_metrics.remove_if_exists("srt_received_bytes", labels);

	vector<pair<string, string>> packet_labels = labels;
	packet_labels.emplace_back("type", "normal");
	global_metrics.remove_if_exists("srt_sent_packets", packet_labels);
	global_metrics.remove_if_exists("srt_received_packets", packet_labels);

	packet_labels.back().second = "lost";
	global_metrics.remove_if_exists("srt_sent_packets", packet_labels);
	global_metrics.remove_if_exists("srt_received_packets", packet_labels);

	packet_labels.back().second = "retransmitted";
	global_metrics.remove_if_exists("srt_sent_packets", packet_labels);
	global_metrics.remove_if_exists("srt_sent_bytes", packet_labels);

	packet_labels.back().second = "ack";
	global_metrics.remove_if_exists("srt_sent_packets", packet_labels);
	global_metrics.remove_if_exists("srt_received_packets", packet_labels);

	packet_labels.back().second = "nak";
	global_metrics.remove_if_exists("srt_sent_packets", packet_labels);
	global_metrics.remove_if_exists("srt_received_packets", packet_labels);

	packet_labels.back().second = "dropped";
	global_metrics.remove_if_exists("srt_sent_packets", packet_labels);
	global_metrics.remove_if_exists("srt_received_packets", packet_labels);
	global_metrics.remove_if_exists("srt_sent_bytes", packet_labels);
	global_metrics.remove_if_exists("srt_received_bytes", packet_labels);

	packet_labels.back().second = "undecryptable";
	global_metrics.remove_if_exists("srt_received_packets", packet_labels);
	global_metrics.remove_if_exists("srt_received_bytes", packet_labels);

	global_metrics.remove_if_exists("srt_filter_sent_extra_packets", labels);
	global_metrics.remove_if_exists("srt_filter_received_extra_packets", labels);
	global_metrics.remove_if_exists("srt_filter_received_rebuilt_packets", labels);
	global_metrics.remove_if_exists("srt_filter_received_lost_packets", labels);

	// Instant measurements (gauges).
	global_metrics.remove_if_exists("srt_packet_sending_period_seconds", labels);
	global_metrics.remove_if_exists("srt_flow_window_packets", labels);
	global_metrics.remove_if_exists("srt_congestion_window_packets", labels);
	global_metrics.remove_if_exists("srt_flight_size_packets", labels);
	global_metrics.remove_if_exists("srt_rtt_seconds", labels);
	global_metrics.remove_if_exists("srt_estimated_bandwidth_bits_per_second", labels);
	global_metrics.remove_if_exists("srt_bandwidth_ceiling_bits_per_second", labels);
	global_metrics.remove_if_exists("srt_send_buffer_available_bytes", labels);
	global_metrics.remove_if_exists("srt_receive_buffer_available_bytes", labels);
	global_metrics.remove_if_exists("srt_mss_bytes", labels);

	global_metrics.remove_if_exists("srt_sender_unacked_packets", labels);
	global_metrics.remove_if_exists("srt_sender_unacked_bytes", labels);
	global_metrics.remove_if_exists("srt_sender_unacked_timespan_seconds", labels);
	global_metrics.remove_if_exists("srt_sender_delivery_delay_seconds", labels);

	global_metrics.remove_if_exists("srt_receiver_unacked_packets", labels);
	global_metrics.remove_if_exists("srt_receiver_unacked_bytes", labels);
	global_metrics.remove_if_exists("srt_receiver_unacked_timespan_seconds", labels);
	global_metrics.remove_if_exists("srt_receiver_delivery_delay_seconds", labels);
}

#ifdef HAVE_SRT
void SRTMetrics::update_srt_stats(int srt_sock)
{
	SRT_TRACEBSTATS stats;
	srt_bistats(srt_sock, &stats, /*clear=*/0, /*instantaneous=*/1);

	metric_srt_uptime_seconds = stats.msTimeStamp * 1e-3;
	metric_srt_send_duration_seconds = stats.usSndDurationTotal * 1e-6;
	metric_srt_sent_bytes = stats.byteSentTotal;
	metric_srt_received_bytes = stats.byteRecvTotal;
	metric_srt_sent_packets_normal = stats.pktSentTotal;
	metric_srt_received_packets_normal = stats.pktRecvTotal;
	metric_srt_sent_packets_lost = stats.pktSndLossTotal;
	metric_srt_received_packets_lost = stats.pktRcvLossTotal;
	metric_srt_sent_packets_retransmitted = stats.pktRetransTotal;
	metric_srt_sent_bytes_retransmitted = stats.byteRetransTotal;
	metric_srt_sent_packets_ack = stats.pktSentACKTotal;
	metric_srt_received_packets_ack = stats.pktRecvACKTotal;
	metric_srt_sent_packets_nak = stats.pktSentNAKTotal;
	metric_srt_received_packets_nak = stats.pktRecvNAKTotal;
	metric_srt_sent_packets_dropped = stats.pktSndDropTotal;
	metric_srt_received_packets_dropped = stats.pktRcvDropTotal;
	metric_srt_sent_bytes_dropped = stats.byteSndDropTotal;
	metric_srt_received_bytes_dropped = stats.byteRcvDropTotal;
	metric_srt_received_packets_undecryptable = stats.pktRcvUndecryptTotal;
	metric_srt_received_bytes_undecryptable = stats.byteRcvUndecryptTotal;
	metric_srt_filter_sent_packets = stats.pktSndFilterExtraTotal;
	metric_srt_filter_received_extra_packets = stats.pktRcvFilterExtraTotal;
	metric_srt_filter_received_rebuilt_packets = stats.pktRcvFilterSupplyTotal;
	metric_srt_filter_received_lost_packets = stats.pktRcvFilterLossTotal;

	// Gauges.
	metric_srt_packet_sending_period_seconds = stats.usPktSndPeriod * 1e-6;
	metric_srt_flow_window_packets = stats.pktFlowWindow;
	metric_srt_congestion_window_packets = stats.pktCongestionWindow;
	metric_srt_flight_size_packets = stats.pktFlightSize;
	metric_srt_rtt_seconds = stats.msRTT * 1e-3;
	metric_srt_estimated_bandwidth_bits_per_second = stats.mbpsBandwidth * 1e6;
	metric_srt_bandwidth_ceiling_bits_per_second = stats.mbpsMaxBW * 1e6;
	metric_srt_send_buffer_available_bytes = stats.byteAvailSndBuf;
	metric_srt_receive_buffer_available_bytes = stats.byteAvailRcvBuf;
	metric_srt_mss_bytes = stats.byteMSS;
	metric_srt_sender_unacked_packets = stats.pktSndBuf;
	metric_srt_sender_unacked_bytes = stats.byteSndBuf;
	metric_srt_sender_unacked_timespan_seconds = stats.msSndBuf * 1e-3;
	metric_srt_sender_delivery_delay_seconds = stats.msSndTsbPdDelay * 1e-3;
	metric_srt_receiver_unacked_packets = stats.pktRcvBuf;
	metric_srt_receiver_unacked_bytes = stats.byteRcvBuf;
	metric_srt_receiver_unacked_timespan_seconds = stats.msRcvBuf * 1e-3;
	metric_srt_receiver_delivery_delay_seconds = stats.msRcvTsbPdDelay * 1e-3;
}
#endif
