/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var validate = require( './../lib/validate.js' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof validate, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns an error if provided an options argument which is not an object', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		'5',
		5,
		null,
		NaN,
		true,
		void 0,
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( {}, values[ i ] );
		t.equal( err instanceof TypeError, true, 'returns a TypeError when provided '+values[ i ] );
	}
	t.end();
});

tape( 'the function returns an error if provided a `depth` option which is not a nonnegative integer', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		'5',
		-1,
		3.14,
		null,
		NaN,
		true,
		void 0,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( {}, {
			'depth': values[ i ]
		});
		t.equal( err instanceof TypeError, true, 'returns a TypeError when provided '+values[ i ] );
	}
	t.end();
});

tape( 'the function returns an error if provided a `copy` option which is not a boolean', function test( t ) {
	var values;
	var err;
	var i;

	values = [
		'5',
		5,
		null,
		NaN,
		void 0,
		[],
		{},
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		err = validate( {}, {
			'copy': values[ i ]
		});
		t.equal( err instanceof TypeError, true, 'returns a TypeError when provided '+values[ i ] );
	}
	t.end();
});

tape( 'the function returns `null` if all options are valid', function test( t ) {
	var opts;
	var obj;

	opts = {
		'depth': 10,
		'copy': false
	};
	obj = {};

	t.equal( validate( obj, opts ), null, 'returns null' );
	t.notEqual( obj, opts, 'different references' );
	t.deepEqual( obj, opts, 'sets options' );

	t.end();
});

tape( 'the function ignores unsupported/unrecognized options', function test( t ) {
	var opts;
	var obj;

	opts = {
		'beep': true, // unrelated options
		'boop': false
	};
	obj = {};

	t.equal( validate( obj, opts ), null, 'returns null' );
	t.deepEqual( obj, {}, 'does not set any properties' );

	t.end();
});
