% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clintables.R
\name{shift_table}
\alias{shift_table}
\title{Create a shift table}
\usage{
shift_table(
  x,
  cn_visit = "VISIT",
  cn_visit_num = "VISITNUM",
  cn_grade = "LBNRIND",
  cn_usubjid = "USUBJID",
  cn_lab_cat = NA_character_,
  cn_is_baseline = "LBBLFL",
  baseline_identifier = "Y",
  cn_treatment = NA_character_,
  grade_levels = c("LOW", "NORMAL", "HIGH"),
  grade_labels = c("Low", "Normal", "High")
)
}
\arguments{
\item{x}{Laboratory Tests Results data frame.}

\item{cn_visit}{column name containing visit names, default to "VISIT".}

\item{cn_visit_num}{column name containing visit numbers, default to "VISITNUM".}

\item{cn_grade}{column name containing reference range indicators, default to "LBNRIND".}

\item{cn_usubjid}{column name containing unique subject inditifiers, default to "USUBJID".}

\item{cn_lab_cat}{column name containing lab tests or examination names, default to "LBTEST".}

\item{cn_is_baseline}{column name containing baseline flags, default to "LBBLFL".}

\item{baseline_identifier}{baseline flag value to use for baseline
identification. Its default is "Y".}

\item{cn_treatment}{column name containing treatment names, default to \code{NA}.}

\item{grade_levels}{levels to use for reference range indicators}

\item{grade_labels}{labels to use for reference range indicators}
}
\value{
the shift table as a data.frame. Additionnal elements are provided
in attributes:
\itemize{
\item "VISIT_N": count of unique subject id per visits, labs and eventually
treatments. This element is supposed to be used as value for argument
\code{hidden_data} of function \code{tabulator()}.
\item "FUN_VISIT": a utility function to easily turn \emph{visit} column as a factor
column. It should be applied after the shift table creation.
\item "FUN_GRADE": a utility function to easily turn \emph{grade} column as a factor
column. It adds "MISSING/Missing" and "SUM/Sum" at the end of the set of values specified
in arguments \code{grade_levels} and \code{grade_labels}. It should be applied after the shift
table creation.
}
}
\description{
Create a shift table ready to be used with \code{tabulator()}.

The function is transforming a dataset representing some
'Laboratory Tests Results' structured as \emph{CDISC clinical trial
data sets} format to a dataset representing the shift table.

Shift tables are tables used in clinical trial analysis.
They show the progression of change from the baseline, with the progression
often being along time; the number of subjects is displayed in different
range (e.g. low, normal, or high) at baseline and at selected time points
or intervals.
}
\examples{
library(data.table)
library(flextable)

# data simulation ----
USUBJID <- sprintf("01-ABC-\%04.0f", 1:200)
VISITS <- c("SCREENING 1", "WEEK 2", "MONTH 3")
LBTEST <- c("Albumin", "Sodium")

VISITNUM <- seq_along(VISITS)
LBBLFL <- rep(NA_character_, length(VISITNUM))
LBBLFL[1] <- "Y"

VISIT <- data.frame(VISIT = VISITS, VISITNUM = VISITNUM,
  LBBLFL = LBBLFL, stringsAsFactors = FALSE)
labdata <- expand.grid(USUBJID = USUBJID,  LBTEST = LBTEST,
                    VISITNUM = VISITNUM,
                    stringsAsFactors = FALSE)
setDT(labdata)

labdata <- merge(labdata, VISIT, by = "VISITNUM")

subject_elts <- unique(labdata[, .SD, .SDcols = "USUBJID"])
subject_elts <- unique(subject_elts)
subject_elts[, c("TREAT") := list(
  sample(x = c("Treatment", "Placebo"), size = .N, replace = TRUE))]
subject_elts[, c("TREAT"):= list(
  factor(.SD$TREAT, levels = c("Treatment", "Placebo")))]
setDF(subject_elts)
labdata <- merge(labdata, subject_elts,
  by = "USUBJID", all.x = TRUE, all.y = FALSE)
labdata[, c("LBNRIND"):= list(
  sample(x = c("LOW", "NORMAL", "HIGH"), size = .N,
         replace = TRUE, prob = c(.03, .9, .07)))]

setDF(labdata)




# shift table calculation ----

SHIFT_TABLE <- shift_table(
  x = labdata, cn_visit = "VISIT",
  cn_grade = "LBNRIND",
  cn_usubjid = "USUBJID",
  cn_lab_cat = "LBTEST",
  cn_treatment = "TREAT",
  cn_is_baseline = "LBBLFL",
  baseline_identifier = "Y",
  grade_levels = c("LOW", "NORMAL", "HIGH"))

# get attrs for post treatment ----
SHIFT_TABLE_VISIT <- attr(SHIFT_TABLE, "VISIT_N")
visit_as_factor <- attr(SHIFT_TABLE, "FUN_VISIT")
range_as_factor <- attr(SHIFT_TABLE, "FUN_GRADE")

# post treatments ----
SHIFT_TABLE$VISIT = visit_as_factor(SHIFT_TABLE$VISIT)
SHIFT_TABLE$BASELINE = range_as_factor(SHIFT_TABLE$BASELINE)
SHIFT_TABLE$LBNRIND = range_as_factor(SHIFT_TABLE$LBNRIND)

SHIFT_TABLE_VISIT$VISIT = visit_as_factor(SHIFT_TABLE_VISIT$VISIT)

# tabulator ----

my_format <- function(z) {
  formatC(z * 100, digits = 1, format = "f",
          flag = "0", width = 4)
}

tab <- tabulator(
  x = SHIFT_TABLE,
  hidden_data = SHIFT_TABLE_VISIT,
  row_compose = list(
    VISIT = as_paragraph(VISIT, "\n(N=", N_VISIT, ")")
  ),
  rows = c("LBTEST", "VISIT", "BASELINE"),
  columns = c("TREAT", "LBNRIND"),
  `n` = as_paragraph(N),
  `\%` = as_paragraph(as_chunk(PCT, formatter = my_format))
)

# as_flextable ----

ft_1 <- as_flextable(
  x = tab, separate_with = "VISIT",
  label_rows = c(LBTEST = "Lab Test", VISIT = "Visit",
                 BASELINE = "Reference Range Indicator"))

ft_1
}
\concept{tools for clinical reporting}
