#!/usr/bin/env python
# -*- coding: UTF-8 -*-

# Copyright (c) 2020, Sandflow Consulting LLC
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice, this
#    list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""Unit tests for the SMPTE Time Codes"""

# pylint: disable=R0201,C0115,C0116,W0212
from fractions import Fraction
import unittest

from ttconv.time_code import SmpteTimeCode, FPS_29_97, FPS_30


class SmpteTimeCodesTest(unittest.TestCase):

  def test_parse_non_drop_frame_time_code(self):
    time_code = SmpteTimeCode.parse("01:02:03:04", FPS_30)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(3, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_30, time_code.get_frame_rate())
    self.assertFalse(time_code.is_drop_frame())
    self.assertEqual(111694, time_code.to_frames())
    self.assertAlmostEqual(3723.133, time_code.to_seconds(), delta=0.001)
    self.assertEqual("01:02:03:04", str(time_code))

  def test_parse_drop_frame_time_code(self):
    time_code = SmpteTimeCode.parse("01:02:03;04", FPS_29_97)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(3, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_29_97, time_code.get_frame_rate())
    self.assertTrue(time_code.is_drop_frame())
    self.assertEqual(111582, time_code.to_frames())
    self.assertEqual(Fraction(111582, Fraction(30000, 1001)), time_code.to_temporal_offset())
    self.assertEqual("01:02:03;04", str(time_code))

    time_code = SmpteTimeCode.parse("01;02;03;04", FPS_29_97)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(3, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_29_97, time_code.get_frame_rate())
    self.assertTrue(time_code.is_drop_frame())
    self.assertEqual(111582, time_code.to_frames())
    self.assertEqual(Fraction(111582, Fraction(30000, 1001)), time_code.to_temporal_offset())
    self.assertEqual("01:02:03;04", str(time_code))

    time_code = SmpteTimeCode.parse("01:02:03;04", FPS_30)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(3, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_29_97, time_code.get_frame_rate())
    self.assertTrue(time_code.is_drop_frame())
    self.assertEqual(111582, time_code.to_frames())
    self.assertEqual(Fraction(111582, Fraction(30000, 1001)), time_code.to_temporal_offset())
    self.assertEqual("01:02:03;04", str(time_code))

    time_code = SmpteTimeCode.parse("01:02:03.04", FPS_29_97)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(3, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_29_97, time_code.get_frame_rate())
    self.assertTrue(time_code.is_drop_frame())
    self.assertEqual(111582, time_code.to_frames())
    self.assertEqual(Fraction(111582, Fraction(30000, 1001)), time_code.to_temporal_offset())
    self.assertEqual("01:02:03;04", str(time_code))

    time_code = SmpteTimeCode.parse("01.02.03.04", FPS_29_97)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(3, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_29_97, time_code.get_frame_rate())
    self.assertTrue(time_code.is_drop_frame())
    self.assertEqual(111582, time_code.to_frames())
    self.assertEqual(Fraction(111582, Fraction(30000, 1001)), time_code.to_temporal_offset())
    self.assertEqual("01:02:03;04", str(time_code))

  def test_time_code_frames_conversion(self):
    time_code = SmpteTimeCode.from_frames(1795, FPS_30)
    self.assertEqual("00:00:59:25", str(time_code))
    self.assertEqual(1795, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1796, FPS_30)
    self.assertEqual("00:00:59:26", str(time_code))
    self.assertEqual(1796, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1797, FPS_30)
    self.assertEqual("00:00:59:27", str(time_code))
    self.assertEqual(1797, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1798, FPS_30)
    self.assertEqual("00:00:59:28", str(time_code))
    self.assertEqual(1798, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1799, FPS_30)
    self.assertEqual("00:00:59:29", str(time_code))
    self.assertEqual(1799, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1800, FPS_30)
    self.assertEqual("00:01:00:00", str(time_code))
    self.assertEqual(1800, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1801, FPS_30)
    self.assertEqual("00:01:00:01", str(time_code))
    self.assertEqual(1801, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1802, FPS_30)
    self.assertEqual("00:01:00:02", str(time_code))
    self.assertEqual(1802, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1803, FPS_30)
    self.assertEqual("00:01:00:03", str(time_code))
    self.assertEqual(1803, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1804, FPS_30)
    self.assertEqual("00:01:00:04", str(time_code))
    self.assertEqual(1804, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1805, FPS_30)
    self.assertEqual("00:01:00:05", str(time_code))
    self.assertEqual(1805, time_code.to_frames())

    time_code = SmpteTimeCode.from_frames(17977, FPS_30)
    self.assertEqual("00:09:59:07", str(time_code))
    self.assertEqual(17977, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17978, FPS_30)
    self.assertEqual("00:09:59:08", str(time_code))
    self.assertEqual(17978, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17979, FPS_30)
    self.assertEqual("00:09:59:09", str(time_code))
    self.assertEqual(17979, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17980, FPS_30)
    self.assertEqual("00:09:59:10", str(time_code))
    self.assertEqual(17980, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17981, FPS_30)
    self.assertEqual("00:09:59:11", str(time_code))
    self.assertEqual(17981, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17982, FPS_30)
    self.assertEqual("00:09:59:12", str(time_code))
    self.assertEqual(17982, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17983, FPS_30)
    self.assertEqual("00:09:59:13", str(time_code))
    self.assertEqual(17983, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17984, FPS_30)
    self.assertEqual("00:09:59:14", str(time_code))
    self.assertEqual(17984, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17985, FPS_30)
    self.assertEqual("00:09:59:15", str(time_code))
    self.assertEqual(17985, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17986, FPS_30)
    self.assertEqual("00:09:59:16", str(time_code))
    self.assertEqual(17986, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17987, FPS_30)
    self.assertEqual("00:09:59:17", str(time_code))
    self.assertEqual(17987, time_code.to_frames())

  def test_drop_frame_time_code_frames_conversion(self):
    time_code = SmpteTimeCode.from_frames(1795, FPS_29_97)
    self.assertEqual("00:00:59;25", str(time_code))
    self.assertEqual(1795, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1796, FPS_29_97)
    self.assertEqual("00:00:59;26", str(time_code))
    self.assertEqual(1796, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1797, FPS_29_97)
    self.assertEqual("00:00:59;27", str(time_code))
    self.assertEqual(1797, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1798, FPS_29_97)
    self.assertEqual("00:00:59;28", str(time_code))
    self.assertEqual(1798, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1799, FPS_29_97)
    self.assertEqual("00:00:59;29", str(time_code))
    self.assertEqual(1799, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1800, FPS_29_97)
    self.assertEqual("00:01:00;02", str(time_code))
    self.assertEqual(1800, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1801, FPS_29_97)
    self.assertEqual("00:01:00;03", str(time_code))
    self.assertEqual(1801, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1802, FPS_29_97)
    self.assertEqual("00:01:00;04", str(time_code))
    self.assertEqual(1802, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1803, FPS_29_97)
    self.assertEqual("00:01:00;05", str(time_code))
    self.assertEqual(1803, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1804, FPS_29_97)
    self.assertEqual("00:01:00;06", str(time_code))
    self.assertEqual(1804, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(1805, FPS_29_97)
    self.assertEqual("00:01:00;07", str(time_code))
    self.assertEqual(1805, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17977, FPS_29_97)

    self.assertEqual("00:09:59;25", str(time_code))
    self.assertEqual(17977, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17978, FPS_29_97)
    self.assertEqual("00:09:59;26", str(time_code))
    self.assertEqual(17978, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17979, FPS_29_97)
    self.assertEqual("00:09:59;27", str(time_code))
    self.assertEqual(17979, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17980, FPS_29_97)
    self.assertEqual("00:09:59;28", str(time_code))
    self.assertEqual(17980, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17981, FPS_29_97)
    self.assertEqual("00:09:59;29", str(time_code))
    self.assertEqual(17981, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17982, FPS_29_97)
    self.assertEqual("00:10:00;00", str(time_code))
    self.assertEqual(17982, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17983, FPS_29_97)
    self.assertEqual("00:10:00;01", str(time_code))
    self.assertEqual(17983, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17984, FPS_29_97)
    self.assertEqual("00:10:00;02", str(time_code))
    self.assertEqual(17984, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17985, FPS_29_97)
    self.assertEqual("00:10:00;03", str(time_code))
    self.assertEqual(17985, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17986, FPS_29_97)
    self.assertEqual("00:10:00;04", str(time_code))
    self.assertEqual(17986, time_code.to_frames())
    time_code = SmpteTimeCode.from_frames(17987, FPS_29_97)
    self.assertEqual("00:10:00;05", str(time_code))
    self.assertEqual(17987, time_code.to_frames())

  def test_time_code_seconds_conversion(self):
    time_code = SmpteTimeCode.from_frames(1795, FPS_30)
    self.assertEqual("00:00:59:25", str(time_code))
    self.assertAlmostEqual(59.833, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1796, FPS_30)
    self.assertEqual("00:00:59:26", str(time_code))
    self.assertAlmostEqual(59.866, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1797, FPS_30)
    self.assertEqual("00:00:59:27", str(time_code))
    self.assertAlmostEqual(59.9, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1798, FPS_30)
    self.assertEqual("00:00:59:28", str(time_code))
    self.assertAlmostEqual(59.933, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1799, FPS_30)
    self.assertEqual("00:00:59:29", str(time_code))
    self.assertAlmostEqual(59.966, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1800, FPS_30)
    self.assertEqual("00:01:00:00", str(time_code))
    self.assertAlmostEqual(60.0, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1801, FPS_30)
    self.assertEqual("00:01:00:01", str(time_code))
    self.assertAlmostEqual(60.033, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1802, FPS_30)
    self.assertEqual("00:01:00:02", str(time_code))
    self.assertAlmostEqual(60.066, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1803, FPS_30)
    self.assertEqual("00:01:00:03", str(time_code))
    self.assertAlmostEqual(60.1, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1804, FPS_30)
    self.assertEqual("00:01:00:04", str(time_code))
    self.assertAlmostEqual(60.133, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1805, FPS_30)
    self.assertEqual("00:01:00:05", str(time_code))
    self.assertAlmostEqual(60.166, time_code.to_seconds(), delta=0.001)

    time_code = SmpteTimeCode.from_frames(17977, FPS_30)
    self.assertEqual("00:09:59:07", str(time_code))
    self.assertAlmostEqual(599.233, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17978, FPS_30)
    self.assertEqual("00:09:59:08", str(time_code))
    self.assertAlmostEqual(599.266, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17979, FPS_30)
    self.assertEqual("00:09:59:09", str(time_code))
    self.assertAlmostEqual(599.3, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17980, FPS_30)
    self.assertEqual("00:09:59:10", str(time_code))
    self.assertAlmostEqual(599.333, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17981, FPS_30)
    self.assertEqual("00:09:59:11", str(time_code))
    self.assertAlmostEqual(599.366, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17982, FPS_30)
    self.assertEqual("00:09:59:12", str(time_code))
    self.assertAlmostEqual(599.4, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17983, FPS_30)
    self.assertEqual("00:09:59:13", str(time_code))
    self.assertAlmostEqual(599.433, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17984, FPS_30)
    self.assertEqual("00:09:59:14", str(time_code))
    self.assertAlmostEqual(599.466, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17985, FPS_30)
    self.assertEqual("00:09:59:15", str(time_code))
    self.assertAlmostEqual(599.5, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17986, FPS_30)
    self.assertEqual("00:09:59:16", str(time_code))
    self.assertAlmostEqual(599.533, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17987, FPS_30)
    self.assertEqual("00:09:59:17", str(time_code))
    self.assertAlmostEqual(599.566, time_code.to_seconds(), delta=0.001)

  def test_drop_frame_time_code_seconds_conversion(self):
    time_code = SmpteTimeCode.from_frames(1795, FPS_29_97)
    self.assertEqual("00:00:59;25", str(time_code))
    self.assertAlmostEqual(59.893, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1796, FPS_29_97)
    self.assertEqual("00:00:59;26", str(time_code))
    self.assertAlmostEqual(59.926, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1797, FPS_29_97)
    self.assertEqual("00:00:59;27", str(time_code))
    self.assertAlmostEqual(59.959, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1798, FPS_29_97)
    self.assertEqual("00:00:59;28", str(time_code))
    self.assertAlmostEqual(59.993, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1799, FPS_29_97)
    self.assertEqual("00:00:59;29", str(time_code))
    self.assertAlmostEqual(60.026, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1800, FPS_29_97)
    self.assertEqual("00:01:00;02", str(time_code))
    self.assertAlmostEqual(60.06, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1801, FPS_29_97)
    self.assertEqual("00:01:00;03", str(time_code))
    self.assertAlmostEqual(60.093, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1802, FPS_29_97)
    self.assertEqual("00:01:00;04", str(time_code))
    self.assertAlmostEqual(60.126, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1803, FPS_29_97)
    self.assertEqual("00:01:00;05", str(time_code))
    self.assertAlmostEqual(60.160, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1804, FPS_29_97)
    self.assertEqual("00:01:00;06", str(time_code))
    self.assertAlmostEqual(60.193, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(1805, FPS_29_97)
    self.assertEqual("00:01:00;07", str(time_code))
    self.assertAlmostEqual(60.226, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17977, FPS_29_97)

    self.assertEqual("00:09:59;25", str(time_code))
    self.assertAlmostEqual(599.832, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17978, FPS_29_97)
    self.assertEqual("00:09:59;26", str(time_code))
    self.assertAlmostEqual(599.865, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17979, FPS_29_97)
    self.assertEqual("00:09:59;27", str(time_code))
    self.assertAlmostEqual(599.899, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17980, FPS_29_97)
    self.assertEqual("00:09:59;28", str(time_code))
    self.assertAlmostEqual(599.932, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17981, FPS_29_97)
    self.assertEqual("00:09:59;29", str(time_code))
    self.assertAlmostEqual(599.966, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17982, FPS_29_97)
    self.assertEqual("00:10:00;00", str(time_code))
    self.assertAlmostEqual(599.999, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17983, FPS_29_97)
    self.assertEqual("00:10:00;01", str(time_code))
    self.assertAlmostEqual(600.032, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17984, FPS_29_97)
    self.assertEqual("00:10:00;02", str(time_code))
    self.assertAlmostEqual(600.066, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17985, FPS_29_97)
    self.assertEqual("00:10:00;03", str(time_code))
    self.assertAlmostEqual(600.099, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17986, FPS_29_97)
    self.assertEqual("00:10:00;04", str(time_code))
    self.assertAlmostEqual(600.132, time_code.to_seconds(), delta=0.001)
    time_code = SmpteTimeCode.from_frames(17987, FPS_29_97)
    self.assertEqual("00:10:00;05", str(time_code))
    self.assertAlmostEqual(600.166, time_code.to_seconds(), delta=0.001)

  def test_from_frames(self):
    time_code = SmpteTimeCode.from_frames(111694, FPS_30)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(3, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_30, time_code.get_frame_rate())
    self.assertFalse(time_code.is_drop_frame())
    self.assertEqual(111694, time_code.to_frames())
    self.assertEqual("01:02:03:04", str(time_code))

    time_code = SmpteTimeCode.from_frames(111582, FPS_29_97)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(3, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_29_97, time_code.get_frame_rate())
    self.assertTrue(time_code.is_drop_frame())
    self.assertEqual(111582, time_code.to_frames())
    self.assertEqual("01:02:03;04", str(time_code))

  def test_add_frames(self):
    time_code = SmpteTimeCode.parse("01:02:03:04", FPS_30)
    time_code.add_frames(30)
    self.assertEqual(1, time_code.get_hours())
    self.assertEqual(2, time_code.get_minutes())
    self.assertEqual(4, time_code.get_seconds())
    self.assertEqual(4, time_code.get_frames())
    self.assertEqual(FPS_30, time_code.get_frame_rate())
    self.assertFalse(time_code.is_drop_frame())
    self.assertEqual(111724, time_code.to_frames())
    self.assertEqual(Fraction(111724, 30), time_code.to_temporal_offset())
    self.assertEqual("01:02:04:04", str(time_code))

    time_code = SmpteTimeCode.parse("00:00:59;29", FPS_29_97)
    time_code.add_frames()
    self.assertEqual(0, time_code.get_hours())
    self.assertEqual(1, time_code.get_minutes())
    self.assertEqual(0, time_code.get_seconds())
    self.assertEqual(2, time_code.get_frames())
    self.assertEqual(FPS_29_97, time_code.get_frame_rate())
    self.assertTrue(time_code.is_drop_frame())
    self.assertEqual(1_800, time_code.to_frames())
    self.assertEqual(Fraction(1800, Fraction(30000, 1001)), time_code.to_temporal_offset())
    self.assertEqual("00:01:00;02", str(time_code))

    time_code = SmpteTimeCode.parse("00:19:59;29", FPS_29_97)
    time_code.add_frames()
    self.assertEqual(0, time_code.get_hours())
    self.assertEqual(20, time_code.get_minutes())
    self.assertEqual(0, time_code.get_seconds())
    self.assertEqual(0, time_code.get_frames())
    self.assertEqual(FPS_29_97, time_code.get_frame_rate())
    self.assertTrue(time_code.is_drop_frame())
    self.assertEqual(35_964, time_code.to_frames())
    self.assertEqual(Fraction(35964, Fraction(30000, 1001)), time_code.to_temporal_offset())
    self.assertEqual("00:20:00;00", str(time_code))

if __name__ == '__main__':
  unittest.main()
