/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the XMPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * XMPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: xmpi.cc,v 1.7 2003/09/03 04:28:53 jsquyres Exp $
 *
 *	Function:	- MPI GUI
 *			- Motif
 */

#include <X11/StringDefs.h>
#include <Xm/MessageB.h>
#include <Xm/Form.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "xmpi.h"
#include "xmpi_run.h"
#include "xmpi_ctl.h"
#include "xmpi_trace.h"
#include "xmpi_dtype.h"
#include "xmpi_misc.h"
#include "xmpi_view.h"
#include "xmpi_tr_dump.h"
#include "xmpi_aschema.h"
#include "xmpi_focus.h"
#include "xmpi_kiviat.h"
#include "xmpi_vendor.h"

/*
 * external functions
 */

extern void _XEditResCheckMessages();
extern char *_path_find();

/*
 * local functions
 */
static void help();

/*
 * global variables
 */
XtAppContext app;		       /* application context */

AppData app_res;		       /* application resources */

Widget xmpi_shell;		       /* toplevel shell */

char xmpi_info[XMPI_PATHMAX];	       /* information string */

/*
 * local variables
 */
static Widget xmpitop;		       /* toplevel manager */

static int hinitmin = -1;	       /* ini. drawable min height */

static int winitmin = -1;	       /* init. drawable min width */

static int hinitmax = -1;	       /* init. drawable max height */

static int winitmax = -1;	       /* init. drawable max width */

static int hsetup = -1;		       /* height setup overhead */

static int wsetup = -1;		       /* width setup overhead */

static XrmOptionDescRec cmdoptions[] = {
  {(char*) "-h", (char*) "*help", XrmoptionNoArg, (XPointer) "TRUE"}
};

static XtResource resources[] = {
  {
    (char*) XtNhelp,
    (char*) XtCHelp,
    XmRBoolean,
    sizeof(Boolean),
    XtOffset(AppDataPtr, ap_help),
    XmRImmediate,
    (caddr_t) False
  },
  {
    (char*) XtNrankFont,
    (char*) XtCRankFont,
    XmRFontStruct,
    sizeof(XFontStruct *),
    XtOffset(AppDataPtr, ap_rankfont),
    XmRString,
    (void*) XMPI_RANKFONT
  },
  {
    (char*) XtNrankFont,
    (char*) XtCRankFont,
    XmRFontStruct,
    sizeof(XFontStruct *),
    XtOffset(AppDataPtr, ap_rankfont),
    XmRString,
    (void*) XMPI_RANKFONT
  },
  {
    (char*) XtNmsgFont,
    (char*) XtCMsgFont,
    XmRFontStruct,
    sizeof(XFontStruct *),
    XtOffset(AppDataPtr, ap_msgfont),
    XmRString,
    (void*) XMPI_MSGFONT
  },
  {
    (char*) XtNlcomCol,
    (char*) XtCLcomCol,
    XtRPixel,
    sizeof(Pixel),
    XtOffset(AppDataPtr, ap_lcomcol),
    XmRString,
    (void*) XMPI_LCOMCOLOUR
  },
  {
    (char*) XtNrcomCol,
    (char*) XtCRcomCol,
    XtRPixel,
    sizeof(Pixel),
    XtOffset(AppDataPtr, ap_rcomcol),
    XmRString,
    (void*) XMPI_RCOMCOLOUR
  },
  {
    (char*) XtNbandCol,
    (char*) XtCBandCol,
    XtRPixel,
    sizeof(Pixel),
    XtOffset(AppDataPtr, ap_bandcol),
    XmRString,
    (void*) XMPI_BANDCOLOUR
  },
  {
    (char*) XtNbandDash,
    (char*) XtCBandDash,
    XtRBoolean,
    sizeof(Boolean),
    XtOffset(AppDataPtr, ap_banddash),
    XmRImmediate,
    (caddr_t) XMPI_BANDDASH
  },
  {
    (char*) XtNbandWidth,
    (char*) XtCBandWidth,
    XtRInt,
    sizeof(int),
    XtOffset(AppDataPtr, ap_bandwidth),
    XmRImmediate,
    (caddr_t) XMPI_BANDWIDTH
  },
  {
    (char*) XtNhelpCmd,
    (char*) XtCHelpCmd,
    XtRString,
    sizeof(String),
    XtOffset(AppDataPtr, ap_helpcmd),
    XmRString,
    (void*) XMPI_HELPCMD
  },
  {
    (char*) XtNinfoLabel,
    (char*) XtCInfoLabel,
    XtRString,
    sizeof(String),
    XtOffset(AppDataPtr, ap_infolbl),
    XmRString,
    (void*) XMPI_INFOLABEL
  },
};

static XtActionsRec actions[] = {
  {
    (char*) "xmpi_run_delete",
    (XtActionProc) xmpi_run_delete
  }
};

/*
 * fallback resources
 */
const _XtString fallbacks[] = {
  "*Title: XMPI",
  "*IconName: XMPI",
  "*multiClickTime: 500",
  "*background: gray",
  "*fontList: -*-helvetica-bold-r-normal--*-120-*-*-*-*-*-*",
  "*fo_func.fontList: -*-helvetica-bold-o-normal--*-120-*-*-*-*-*-*",
  "*dt_dtype.fontList: -*-helvetica-medium-r-normal--*-100-*-*-*-*-*-*",
  "*ctl_bar.topShadowColor: lightslateblue",
  "*ctl_bar.bottomShadowColor: darkslateblue",
  "*ctl_bar.background: slateblue",
  "*ctl_bar.foreground: white",
  "*banner.background: slateblue",
  "*banner.foreground: white",
  "*trace_banner.background: white",
  "*trace_banner.foreground: slateblue",
  "*app_list.visibleItemCount: 8",
  "*aschema_list.visibleItemCount: 20",
  "*aschema_text.columns: 24",
  "*prog_mgr*columns: 16",
  "*label_frame.XmLabel.background: #D3B5B5",
  "*XmToggleButtonGadget.selectColor: red",
  "*XmToggleButton.selectColor: red",
  "*trace_selector.dirMask: *.lamtr",
  "*popup_help.background: white",
  "*popup_help.foreground: black",
/*
 * Workaround for problem with HP-VUE.
 */

#ifdef HPUX
  "*view_draw.background: gray",
  "*view_draw.foreground: black",
  "*trace_draw.background: gray",
  "*trace_draw.foreground: black",
  "*kiviat_draw.background: gray",
  "*kiviat_draw.foreground: black",
  "*matrix_draw.background: gray",
  "*matrix_draw.foreground: black",
#endif

  NULL
};

/*
 *	main
 */
int
main(int argc, char *argv[])
{
  Widget w1, w2;
  char *info = 0;
  int i;

  xmpi_shell = XtVaAppInitialize(&app, strdup("XMPI"), cmdoptions,
				 XtNumber(cmdoptions), &argc, argv, 
				 (String*) fallbacks, NULL);

/*
 * Enabling this lets "editres" work with XMPI.  Might have to link with
 * -lXmu as well.
 */

#ifdef DEBUG
  XtAddEventHandler(xmpi_shell, (EventMask) 0, True,
		    _XEditResCheckMessages, NULL);
#endif

/*
 * Read in and check resources.
 */
  XtGetApplicationResources(xmpi_shell,
		      &app_res, resources, XtNumber(resources), NULL, 0);

  if (app_res.ap_help) {
    help();
    exit(0);
  }
  if (app_res.ap_bandwidth < 0) {
    app_res.ap_bandwidth = XMPI_BANDWIDTH;
  }
/*
 * Look for information string.
 */
  for (i = 1; i < argc; ++i) {

    if ((*argv[i] == '-') || (info)) {
      fprintf(stderr,
	      "xmpi (usage): xmpi [-h] [<info>]\n");
      exit(EUSAGE);
    } else {
      info = argv[i];
    }
  }

  if (info) {
    strncpy(xmpi_info, info, XMPI_PATHMAX);
  } else {
    xmpi_info[0] = 0;
  }

  XtAppAddActions(app, actions, XtNumber(actions));

  xmpitop = XtVaCreateWidget("top_mgr",
			     xmFormWidgetClass, xmpi_shell,
			     XmNmarginHeight, 2,
			     XmNmarginWidth, 2,
			     NULL);
/*
 * Create main application view.
 */
  w1 = xmpi_ctl_build(xmpitop);
  XtVaSetValues(w1,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);

  w2 = xmpi_vw_create(xmpitop);
  XtVaSetValues(w2,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, w1,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);

  XtManageChild(xmpitop);
  XSetIOErrorHandler((XIOErrorHandler) xmpi_bail);
  xmpi_atexit(xmpi_cleanup);
  XtRealizeWidget(xmpi_shell);

  if (xmpi_sys_init()) {
    perror("xmpi: system specific initialization");
    exit(errno);
  }
  xmpi_setminmax(winitmin, hinitmin, winitmax, hinitmax);

  XtAppMainLoop(app);
  return (0);
}

/*
 *	xmpi_cleanup
 *
 *	Function:	- calls cleanup routines from all modules
 */
void
xmpi_cleanup()
{
  xmpi_run_cleanup();
  xmpi_vendor_cleanup();
  xmpi_tr_cleanup();
  xmpi_sys_finalize();
}

/*
 *	xmpi_bail
 *
 *	Function:	- exits gracefully with 0 status
 */
void
xmpi_bail()
{
  xmpi_cleanup();
  _exit(0);
}

/*
 *	xmpi_busy
 *
 *	Function:	- make all windows in XMPI app. have busy cursors
 */
void
xmpi_busy()
{
  xmpi_busy_widget(xmpi_shell);
  xmpi_aschema_busy();
  xmpi_run_busy();
  xmpi_fo_busy();
  xmpi_kv_busy();
  xmpi_dt_busy();
  xmpi_tr_busy();
}

/*
 *	xmpi_unbusy
 *
 *	Function:	- make all windows in XMPI app. have normal cursors
 */
void
xmpi_unbusy()
{
  xmpi_unbusy_widget(xmpi_shell);
  xmpi_aschema_unbusy();
  xmpi_run_unbusy();
  xmpi_fo_unbusy();
  xmpi_kv_unbusy();
  xmpi_dt_unbusy();
  xmpi_tr_unbusy();
}

/*
 *        help
 *
 *        Function:     - prints helpful information on this command
 */
static void
help()
{
  printf("\nSynopsis:\txmpi [<options>] [<info>]\n");
  printf("\nDescription:\tRun and monitor MPI programs.\n");
  printf("\nOptions:        -h\tPrint this help message.\n");
}

/*
 *	xmpi_wmshell
 *
 *	Function:	- returns the first WMShell ancestor
 *	Accepts:	- widget
 *	Returns:	- window manager shell
 */
Widget
xmpi_wmshell(Widget w)
{
  while ((w) && (!XtIsWMShell(w))) {
    w = XtParent(w);
  }

  return (w);
}

/*
 *	xmpi_setminmax
 *
 *	Function:	- set min/max toplevel shell size
 *	Accepts:	- min width
 *			- min height
 *			- max width
 *			- max height
 */
void
xmpi_setminmax(int wmin, int hmin, int wmax, int hmax)
{
  Dimension height;

  Dimension width;

/*
 * The first time around, just store the drawable sizes.
 */
  if (winitmin < 0) {
    winitmin = wmin;
    hinitmin = hmin;
    winitmax = wmax;
    hinitmax = hmax;
  } else {
/*
 * The second time around, compute the width/height window overhead.
 */
    if (wsetup < 0) {
      XtVaGetValues(xmpi_shell, XmNwidth, &width,
		    XmNheight, &height, NULL);

      wsetup = width - wmax;
      hsetup = height - hmax;
    }
/*
 * Set the size boundaries.
 */
    XtVaSetValues(xmpi_shell,
		  XmNminWidth, wsetup + wmin,
		  XmNminHeight, hsetup + hmin,
		  XmNmaxWidth, wsetup + wmax,
		  XmNmaxHeight, hsetup + hmax,
		  NULL);
  }
}

/*
 *	xmpi_setsize
 *
 *	Function:	- set toplevel shell size
 *	Accepts:	- width
 *			- height
 */
void
xmpi_setsize(int width, int height)
{
  XResizeWindow(XtDisplay(xmpi_shell), XtWindow(xmpi_shell),
		(unsigned) wsetup + width, (unsigned) hsetup + height);
}

/*
 *	xmpi_seticon
 *
 *	Function:	- set the tool icon
 *	Accepts:	- icon pixmap
 */
void
xmpi_seticon(Pixmap iconpix)
{
  Window win;

  Window root;

  Display *disp;

  int x, y;

  unsigned int width, height, bwidth, depth;

  disp = XtDisplay(xmpi_shell);
  XtVaGetValues(xmpi_shell, XmNiconWindow, &win, NULL);

  if (!win) {
    if (!XGetGeometry(disp, iconpix, &root, &x, &y,
		      &width, &height, &bwidth, &depth) ||
	!(win = XCreateSimpleWindow(disp, root, 0, 0,
				    width, height, (unsigned int) 0,
				    CopyFromParent, CopyFromParent))) {

      XtVaSetValues(xmpi_shell, XmNiconPixmap, iconpix,
		    NULL);
      return;
    }
    XtVaSetValues(xmpi_shell, XmNiconWindow, win, NULL);
  }
  XSetWindowBackgroundPixmap(disp, win, iconpix);
  XClearWindow(disp, win);
}
