#!/bin/sh
set -o nounset

oneTimeSetUp() {
	# Load the ./play.it library
	. lib/libplayit2.sh
	# Set up a couple fake game scripts
	TEST_TEMP_DIR=$(mktemp --directory)
	export TEST_TEMP_DIR
	mkdir \
		"${TEST_TEMP_DIR}/00_scripts-collection" \
		"${TEST_TEMP_DIR}/10_another-scripts-collection" \
		"${TEST_TEMP_DIR}/99_old-scripts-collection"
	cat > "${TEST_TEMP_DIR}/00_scripts-collection/play-some-game.sh" <<- EOF
	ARCHIVE_BASE_0_NAME='some_game_archive.tar.gz'
	ARCHIVE_BASE_0_MD5='d41d8cd98f00b204e9800998ecf8427e'
	ARCHIVE_BASE_1_NAME='some_game_archive.tar.gz'
	ARCHIVE_BASE_1_MD5='d401431ff9c1c7526c3104194409bd6e'
	EOF
	cat > "${TEST_TEMP_DIR}/00_scripts-collection/play-some-other-game.sh" <<- EOF
	ARCHIVE_BASE_0_NAME='another_game_archive.zip'
	EOF
	cat > "${TEST_TEMP_DIR}/10_another-scripts-collection/play-some-game.sh" <<- EOF
	ARCHIVE_BASE_0_NAME='some_game_archive.tar.gz'
	ARCHIVE_BASE_0_MD5='d41d8cd98f00b204e9800998ecf8427e'
	EOF
	cat > "${TEST_TEMP_DIR}/10_another-scripts-collection/play-yet-another-game.sh" <<- EOF
	ARCHIVE_BASE_0_NAME='some_game_archive.tar.gz'
	ARCHIVE_BASE_0_MD5='d401431ff9c1c7526c3104194409bd6e'
	EOF

	export PLAYIT_OPTION_COLLECTION_PATH="$TEST_TEMP_DIR"
}

oneTimeTearDown() {
	rm --force --recursive "$TEST_TEMP_DIR"
}

test_games_list_scripts_all() {
	local game_scripts_list game_scripts_list_expected

	game_scripts_list_expected="${TEST_TEMP_DIR}/00_scripts-collection/play-some-game.sh
${TEST_TEMP_DIR}/00_scripts-collection/play-some-other-game.sh
${TEST_TEMP_DIR}/10_another-scripts-collection/play-some-game.sh
${TEST_TEMP_DIR}/10_another-scripts-collection/play-yet-another-game.sh"
	game_scripts_list=$(games_list_scripts_all)
	assertEquals "$game_scripts_list_expected" "$game_scripts_list"
}

test_games_find_scripts_for_archive() {
	local game_scripts game_scripts_expected

	game_scripts_expected="${TEST_TEMP_DIR}/00_scripts-collection/play-some-game.sh
${TEST_TEMP_DIR}/10_another-scripts-collection/play-some-game.sh
${TEST_TEMP_DIR}/10_another-scripts-collection/play-yet-another-game.sh"
	game_scripts=$(games_find_scripts_for_archive 'some_game_archive.tar.gz')
	assertEquals "$game_scripts_expected" "$game_scripts"
}

test_games_find_script_for_archive() {
	local game_script game_script_expected SOURCE_ARCHIVE_NAME

	game_script_expected="${TEST_TEMP_DIR}/00_scripts-collection/play-some-other-game.sh"
	game_script=$(games_find_script_for_archive 'another_game_archive.zip')
	assertEquals "$game_script_expected" "$game_script"

	SOURCE_ARCHIVE_NAME='some_game_archive.tar.gz'
	SOURCE_ARCHIVE_PATH="${TEST_TEMP_DIR}/some_game_archive.tar.gz"
	touch "$SOURCE_ARCHIVE_PATH"
	game_script_expected="${TEST_TEMP_DIR}/00_scripts-collection/play-some-game.sh"
	game_script=$(games_find_script_for_archive 'some_game_archive.tar.gz' 2>/dev/null)
	assertEquals 'Failed to select a single game script after a MD5-based archive identification.' \
		"$game_script_expected" "$game_script"
	rm "$SOURCE_ARCHIVE_PATH"
}

test_script_version() {
	local version_string script_version

	script_version='19700101.1'
	version_string=$(script_version)
	assertEquals '19700101.1' "$version_string"

	# An error is thrown if the script version is not set
	unset script_version
	assertFalse 'script_version'

	# An error is thrown is the script version does not follow the expected "YYYYMMDD.N" format
	script_version='19700101'
	assertFalse 'script_version'
}
