/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/interactive_test_data_adaptor.hpp
 *
 * @brief [LEVEL: beta] @ref
 * diagnostics::unittest::Interactive_Test_Data_Adaptor class
 *
 * $Id: interactive_test_data_adaptor.hpp,v 1.7 2005/06/23 09:54:25 esdentem Exp $
 *
 * @test none
 *
 * @todo add test
 * 
 * @author Christian Schallhart
 */
#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__INTERACTIVE_TEST_DATA_ADAPTOR_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__INTERACTIVE_TEST_DATA_ADAPTOR_HPP__INCLUDE_GUARD

// used as base class
#include <diagnostics/unittest/test_system/test_data_adaptor.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

/**
 * @class Interactive_Test_Data_Adaptor diagnostics/unittest/test_system/interactive_test_data_adaptor.hpp
 *
 * @brief The Interactive_Test_Data_Adaptor adapts a @ref
 * Test_Data_Source to a @ref Test_Data such that in case of a mssing
 * entry or a missmatching entry in the @ref Test_Data_Source a
 * corresponding hook-method is called. 
 *
 * @nosubgrouping
 */
class Interactive_Test_Data_Adaptor :
    public Test_Data_Adaptor
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Interactive_Test_Data_Adaptor Self;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disabled Creation
     * @{
     */
private:
    Interactive_Test_Data_Adaptor(Self const & other);
    Self & operator=(Self const & other);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief the only way of construction. backend()==NULL
     *
     * @throw never
     */
	Interactive_Test_Data_Adaptor();

    /**
     * @brief noop
     *
     * @throw never
     */
    virtual ~Interactive_Test_Data_Adaptor();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Implementation of the @ref Test_Data_Adaptor Interface
     * (including the @ref Test_Data interface)
     *
     * @{
     */
public:
    virtual void attach(Test_Data_Source * const test_data_source);
    virtual Test_Data_Source* backend() const;

    /**
     * @brief If backend().exists_entry(@a id)==false, p_missing_test_data(@a
	 * id,data) with data="" is called. If it returns true, then a new
	 * entry id,data (data possibly modified) is created.
	 *
     * @pre backend()!=NULL
     *
     * @throw Test_System_Exception if backend()==NULL
     * @throw Test_Exception if the entry is missing and p_missing_test_data returns false
     */
    virtual ::std::string const & get(::std::string const & id) const;
    
    /**
     * @brief checks backend().get_entry(@a id)== @a data -- if true,
     * true is returned. If false or the entry does not exist,
     * either p_missing_test_data or p_mismatching_test_data is called.
     *
     * If backend().exists_entry(@a id) and backend().get_entry(@a
     * id)==@a data, then true is returned.
     *
	 * If backend().exists_entry(@a id)==false, then
	 * p_missing_test_data(@a id, @a data) is called. If it returns
	 * true, then an entry @a id, @a data (@a data possibly modified) is created.
	 * Otherwise, Test_Exception is thrown.
	 *
	 * If backend().exists_entry(@a id)==true but
	 * backend().get_entry(@a id)!= @a data, p_missing_test_data(@a
	 * id, backend().get_entry(@a id), @ data) is invoked. If it
	 * returns true, then the entry is changed to @a id, @a data (@a
	 * data possibly modified). Otherwise, Test_Exception is thrown.
	 *
     * @pre backend()!=NULL
     *
     * @throw Test_System_Exception if backend()==NULL
     * @throw Test_Exception if there is missing/mismatching and
     * uncorrected request (see above)
     */    
    virtual bool compare(::std::string const & id,::std::string const & data) const;
    // @}


	////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Interaction Interface
     * @{
     */
protected:
	/**
     * @brief If an entry in the testdata database is missing, this
     * method is called. @a id is the id of the missing data while @a
     * data is the proposed value.
	 *
	 * @return true if the @a id is a accepted with a possible
	 * modified value @a data
	 *
	 * @throw never
     */ 
	virtual bool p_missing_test_data(::std::string const & id, 
									 ::std::string & data) const =0;

	/**
     * @brief If an entry in the testdata database is not matched,
     * this method is called. @a id is the id of the mismatched data
     * while @a data is the old stored value. In @a new_data the newly
     * occurring data is stored.
	 *
	 * @return true if the @a id is a accepted with a possible
	 * modified value @a new_data
	 *
	 * @throw never
     */ 
	virtual bool p_mismatching_test_data(::std::string const & id,
										 ::std::string const & data,
										 ::std::string & new_data) const =0;
	// @}


    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated State
     * @{
     */    
private:
    Test_Data_Source * m_test_data_source;
    // @}
};

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif
// vim:ts=4:sw=4
