/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/


#include "keyboard_keyboard.h"
#include "scale_info.h"
#include "input.h"
#include "key_consts.h"
#include "ui_list.h"
#include "audio.h"
#include "main.h"
#include "din.h"
#include "ansi_color_codes.h"
#include "audio.h"
#include <math.h>

using namespace std;

extern int mousex, mousey;

extern din din0;
extern beat2value octave_shift;

extern float ATTACK_TIME, DECAY_TIME, DELTA_TIME;
extern float DELTA_BPM;
extern float NOTE_VOLUME;

extern float PITCH_BEND;
static float bendx = 0;

extern audio_out aout;
extern viewport view;
extern font fnt;
extern chrono clk;

extern ui_list uis;

keyboard_keyboard::keyboard_keyboard () :

wave ("waveform2.crv"), waved ("waveform2.ed"), wavlis (wave_listener::KEYBOARD_KEYBOARD),
attackcrv ("attack.crv"), decaycrv ("decay.crv"),
attacked ("attack.ed"), decayed ("decay.ed"),
attacklis (this), decaylis (this),
attacklib ("attack.lib", 1), decaylib ("decay.lib"),
helptext ("keyboard-keyboard.hlp"), velcrv ("velocity.crv"), vellis (this), veled ("velocity.ed"), vellib ("velocity.lib"), velsol (&velcrv)
{

  name = "keyboard-keyboard";

  /*
   * assign computer keyboard keys to notes of the scale
   *
   * assumptions:
   *  3 octaves
   *  scale must have 5 or 6 or 7 notes per octave
  */

  vector<key_info> ki;

  int ky [] = { // the keyboard keys
    k_t, k_r, k_e, k_w, k_q, k_a, k_s,
    k_d, k_f, k_g, k_h, k_j, k_k, k_l,
    k_semicolon, k_quote, k_right_bracket, k_left_bracket, k_p, k_o, k_i,
    k_u
  };

  char kz [] = { // the keys as chars
    't', 'r', 'e', 'w', 'q', 'a', 's',
    'd', 'f', 'g', 'h', 'j', 'k', 'l',
    ';', '\'', ']', '[', 'p', 'o', 'i', 'u',
  };

  // assign keys to scale
  for (int i = 5, j = 2; i < 8; ++i, --j) {
    ki.clear ();
    for (int k = 0, noct = 3, l = noct * i + 1; k < l; ++k) {
      int jk = j + k;
      ki.push_back ( key_info (ky [jk], kz [jk]) );
    }
    scale2keybd [i] = ki;
  }

  // setup waveform
  extern curve_library wavlib;
  waved.add (&wave, &wavlis);
  waved.attach_library (&wavlib);
  wavlis.edited (&waved, 0);

  // setup attack curve
  attacked.add (&attackcrv, &attacklis);
  attacked.attach_library (&attacklib);

  // setup decay curve
  decayed.add (&decaycrv, &decaylis);
  decayed.attach_library (&decaylib);

  // setup velocity curve (used on MIDI triggered notes)
  veled.add (&velcrv, &vellis);
  veled.attach_library (&vellib);

  prev_mousex = prev_mousey = -1;

}

keyboard_keyboard::~keyboard_keyboard () {
  wave.save ("waveform2.crv");
  attackcrv.save ("attack.crv");
  decaycrv.save ("decay.crv");
  velcrv.save ("velocity.crv");
  cout << FAIL << "--- destroyed keyboard-keyboard ---" << ENDL;
}

void reduce (float& what, float& by, float limit) {
  what -= by;
  if (what <= limit) what = by;
}

void keyboard_keyboard::bg () {
  remove_finished_notes (); // kill notes decayed to silence
}

bool keyboard_keyboard::handle_input () {

  din0.dinfo.delay = uis.w_delay.is_on ();

  mouse_bend (); // pitch bend via mouse x movement

  // trigger notes
  //
  for (int i = 0, j = keys.size (); i < j; ++i) {
    key_info& ki = keys[i];
    if (keypressed (ki.id)) { // pressed key of note
      attack_note (ki.id); // trigger note
      ki.attacked = 1;
    } else if (!keydown (ki.id) && ki.attacked) { // released key of note
      decay_note (ki.id); // decay
      ki.attacked = 0;
    }
  }

  // octave shift
  //
  if (keypressed (k_z)) modulate_up ();
  else if (keypressed (k_x)) modulate_down ();
  else if (keypressedd (k_f11)) { // decrease octave shift bpm
    octave_shift.set_bpm (octave_shift.get_bpm () - DELTA_BPM);
    cons << "octave shift bpm: " << octave_shift.get_bpm () << eol;
  } else if (keypressedd (k_f12)) { // increase octave bpm
    octave_shift.set_bpm (octave_shift.get_bpm () + DELTA_BPM);
    cons << "octave shift bpm: " << octave_shift.get_bpm () << eol;
  }

  else if (keypressedd (k_n)) { // reduce attack time
    reduce (ATTACK_TIME, DELTA_TIME, 0);
    cons << console::yellow << "attack time: " << ATTACK_TIME << eol;
  }
  else if (keypressedd (k_m)) { // increase attack time
    ATTACK_TIME += DELTA_TIME;
    cons << console::yellow << "attack time: " << ATTACK_TIME << eol;
  }
  else if (keypressedd (k_comma)) { // reduce decay time
    reduce (DECAY_TIME, DELTA_TIME, 0);
    cons << console::yellow << "decay time: " << DECAY_TIME << eol;
  }
  else if (keypressedd (k_period)) { // increase decay time
    DECAY_TIME += DELTA_TIME;
    cons << console::yellow << "decay time: " << DECAY_TIME << eol;
  }

  else if (keypressed (k_f1)) helptext(); // show help

  return true;

}

void keyboard_keyboard::setup_notes (int overwrite) {

  // setup 3 octaves worth of notes of the scale
  //

  notes.clear ();

  extern scale_info scaleinfo;
  vector<std::string>& scale_notes = scaleinfo.notes;

  int nnotes = scale_notes.size () - 1; // bcos note & octave included in scale definition

  float tonic = scaleinfo.tonic / 2; // start from low octave

  extern map<std::string, float> INTERVALS; // tuning

  // 3 octaves worth of notes
  for (int i = 0, noct = 3; i < noct; ++i) {
    for (int j = 0; j < nnotes; ++j) {
      std::string& name = scale_notes [j];
      notes.push_back (note());
      note& last = notes [notes.size() - 1];
      last.name = name;
      last.set_frequency (tonic * INTERVALS [name]);
    }
    tonic *= 2; // next octave
  }

  // bring up the rear
  notes.push_back (note());
  note& last = notes [notes.size() - 1];
  last.name = "1";
  last.set_frequency (tonic);

  memset (keybd2notes, 0, MAX_KEYS * sizeof (int));

  if (overwrite) keys = scale2keybd [nnotes];

  // green tonic key
  for (int i = 0, j = keys.size (), k = 0; i < j; ++i) {
    key_info& ki = keys[i];
    keybd2notes [ki.id] = k;
    if (notes[k++].name == "1") {
      ki.r = ki.b = 0;
    }
  }

}

void keyboard_keyboard::attack_note (int ky) {

  int ni = keybd2notes [ky];
  note& n = notes [ni];
  key_info& ki = keys [ni];

  static rnd<float> rd (0, 1);
  float r = rd(), g = rd(), b = rd ();

  marker_x = mousex;
  triggered_notes.push_back (triggered_note (n, ky, &wave, &attackcrv, &decaycrv, NOTE_VOLUME, ki.x, ki.y, r, g, b, marker_x));

}

void keyboard_keyboard::decay_note (int ky) {

  extern float DECAY_TIME;
  extern float SAMPLE_DURATION;

  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j; ++i) {
    triggered_note& ti = *i;
    if (ti.key == ky) {
      if (ti.state == triggered_note::ATTACK) {
        ti.state = triggered_note::DECAY;
        if (ti.volume_max == 0) {
          ti.state = triggered_note::FINISHED;
        } else {
          ti.decay_time = fabs (ti.decay_start_volume) / ti.volume_max * DECAY_TIME;
          if (ti.decay_time == 0) ti.state = triggered_note::FINISHED; else ti.delta_decay = SAMPLE_DURATION * 1. / ti.decay_time;
          ti.abs_decay = 0;
        }
      }
    }
  }

}

void keyboard_keyboard::update_attack () {
  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j; ++i) (*i).attack.update ();
}

void keyboard_keyboard::update_decay () {
  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j; ++i) (*i).decay.update ();
}

void keyboard_keyboard::update_waveform () {
  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j; ++i) (*i).sol.update ();
}

triggered_note::triggered_note (const note& n, int k, multi_curve* wav, multi_curve* atk, multi_curve* dk, float vmax, float xx, float yy, float rr, float gg, float bb, int mox) {

  tn = n;
  start_hz = tn.hz; // for pitch bend

  sol (wav);
  player.set_step (tn.step);

  key = k;

  attack (atk);
  decay (dk);

  state = ATTACK;
  abs_attack = 0;

  extern float ATTACK_TIME;
  extern float SAMPLE_DURATION;

  delta_attack = SAMPLE_DURATION * 1. / ATTACK_TIME;

  x = xx; y = yy;

  r = rr; g = gg; b = bb;

  mx = mox;
  bendx = 0;

  volume_now = 0;
  volume_max = vmax;

  decay_time = 0;
  decay_start_volume = 0;

}

void triggered_note::eval (float* left, float* right, float* wav, float* vol, int n) {

  player.set_wave (&sol);

  extern audio_out aout;

  switch (state) {

    case ATTACK:

      attack (abs_attack, delta_attack, n, vol, _atmin, _gotog);
      for (int i = 0; i < n; ++i) vol[i] *= volume_max;
      volume_now = vol[n-1];

      player.solve (wav, n);
      player (left, n, wav, vol);
      memcpy (right, left, aout.samples_channel_size);
      decay_start_volume = vol[n-1];

      break;

    case DECAY:

      decay (abs_decay, delta_decay, n, vol, _atmin, _atmax);

      for (int i = 0; i < n; ++i) vol[i] *= decay_start_volume;

      player.solve (wav, n);
      player (left, n, wav, vol);
      memcpy (right, left, aout.samples_channel_size);

      volume_now = vol[n-1];

      float x, y; decay.mcrv->get_vertex (decay.mcrv->num_vertices () - 1, x, y);
      if (abs_decay >= x) state = FINISHED;

      break;

  }


}

void triggered_note::set_frequency (float f) {
  tn.set_frequency (f);
  player.set_step (tn.step);
}

void keyboard_keyboard::render_audio (float* L, float* R) {

  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j; ++i) {
    triggered_note& ti = *i;
    if (ti.state != triggered_note::FINISHED) ti.eval (L, R, aout.result, aout.vol, aout.samples_per_channel);
  }

}

void keyboard_keyboard::remove_finished_notes () {
  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j;) {
    triggered_note& ti = *i;
    if (ti.state == triggered_note::FINISHED) {
      list<triggered_note>::iterator now = i, next = ++i;
      triggered_notes.erase (now);
      i = next;
      j = triggered_notes.end ();
    } else ++i;
  }
}

void keyboard_keyboard::calc_visual_params () {

  static const int pad = 4, halfpad = pad / 2;
  int nkeys = keys.size ();
  int delta = view.xmax / (nkeys + pad);
  int x = halfpad * delta;
  int half_height = view.ymax / 2;
  for (int i = 0; i < nkeys; ++i) {
    key_info& ki = keys[i];
    ki.x = x;
    ki.y = half_height;
    x += delta;
  }

  extern float NOTE_VOLUME;
  arm = 0.35 * view.ymax / NOTE_VOLUME;

  int lh = get_line_height ();
  ymarker = 2 * lh + 2;
  ychars = ymarker + lh;

  if (nkeys) spacing = (view.width - 2 * keys[0].x) / 12;

}

void keyboard_keyboard::draw () {

  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  glOrtho (0, view.xmax, 0, view.ymax, -1, 1);

  glMatrixMode (GL_MODELVIEW);
  glLoadIdentity ();

  // mark midi notes
  glEnable (GL_LINE_STIPPLE);
  glLineStipple (1, 0x1010);
  glColor3f (1, 1, 1);
  if (aout.num_midi_con) {
    int startx = keys[0].x, starty = keys[0].y;
    int marker = 20;
    for (int i = 0, num_notes = 12; i < num_notes; ++i) {
      glBegin (GL_LINES);
        glVertex2i (startx - marker, starty);
        glVertex2i (startx + marker, starty);
        glVertex2i (startx, starty - marker);
        glVertex2i (startx, starty + marker);
      glEnd ();
      startx += spacing;
    }
  }

  glBegin (GL_LINES);
    glVertex2i (marker_x, 0);
    glVertex2i (marker_x, view.ymax);
  glEnd ();

  glDisable (GL_LINE_STIPPLE);

  glEnable (GL_BLEND);
  glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  const float alpha = 0.9;
  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j; ++i) {

    triggered_note& ti = *i;

    // fill
    glColor4f (ti.r, ti.g, ti.b, alpha);
    int arm1 = ti.volume_now * arm;
    glRecti (ti.x - arm1 - ti.bendx, ti.y - arm1, ti.x + arm1 + ti.bendx, ti.y + arm1);

    // outline
    glColor4f (alpha, alpha, alpha, alpha);
    int xp = ti.x + arm1 + ti.bendx, xm = ti.x - arm1 - ti.bendx, yp = ti.y + arm1, ym = ti.y - arm1;
    glLineWidth (2);
    glBegin (GL_LINE_LOOP);
      glVertex2i (xm, ym);
      glVertex2i (xp, ym);
      glVertex2i (xp, yp);
      glVertex2i (xm, yp);
    glEnd ();
    glLineWidth (1);
  }

  glDisable (GL_BLEND);


  for (int i = 0, j = keys.size (); i < j; ++i) {
    key_info& ki = keys[i];
    glColor3f (ki.r, ki.g, ki.b);
    fnt.draw_char (ki.ch, ki.x, ychars);
    if (ki.attacked) { // hilite key that triggered last note
      glBegin (GL_LINES);
        glVertex2i (ki.x, ymarker);
        glVertex2i (ki.x + fnt.max_char_width, ymarker);
      glEnd ();
    }
  }


}

void keyboard_keyboard::setup_midi_notes () {

  const float middlec = 261.626;
  float start = middlec / 32;

  extern int NUM_INTERVALS;
  extern std::vector<float> INTERVALS_SEQ;

  int id = 0;
  while (1) {
    int j = NUM_INTERVALS - 1;
    for (int i = 0; i < j; ++i) {
      if (id < MIDI_MAX) {
        midi_notes[id].set_frequency (start *  INTERVALS_SEQ [i]);
        id++;
      } else return;
    }
    start = start * INTERVALS_SEQ [j];
  }

}

void keyboard_keyboard::note_on (unsigned char id, unsigned char vel) {

  note& n = midi_notes [id];
  int i12 = id % 12;
  int ci = color_index [i12];
  float r = note_color[ci].r;
  float g = note_color[ci].g;
  float b = note_color[ci].b;

  marker_x = mousex;
  triggered_notes.push_back (triggered_note (n, id, &wave, &attackcrv, &decaycrv, velsol(vel) * NOTE_VOLUME, keys[0].x + i12 * spacing, keys[0].y, r, g, b, marker_x));

}

void keyboard_keyboard::note_off (unsigned char id) {
  decay_note (id);
}

void keyboard_keyboard::pitch_bend (float v) {
  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j; ++i) {
    triggered_note& ti = *i;
    float pv = PITCH_BEND * v;
    ti.set_frequency (ti.start_hz + pv);
    bendx = fabs (pv);
  }
}

void keyboard_keyboard::mouse_bend () {

  float PITCH_BEND_PIXEL = 0.0075 * PITCH_BEND;

  int delta_x = 0;
  float pv = 0;

  for (list<triggered_note>::iterator i = triggered_notes.begin (), j = triggered_notes.end (); i != j; ++i) {
    triggered_note& ti = *i;
    delta_x = mousex - ti.mx;
    pv = delta_x * PITCH_BEND_PIXEL;
    ti.set_frequency (ti.start_hz + pv);
    ti.bendx = fabs (pv);
  }

}

void keyboard_keyboard::leave () {
  prev_mousex = mousex;
  prev_mousey = mousey;
}

void keyboard_keyboard::enter () {
  if (prev_mousex == -1) warp_pointer_abs (view.xmax / 2, view.ymax / 2); else warp_pointer_abs (prev_mousex, prev_mousey);
}
