/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#include "dingl.h"
#include "oscilloscope.h"
#include "console.h"
#include "font.h"
#include "viewwin.h"
#include "input.h"

#include <string>
#include <algorithm>

using namespace std;

extern console cons;

oscilloscope::oscilloscope () : win (0, 0, 2 * MAX_SAMPLES, 250) {

  vertex_array = new float [8 * MAX_SAMPLES];

  color_array = new float [16 * MAX_SAMPLES];
  /*lr = 1; lg = 0.7; lb = 1;
  rr = 1; rg = 1; rb = 0.7;*/

  viewr = 0;
  addr = 1;
  if (!load_last ()) {
    paused = false;
    visible = true;
    win (0, 0, 2 * MAX_SAMPLES, 250);
  }
  calc_draw_params ();
}

bool oscilloscope::load_last () {
  extern string dotdin;
  ifstream file ((dotdin + "oscilloscope").c_str(), ios::in);
  if (!file) return false;
  string ignore;
  int l, b, r, t; file >> ignore >> l >> b >> r >> t; win (l, b, r, t);
  file >> ignore >> paused >> ignore >> visible;
  return true;
}

oscilloscope::~oscilloscope () {
  delete[] vertex_array;
  delete[] color_array;
  save_last ();
}

bool oscilloscope::save_last () {
  extern string dotdin;
  ofstream file ((dotdin + "oscilloscope").c_str(), ios::out);
  if (!file) return false;
  file << "win " << win.left << ' ' << win.bottom << ' ' << win.right << ' ' << win.top << endl;
  file << "paused " << paused << endl;
  file << "visible " << visible << endl;
  return true;
}

void oscilloscope::calc_draw_params () {

  leftx = win.left;
  num_samples = min (win.width / 2, (int) MAX_SAMPLES);
  if (num_samples == 0) num_samples = 1;
  rightx = leftx + num_samples;
  base = (win.bottom + win.top)/2;
  ndraw = 4 * num_samples;
  endx = rightx + num_samples;

  int dheight = 5;
  pick_win (win.left, win.midy - dheight, win.right, win.midy + dheight);

}

bool oscilloscope::handle_input () {

  extern int lmb, rmb;
  static int clicked = 0, canmove = 0, rclicked = 0, canresize = 0;

  extern int mousex, mousey;
  extern viewport view;

  int x = mousex;
  int y = view.ymax - mousey;
  static int px = -1, py = -1;

  bool processed = false;

  if (lmb) {
    if (!clicked) {
      clicked = 1;
      if (inbox (pick_win, x, y)) {
        canmove = 1;
        if (px == -1) px = x;
        if (py == -1) py = y;
      } else {
        canmove = 0;
      }
    } else if (canmove) {
      win.move (x-px, y-py);
      calc_draw_params ();
      processed = true;
    }
  } else {
    clicked = 0;
    canmove = 0;
  }

  if (rmb) {
    if (!rclicked) {
      rclicked = 1;
      if (inbox (win, x, y)) {
        canresize = 1;
        if (px == -1) px = x;
        if (py == -1) py = y;
      } else canresize = 0;
    } else if (canresize) {
      win.resize (x-px, y-py);
      calc_draw_params ();
      processed = true;
    }
  } else {
    rclicked = 0;
    canresize = 0;
  }

  px = x;
  py = y;

  return processed;

}

void oscilloscope::toggle_pause () {
  paused = !paused;
  if (paused) cons << "scope is paused." << eol; else cons << "scope running." << eol;
}

void oscilloscope::toggle_visible () {
  visible = !visible;
  if (visible) cons << "scope is visible." << eol; else cons << "scope is hidden." << eol;
}

void oscilloscope::add_samples (float* outl, float* outr, int n) {
  for (int i = 0; i < n; ++i) {
    sample_t& sa = samples[addr];
    float l = outl[i], r = outr[i];
    sa.left = l; sa.right = r;
    sample_t::lmin = min (sa.left, sample_t::lmin);
    sample_t::lmax = max (sa.left, sample_t::lmax);
    sample_t::rmin = min (sa.right, sample_t::rmin);
    sample_t::rmax = max (sa.right, sample_t::rmax);
    if (++addr >= num_samples) {
      addr = 0;
      /*static int minner = 0, max_minner = 25;
      if (++minner > max_minner) {
        minner = 0;*/
        sample_t::lmin = sample_t::lmax = sample_t::rmin = sample_t::rmax = 0; // ok for oscilloscope.
      //}
    }
    if (addr == viewr) if (++viewr >= num_samples) viewr = 0;
  }
}

void oscilloscope::draw () {

  int x1 = leftx, x2 = rightx, y0 = base;

  static const int label_delta = 20;

  glColor3f (lr , lg, lb);
  static char buf [256];
  sprintf (buf, "min/max: %1.2f, %1.2f", sample_t::lmin, sample_t::lmax);
  draw_string (buf, x1, y0 + (int)(sample_t::lmax * win.height) + label_delta, 0);
  glBegin (GL_LINES);
    glVertex2i (leftx, win.midy);
    glVertex2i (rightx, win.midy);
  glEnd ();

  glColor3f (rr, rg, rb);
  sprintf (buf, "min/max: %1.2f, %1.2f", sample_t::rmin, sample_t::rmax);
  draw_string (buf, x2, y0 + (int)(sample_t::rmax * win.height) + label_delta, 0);
  glBegin (GL_LINES);
    glVertex2i (rightx, win.midy);
    glVertex2i (endx, win.midy);
  glEnd ();

  glEnableClientState (GL_VERTEX_ARRAY);
  glVertexPointer (2, GL_FLOAT, 0, vertex_array);

  glEnableClientState (GL_COLOR_ARRAY);
  glColorPointer (4, GL_FLOAT, 0, color_array);

  for (int i = 0, j = viewr, vi = 0, ci = 0; i < num_samples; ++i, vi += 8, ci += 16) {

    float l = samples[j].left, r = samples[j].right;
    float ly = l * win.height, ry = r * win.height;

    float lrr = lr, lgg = lg, lbb = lb; if (l > 1 || l < -1) { lrr = 1; lgg = 0; lbb = 0;}
    float rrr = rr, rgg = rg, rbb = rb; if (r > 1 || r < -1) { rrr = 1; rgg = 0; rbb = 0;}

    vertex_array [vi] = x1;
    vertex_array [vi+1] = y0;
    vertex_array [vi+2] = x1;
    vertex_array [vi+3] = y0 + ly;
    vertex_array [vi+4] = x2;
    vertex_array [vi+5] = y0;
    vertex_array [vi+6] = x2;
    vertex_array [vi+7] = y0 + ry;

    color_array [ci] = color_array[ci+4] = lrr;
    color_array [ci+1] = color_array[ci+5] = lgg;
    color_array [ci+2] = color_array[ci+6] = lbb;
    color_array [ci+3] = color_array[ci+7] = 1;
    color_array [ci+8] = color_array[ci+12] = rrr;
    color_array [ci+9] = color_array[ci+13] = rgg;
    color_array [ci+10] = color_array[ci+14] = rbb;
    color_array [ci+11] = color_array[ci+15] = 1;

    if (++j >= num_samples) j = 0;
    ++x1; ++x2;
  }

  glDrawArrays (GL_LINES, 0, ndraw);

  glDisableClientState (GL_VERTEX_ARRAY);
  glDisableClientState (GL_COLOR_ARRAY);

}
