////////////////////////////////////////////////////////////////////////////////////////
//
//  Copyright 2023 OVITO GmbH, Germany
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify it either under the
//  terms of the GNU General Public License version 3 as published by the Free Software
//  Foundation (the "GPL") or, at your option, under the terms of the MIT License.
//  If you do not alter this notice, a recipient may use your version of this
//  file under either the GPL or the MIT License.
//
//  You should have received a copy of the GPL along with this program in a
//  file LICENSE.GPL.txt.  You should have received a copy of the MIT License along
//  with this program in a file LICENSE.MIT.txt
//
//  This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND,
//  either express or implied. See the GPL or the MIT License for the specific language
//  governing rights and limitations.
//
////////////////////////////////////////////////////////////////////////////////////////

#pragma once

//-----------------------------------------------------------------------------
/*
 * Cube description:
 *         7 ________ 6           _____6__             ________
 *         /|       /|         7/|       /|          /|       /|
 *       /  |     /  |        /  |     /5 |        /  6     /  |
 *   4 /_______ /    |      /__4____ /    10     /_______3/    |
 *    |     |  |5    |     |    11  |     |     |     |  |   2 |
 *    |    3|__|_____|2    |     |__|__2__|     | 4   |__|_____|
 *    |    /   |    /      8   3/   9    /      |    /   |    /
 *    |  /     |  /        |  /     |  /1       |  /     5  /
 *    |/_______|/          |/___0___|/          |/_1_____|/
 *   0          1
 */
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Trivial case, where all atoms are either above or below the threshold meaning
// that they all belong to the same region.
static const signed char vertexRegion0[8] = {0, 0, 0, 0, 0, 0, 0, 0};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion1[16][8] = {
    /*   1: 0, , , , , , , :  0, 8, 3 */ {0, 1, 1, 1, 1, 1, 1, 1},
    /*   2:  ,1, , , , , , :  0, 1, 9 */ {0, 1, 0, 0, 0, 0, 0, 0},
    /*   4:  , ,2, , , , , :  1, 2,10 */ {0, 0, 1, 0, 0, 0, 0, 0},
    /*   8:  , , ,3, , , , :  3,11, 2 */ {0, 0, 0, 1, 0, 0, 0, 0},
    /*  16:  , , , ,4, , , :  4, 7, 8 */ {0, 0, 0, 0, 1, 0, 0, 0},
    /*  32:  , , , , ,5, , :  9, 5, 4 */ {0, 0, 0, 0, 0, 1, 0, 0},
    /*  64:  , , , , , ,6, : 10, 6, 5 */ {0, 0, 0, 0, 0, 0, 1, 0},
    /* 128:  , , , , , , ,7:  7, 6,11 */ {0, 0, 0, 0, 0, 0, 0, 1},
    /* 127: 0,1,2,3,4,5,6, :  7,11, 6 */ {0, 0, 0, 0, 0, 0, 0, 1},
    /* 191: 0,1,2,3,4,5, ,7: 10, 5, 6 */ {0, 0, 0, 0, 0, 0, 1, 0},
    /* 223: 0,1,2,3,4, ,6,7:  9, 4, 5 */ {0, 0, 0, 0, 0, 1, 0, 0},
    /* 239: 0,1,2,3, ,5,6,7:  4, 8, 7 */ {0, 0, 0, 0, 1, 0, 0, 0},
    /* 247: 0,1,2, ,4,5,6,7:  3, 2,11 */ {0, 0, 0, 1, 0, 0, 0, 0},
    /* 251: 0,1, ,3,4,5,6,7:  1,10, 2 */ {0, 0, 1, 0, 0, 0, 0, 0},
    /* 253: 0, ,2,3,4,5,6,7:  0, 9, 1 */ {0, 1, 0, 0, 0, 0, 0, 0},
    /* 254:  ,1,2,3,4,5,6,7:  0, 3, 8 */ {0, 1, 1, 1, 1, 1, 1, 1},
};
static const signed char triangleRegion1[16][2] = {
    /*   1: 0, , , , , , , :  0, 8, 3 */ {0, 1},
    /*   2:  ,1, , , , , , :  0, 1, 9 */ {1, 0},
    /*   4:  , ,2, , , , , :  1, 2,10 */ {1, 0},
    /*   8:  , , ,3, , , , :  3,11, 2 */ {1, 0},
    /*  16:  , , , ,4, , , :  4, 7, 8 */ {1, 0},
    /*  32:  , , , , ,5, , :  9, 5, 4 */ {1, 0},
    /*  64:  , , , , , ,6, : 10, 6, 5 */ {1, 0},
    /* 128:  , , , , , , ,7:  7, 6,11 */ {1, 0},
    /* 127: 0,1,2,3,4,5,6, :  7,11, 6 */ {0, 1},
    /* 191: 0,1,2,3,4,5, ,7: 10, 5, 6 */ {0, 1},
    /* 223: 0,1,2,3,4, ,6,7:  9, 4, 5 */ {0, 1},
    /* 239: 0,1,2,3, ,5,6,7:  4, 8, 7 */ {0, 1},
    /* 247: 0,1,2, ,4,5,6,7:  3, 2,11 */ {0, 1},
    /* 251: 0,1, ,3,4,5,6,7:  1,10, 2 */ {0, 1},
    /* 253: 0, ,2,3,4,5,6,7:  0, 9, 1 */ {0, 1},
    /* 254:  ,1,2,3,4,5,6,7:  0, 3, 8 */ {1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion2[24][8] = {
    /*   3: 0,1, , , , , , :  1, 8, 3, 9, 8, 1 */ {0, 0, 1, 1, 1, 1, 1, 1},
    /*   9: 0, , ,3, , , , :  0,11, 2, 8,11, 0 */ {0, 1, 1, 0, 1, 1, 1, 1},
    /*  17: 0, , , ,4, , , :  4, 3, 0, 7, 3, 4 */ {0, 1, 1, 1, 0, 1, 1, 1},
    /*   6:  ,1,2, , , , , :  9, 2,10, 0, 2, 9 */ {0, 1, 1, 0, 0, 0, 0, 0},
    /*  34:  ,1, , , ,5, , :  0, 5, 4, 1, 5, 0 */ {0, 1, 0, 0, 0, 1, 0, 0},
    /*  12:  , ,2,3, , , , :  3,10, 1,11,10, 3 */ {0, 0, 1, 1, 0, 0, 0, 0},
    /*  68:  , ,2, , , ,6, :  1, 6, 5, 2, 6, 1 */ {0, 0, 1, 0, 0, 0, 1, 0},
    /* 136:  , , ,3, , , ,7:  7, 2, 3, 6, 2, 7 */ {0, 0, 0, 1, 0, 0, 0, 1},
    /*  48:  , , , ,4,5, , :  9, 7, 8, 5, 7, 9 */ {0, 0, 0, 0, 1, 1, 0, 0},
    /* 144:  , , , ,4, , ,7:  6, 8, 4,11, 8, 6 */ {0, 0, 0, 0, 1, 0, 0, 1},
    /*  96:  , , , , ,5,6, : 10, 4, 9, 6, 4,10 */ {0, 0, 0, 0, 0, 1, 1, 0},
    /* 192:  , , , , , ,6,7: 11, 5,10, 7, 5,11 */ {0, 0, 0, 0, 0, 0, 1, 1},
    /*  63: 0,1,2,3,4,5, , : 11,10, 5, 7,11, 5 */ {0, 0, 0, 0, 0, 0, 1, 1},
    /* 159: 0,1,2,3,4, , ,7: 10, 9, 4, 6,10, 4 */ {0, 0, 0, 0, 0, 1, 1, 0},
    /* 111: 0,1,2,3, ,5,6, :  6, 4, 8,11, 6, 8 */ {0, 0, 0, 0, 1, 0, 0, 1},
    /* 207: 0,1,2,3, , ,6,7:  9, 8, 7, 5, 9, 7 */ {0, 0, 0, 0, 1, 1, 0, 0},
    /* 119: 0,1,2, ,4,5,6, :  7, 3, 2, 6, 7, 2 */ {0, 0, 0, 1, 0, 0, 0, 1},
    /* 187: 0,1, ,3,4,5, ,7:  1, 5, 6, 2, 1, 6 */ {0, 0, 1, 0, 0, 0, 1, 0},
    /* 243: 0,1, , ,4,5,6,7:  3, 1,10,11, 3,10 */ {0, 0, 1, 1, 0, 0, 0, 0},
    /* 221: 0, ,2,3,4, ,6,7:  0, 4, 5, 1, 0, 5 */ {0, 1, 0, 0, 0, 1, 0, 0},
    /* 249: 0, , ,3,4,5,6,7:  9,10, 2, 0, 9, 2 */ {0, 1, 1, 0, 0, 0, 0, 0},
    /* 238:  ,1,2,3, ,5,6,7:  4, 0, 3, 7, 4, 3 */ {0, 1, 1, 1, 0, 1, 1, 1},
    /* 246:  ,1,2, ,4,5,6,7:  0, 2,11, 8, 0,11 */ {0, 1, 1, 0, 1, 1, 1, 1},
    /* 252:  , ,2,3,4,5,6,7:  1, 3, 8, 9, 1, 8 */ {0, 0, 1, 1, 1, 1, 1, 1},
};
static const signed char triangleRegion2[24][4] = {
    /*   3: 0,1, , , , , , :  1, 8, 3, 9, 8, 1 */ {0, 1, 0, 1},
    /*   9: 0, , ,3, , , , :  0,11, 2, 8,11, 0 */ {0, 1, 0, 1},
    /*  17: 0, , , ,4, , , :  4, 3, 0, 7, 3, 4 */ {0, 1, 0, 1},
    /*   6:  ,1,2, , , , , :  9, 2,10, 0, 2, 9 */ {1, 0, 1, 0},
    /*  34:  ,1, , , ,5, , :  0, 5, 4, 1, 5, 0 */ {1, 0, 1, 0},
    /*  12:  , ,2,3, , , , :  3,10, 1,11,10, 3 */ {1, 0, 1, 0},
    /*  68:  , ,2, , , ,6, :  1, 6, 5, 2, 6, 1 */ {1, 0, 1, 0},
    /* 136:  , , ,3, , , ,7:  7, 2, 3, 6, 2, 7 */ {1, 0, 1, 0},
    /*  48:  , , , ,4,5, , :  9, 7, 8, 5, 7, 9 */ {1, 0, 1, 0},
    /* 144:  , , , ,4, , ,7:  6, 8, 4,11, 8, 6 */ {1, 0, 1, 0},
    /*  96:  , , , , ,5,6, : 10, 4, 9, 6, 4,10 */ {1, 0, 1, 0},
    /* 192:  , , , , , ,6,7: 11, 5,10, 7, 5,11 */ {1, 0, 1, 0},
    /*  63: 0,1,2,3,4,5, , : 11,10, 5, 7,11, 5 */ {0, 1, 0, 1},
    /* 159: 0,1,2,3,4, , ,7: 10, 9, 4, 6,10, 4 */ {0, 1, 0, 1},
    /* 111: 0,1,2,3, ,5,6, :  6, 4, 8,11, 6, 8 */ {0, 1, 0, 1},
    /* 207: 0,1,2,3, , ,6,7:  9, 8, 7, 5, 9, 7 */ {0, 1, 0, 1},
    /* 119: 0,1,2, ,4,5,6, :  7, 3, 2, 6, 7, 2 */ {0, 1, 0, 1},
    /* 187: 0,1, ,3,4,5, ,7:  1, 5, 6, 2, 1, 6 */ {0, 1, 0, 1},
    /* 243: 0,1, , ,4,5,6,7:  3, 1,10,11, 3,10 */ {0, 1, 0, 1},
    /* 221: 0, ,2,3,4, ,6,7:  0, 4, 5, 1, 0, 5 */ {0, 1, 0, 1},
    /* 249: 0, , ,3,4,5,6,7:  9,10, 2, 0, 9, 2 */ {0, 1, 0, 1},
    /* 238:  ,1,2,3, ,5,6,7:  4, 0, 3, 7, 4, 3 */ {1, 0, 1, 0},
    /* 246:  ,1,2, ,4,5,6,7:  0, 2,11, 8, 0,11 */ {1, 0, 1, 0},
    /* 252:  , ,2,3,4,5,6,7:  1, 3, 8, 9, 1, 8 */ {1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion3_1[24][8] = {
    /*   5: 0, ,2, , , , , :  0, 8, 3, 1, 2,10 */ {0, 1, 2, 1, 1, 1, 1, 1},
    /*  33: 0, , , , ,5, , :  9, 5, 4, 0, 8, 3 */ {0, 1, 1, 1, 1, 2, 1, 1},
    /* 129: 0, , , , , , ,7:  3, 0, 8,11, 7, 6 */ {0, 1, 1, 1, 1, 1, 1, 2},
    /*  10:  ,1, ,3, , , , :  1, 9, 0, 2, 3,11 */ {0, 1, 0, 2, 0, 0, 0, 0},
    /*  18:  ,1, , ,4, , , :  0, 1, 9, 8, 4, 7 */ {0, 1, 0, 0, 2, 0, 0, 0},
    /*  66:  ,1, , , , ,6, :  9, 0, 1, 5,10, 6 */ {0, 1, 0, 0, 0, 0, 2, 0},
    /*  36:  , ,2, , ,5, , :  1, 2,10, 9, 5, 4 */ {0, 0, 1, 0, 0, 2, 0, 0},
    /* 132:  , ,2, , , , ,7: 10, 1, 2, 6,11, 7 */ {0, 0, 1, 0, 0, 0, 0, 2},
    /*  24:  , , ,3,4, , , :  8, 4, 7, 3,11, 2 */ {0, 0, 0, 1, 2, 0, 0, 0},
    /*  72:  , , ,3, , ,6, :  2, 3,11,10, 6, 5 */ {0, 0, 0, 1, 0, 0, 2, 0},
    /*  80:  , , , ,4, ,6, :  5,10, 6, 4, 7, 8 */ {0, 0, 0, 0, 1, 0, 2, 0},
    /* 160:  , , , , ,5, ,7:  4, 9, 5, 7, 6,11 */ {0, 0, 0, 0, 0, 1, 0, 2},
    /*  95: 0,1,2,3,4, ,6, :  5, 9, 4,11, 6, 7 */ {0, 0, 0, 0, 0, 1, 0, 2},
    /* 175: 0,1,2,3, ,5, ,7:  6,10, 5, 8, 7, 4 */ {0, 0, 0, 0, 1, 0, 2, 0},
    /* 183: 0,1,2, ,4,5, ,7: 11, 3, 2, 5, 6,10 */ {0, 0, 0, 1, 0, 0, 2, 0},
    /* 231: 0,1,2, , ,5,6,7:  7, 4, 8, 2,11, 3 */ {0, 0, 0, 1, 2, 0, 0, 0},
    /* 123: 0,1, ,3,4,5,6, :  2, 1,10, 7,11, 6 */ {0, 0, 1, 0, 0, 0, 0, 2},
    /* 219: 0,1, ,3,4, ,6,7: 10, 2, 1, 4, 5, 9 */ {0, 0, 1, 0, 0, 2, 0, 0},
    /* 189: 0, ,2,3,4,5, ,7:  1, 0, 9, 6,10, 5 */ {0, 1, 0, 0, 0, 0, 2, 0},
    /* 237: 0, ,2,3, ,5,6,7:  9, 1, 0, 7, 4, 8 */ {0, 1, 0, 0, 2, 0, 0, 0},
    /* 245: 0, ,2, ,4,5,6,7:  0, 9, 1,11, 3, 2 */ {0, 1, 0, 2, 0, 0, 0, 0},
    /* 126:  ,1,2,3,4,5,6, :  8, 0, 3, 6, 7,11 */ {0, 1, 1, 1, 1, 1, 1, 2},
    /* 222:  ,1,2,3,4, ,6,7:  4, 5, 9, 3, 8, 0 */ {0, 1, 1, 1, 1, 2, 1, 1},
    /* 250:  ,1, ,3,4,5,6,7:  3, 8, 0,10, 2, 1 */ {0, 1, 2, 1, 1, 1, 1, 1},
};
static const signed char triangleRegion3_1[24][4] = {
    /*   5: 0, ,2, , , , , :  0, 8, 3, 1, 2,10 */ {0, 1, 2, 1},
    /*  33: 0, , , , ,5, , :  9, 5, 4, 0, 8, 3 */ {2, 1, 0, 1},
    /* 129: 0, , , , , , ,7:  3, 0, 8,11, 7, 6 */ {0, 1, 2, 1},
    /*  10:  ,1, ,3, , , , :  1, 9, 0, 2, 3,11 */ {1, 0, 2, 0},
    /*  18:  ,1, , ,4, , , :  0, 1, 9, 8, 4, 7 */ {1, 0, 2, 0},
    /*  66:  ,1, , , , ,6, :  9, 0, 1, 5,10, 6 */ {1, 0, 2, 0},
    /*  36:  , ,2, , ,5, , :  1, 2,10, 9, 5, 4 */ {1, 0, 2, 0},
    /* 132:  , ,2, , , , ,7: 10, 1, 2, 6,11, 7 */ {1, 0, 2, 0},
    /*  24:  , , ,3,4, , , :  8, 4, 7, 3,11, 2 */ {2, 0, 1, 0},
    /*  72:  , , ,3, , ,6, :  2, 3,11,10, 6, 5 */ {1, 0, 2, 0},
    /*  80:  , , , ,4, ,6, :  5,10, 6, 4, 7, 8 */ {2, 0, 1, 0},
    /* 160:  , , , , ,5, ,7:  4, 9, 5, 7, 6,11 */ {1, 0, 2, 0},
    /*  95: 0,1,2,3,4, ,6, :  5, 9, 4,11, 6, 7 */ {0, 1, 0, 2},
    /* 175: 0,1,2,3, ,5, ,7:  6,10, 5, 8, 7, 4 */ {0, 2, 0, 1},
    /* 183: 0,1,2, ,4,5, ,7: 11, 3, 2, 5, 6,10 */ {0, 1, 0, 2},
    /* 231: 0,1,2, , ,5,6,7:  7, 4, 8, 2,11, 3 */ {0, 2, 0, 1},
    /* 123: 0,1, ,3,4,5,6, :  2, 1,10, 7,11, 6 */ {0, 1, 0, 2},
    /* 219: 0,1, ,3,4, ,6,7: 10, 2, 1, 4, 5, 9 */ {0, 1, 0, 2},
    /* 189: 0, ,2,3,4,5, ,7:  1, 0, 9, 6,10, 5 */ {0, 1, 0, 2},
    /* 237: 0, ,2,3, ,5,6,7:  9, 1, 0, 7, 4, 8 */ {0, 1, 0, 2},
    /* 245: 0, ,2, ,4,5,6,7:  0, 9, 1,11, 3, 2 */ {0, 1, 0, 2},
    /* 126:  ,1,2,3,4,5,6, :  8, 0, 3, 6, 7,11 */ {1, 0, 1, 2},
    /* 222:  ,1,2,3,4, ,6,7:  4, 5, 9, 3, 8, 0 */ {1, 2, 1, 0},
    /* 250:  ,1, ,3,4,5,6,7:  3, 8, 0,10, 2, 1 */ {1, 0, 1, 2},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion3_2[24][8] = {
    /*   5: 0, ,2, , , , , : 10, 3, 2,10, 8, 3,10, 1, 0, 8,10, 0 */ {0, 1, 0, 1, 1, 1, 1, 1},
    /*  33: 0, , , , ,5, , :  3, 4, 8, 3, 5, 4, 3, 0, 9, 5, 3, 9 */ {0, 1, 1, 1, 1, 0, 1, 1},
    /* 129: 0, , , , , , ,7:  6, 8, 7, 6, 0, 8, 6,11, 3, 0, 6, 3 */ {0, 1, 1, 1, 1, 1, 1, 0},
    /*  10:  ,1, ,3, , , , : 11, 0, 3,11, 9, 0,11, 2, 1, 9,11, 1 */ {0, 1, 0, 1, 0, 0, 0, 0},
    /*  18:  ,1, , ,4, , , :  7, 9, 4, 7, 1, 9, 7, 8, 0, 1, 7, 0 */ {0, 1, 0, 0, 1, 0, 0, 0},
    /*  66:  ,1, , , , ,6, :  6, 1,10, 6, 0, 1, 9, 0, 6, 9, 6, 5 */ {0, 1, 0, 0, 0, 0, 1, 0},
    /*  36:  , ,2, , ,5, , :  4,10, 5, 4, 2,10, 4, 9, 1, 2, 4, 1 */ {0, 0, 1, 0, 0, 1, 0, 0},
    /* 132:  , ,2, , , , ,7:  7, 2,11, 7, 1, 2, 7, 6,10, 1, 7,10 */ {0, 0, 1, 0, 0, 0, 0, 1},
    /*  24:  , , ,3,4, , , :  2, 7,11, 2, 4, 7, 2, 3, 8, 4, 2, 8 */ {0, 0, 0, 1, 1, 0, 0, 0},
    /*  72:  , , ,3, , ,6, :  5,11, 6, 5, 3,11, 5,10, 2, 3, 5, 2 */ {0, 0, 0, 1, 0, 0, 1, 0},
    /*  80:  , , , ,4, ,6, :  8, 6, 7, 8,10, 6, 8, 4, 5,10, 8, 5 */ {0, 0, 0, 0, 1, 0, 1, 0},
    /* 160:  , , , , ,5, ,7: 11, 5, 6,11, 9, 5,11, 7, 4, 9,11, 4 */ {0, 0, 0, 0, 0, 1, 0, 1},
    /*  95: 0,1,2,3,4, ,6, :  6, 5,11, 5, 9,11, 4, 7,11, 4,11, 9 */ {0, 0, 0, 0, 0, 1, 0, 1},
    /* 175: 0,1,2,3, ,5, ,7:  7, 6, 8, 6,10, 8, 5, 4, 8, 5, 8,10 */ {0, 0, 0, 0, 1, 0, 1, 0},
    /* 183: 0,1,2, ,4,5, ,7:  6,11, 5,11, 3, 5, 2,10, 5, 2, 5, 3 */ {0, 0, 0, 1, 0, 0, 1, 0},
    /* 231: 0,1,2, , ,5,6,7: 11, 7, 2, 7, 4, 2, 8, 3, 2, 8, 2, 4 */ {0, 0, 0, 1, 1, 0, 0, 0},
    /* 123: 0,1, ,3,4,5,6, : 11, 2, 7, 2, 1, 7,10, 6, 7,10, 7, 1 */ {0, 0, 1, 0, 0, 0, 0, 1},
    /* 219: 0,1, ,3,4, ,6,7:  5,10, 4,10, 2, 4, 1, 9, 4, 1, 4, 2 */ {0, 0, 1, 0, 0, 1, 0, 0},
    /* 189: 0, ,2,3,4,5, ,7: 10, 1, 6, 1, 0, 6, 6, 0, 9, 5, 6, 9 */ {0, 1, 0, 0, 0, 0, 1, 0},
    /* 237: 0, ,2,3, ,5,6,7:  4, 9, 7, 9, 1, 7, 0, 8, 7, 0, 7, 1 */ {0, 1, 0, 0, 1, 0, 0, 0},
    /* 245: 0, ,2, ,4,5,6,7:  3, 0,11, 0, 9,11, 1, 2,11, 1,11, 9 */ {0, 1, 0, 1, 0, 0, 0, 0},
    /* 126:  ,1,2,3,4,5,6, :  7, 8, 6, 8, 0, 6, 3,11, 6, 3, 6, 0 */ {0, 1, 1, 1, 1, 1, 1, 0},
    /* 222:  ,1,2,3,4, ,6,7:  8, 4, 3, 4, 5, 3, 9, 0, 3, 9, 3, 5 */ {0, 1, 1, 1, 1, 0, 1, 1},
    /* 250:  ,1, ,3,4,5,6,7:  2, 3,10, 3, 8,10, 0, 1,10, 0,10, 8 */ {0, 1, 0, 1, 1, 1, 1, 1},
};
static const signed char triangleRegion3_2[24][8] = {
    /*   5: 0, ,2, , , , , : 10, 3, 2,10, 8, 3,10, 1, 0, 8,10, 0 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  33: 0, , , , ,5, , :  3, 4, 8, 3, 5, 4, 3, 0, 9, 5, 3, 9 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 129: 0, , , , , , ,7:  6, 8, 7, 6, 0, 8, 6,11, 3, 0, 6, 3 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  10:  ,1, ,3, , , , : 11, 0, 3,11, 9, 0,11, 2, 1, 9,11, 1 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /*  18:  ,1, , ,4, , , :  7, 9, 4, 7, 1, 9, 7, 8, 0, 1, 7, 0 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /*  66:  ,1, , , , ,6, :  6, 1,10, 6, 0, 1, 9, 0, 6, 9, 6, 5 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /*  36:  , ,2, , ,5, , :  4,10, 5, 4, 2,10, 4, 9, 1, 2, 4, 1 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 132:  , ,2, , , , ,7:  7, 2,11, 7, 1, 2, 7, 6,10, 1, 7,10 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /*  24:  , , ,3,4, , , :  2, 7,11, 2, 4, 7, 2, 3, 8, 4, 2, 8 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /*  72:  , , ,3, , ,6, :  5,11, 6, 5, 3,11, 5,10, 2, 3, 5, 2 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /*  80:  , , , ,4, ,6, :  8, 6, 7, 8,10, 6, 8, 4, 5,10, 8, 5 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 160:  , , , , ,5, ,7: 11, 5, 6,11, 9, 5,11, 7, 4, 9,11, 4 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /*  95: 0,1,2,3,4, ,6, :  6, 5,11, 5, 9,11, 4, 7,11, 4,11, 9 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 175: 0,1,2,3, ,5, ,7:  7, 6, 8, 6,10, 8, 5, 4, 8, 5, 8,10 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 183: 0,1,2, ,4,5, ,7:  6,11, 5,11, 3, 5, 2,10, 5, 2, 5, 3 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 231: 0,1,2, , ,5,6,7: 11, 7, 2, 7, 4, 2, 8, 3, 2, 8, 2, 4 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 123: 0,1, ,3,4,5,6, : 11, 2, 7, 2, 1, 7,10, 6, 7,10, 7, 1 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 219: 0,1, ,3,4, ,6,7:  5,10, 4,10, 2, 4, 1, 9, 4, 1, 4, 2 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 189: 0, ,2,3,4,5, ,7: 10, 1, 6, 1, 0, 6, 6, 0, 9, 5, 6, 9 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 237: 0, ,2,3, ,5,6,7:  4, 9, 7, 9, 1, 7, 0, 8, 7, 0, 7, 1 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 245: 0, ,2, ,4,5,6,7:  3, 0,11, 0, 9,11, 1, 2,11, 1,11, 9 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 126:  ,1,2,3,4,5,6, :  7, 8, 6, 8, 0, 6, 3,11, 6, 3, 6, 0 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 222:  ,1,2,3,4, ,6,7:  8, 4, 3, 4, 5, 3, 9, 0, 3, 9, 3, 5 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 250:  ,1, ,3,4,5,6,7:  2, 3,10, 3, 8,10, 0, 1,10, 0,10, 8 */ {1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion4_1[8][8] = {
    /*  65: 0, , , , , ,6, :  0, 8, 3, 5,10, 6 */ {0, 1, 1, 1, 1, 1, 2, 1},
    /* 130:  ,1, , , , , ,7:  0, 1, 9,11, 7, 6 */ {0, 1, 0, 0, 0, 0, 0, 2},
    /*  20:  , ,2, ,4, , , :  1, 2,10, 8, 4, 7 */ {0, 0, 1, 0, 2, 0, 0, 0},
    /*  40:  , , ,3, ,5, , :  9, 5, 4, 2, 3,11 */ {0, 0, 0, 1, 0, 2, 0, 0},
    /* 215: 0,1,2, ,4, ,6,7:  4, 5, 9,11, 3, 2 */ {0, 0, 0, 1, 0, 2, 0, 0},
    /* 235: 0,1, ,3, ,5,6,7: 10, 2, 1, 7, 4, 8 */ {0, 0, 1, 0, 2, 0, 0, 0},
    /* 125: 0, ,2,3,4,5,6, :  9, 1, 0, 6, 7,11 */ {0, 1, 0, 0, 0, 0, 0, 2},
    /* 190:  ,1,2,3,4,5, ,7:  3, 8, 0, 6,10, 5 */ {0, 1, 1, 1, 1, 1, 2, 1},
};
static const signed char triangleRegion4_1[8][4] = {
    /*  65: 0, , , , , ,6, :  0, 8, 3, 5,10, 6 */ {0, 1, 2, 1},
    /* 130:  ,1, , , , , ,7:  0, 1, 9,11, 7, 6 */ {1, 0, 2, 0},
    /*  20:  , ,2, ,4, , , :  1, 2,10, 8, 4, 7 */ {1, 0, 2, 0},
    /*  40:  , , ,3, ,5, , :  9, 5, 4, 2, 3,11 */ {2, 0, 1, 0},
    /* 215: 0,1,2, ,4, ,6,7:  4, 5, 9,11, 3, 2 */ {0, 2, 0, 1},
    /* 235: 0,1, ,3, ,5,6,7: 10, 2, 1, 7, 4, 8 */ {0, 1, 0, 2},
    /* 125: 0, ,2,3,4,5,6, :  9, 1, 0, 6, 7,11 */ {0, 1, 0, 2},
    /* 190:  ,1,2,3,4,5, ,7:  3, 8, 0, 6,10, 5 */ {1, 0, 1, 2},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion4_2[8][8] = {
    /*  65: 0, , , , , ,6, :  8, 5, 0, 5, 8, 6, 3, 6, 8, 6, 3,10, 0,10, 3,10, 0, 5 */ {0, 1, 1, 1, 1, 1, 0, 1},
    /* 130:  ,1, , , , , ,7:  9, 6, 1, 6, 9, 7, 0, 7, 9, 7, 0,11, 1,11, 0,11, 1, 6 */ {0, 1, 0, 0, 0, 0, 0, 1},
    /*  20:  , ,2, ,4, , , : 10, 7, 2, 7,10, 4, 1, 4,10, 4, 1, 8, 2, 8, 1, 8, 2, 7 */ {0, 0, 1, 0, 1, 0, 0, 0},
    /*  40:  , , ,3, ,5, , : 11, 4, 3, 4,11, 5, 2, 5,11, 5, 2, 9, 3, 9, 2, 9, 3, 4 */ {0, 0, 0, 1, 0, 1, 0, 0},
    /* 215: 0,1,2, ,4, ,6,7:  3, 4,11, 5,11, 4,11, 5, 2, 9, 2, 5, 2, 9, 3, 4, 3, 9 */ {0, 0, 0, 1, 0, 1, 0, 0},
    /* 235: 0,1, ,3, ,5,6,7:  2, 7,10, 4,10, 7,10, 4, 1, 8, 1, 4, 1, 8, 2, 7, 2, 8 */ {0, 0, 1, 0, 1, 0, 0, 0},
    /* 125: 0, ,2,3,4,5,6, :  1, 6, 9, 7, 9, 6, 9, 7, 0,11, 0, 7, 0,11, 1, 6, 1,11 */ {0, 1, 0, 0, 0, 0, 0, 1},
    /* 190:  ,1,2,3,4,5, ,7:  0, 5, 8, 6, 8, 5, 8, 6, 3,10, 3, 6, 3,10, 0, 5, 0,10 */ {0, 1, 1, 1, 1, 1, 0, 1},
};
static const signed char triangleRegion4_2[8][12] = {
    /*  65: 0, , , , , ,6, :  8, 5, 0, 5, 8, 6, 3, 6, 8, 6, 3,10, 0,10, 3,10, 0, 5 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 130:  ,1, , , , , ,7:  9, 6, 1, 6, 9, 7, 0, 7, 9, 7, 0,11, 1,11, 0,11, 1, 6 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  20:  , ,2, ,4, , , : 10, 7, 2, 7,10, 4, 1, 4,10, 4, 1, 8, 2, 8, 1, 8, 2, 7 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  40:  , , ,3, ,5, , : 11, 4, 3, 4,11, 5, 2, 5,11, 5, 2, 9, 3, 9, 2, 9, 3, 4 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 215: 0,1,2, ,4, ,6,7:  3, 4,11, 5,11, 4,11, 5, 2, 9, 2, 5, 2, 9, 3, 4, 3, 9 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 235: 0,1, ,3, ,5,6,7:  2, 7,10, 4,10, 7,10, 4, 1, 8, 1, 4, 1, 8, 2, 7, 2, 8 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 125: 0, ,2,3,4,5,6, :  1, 6, 9, 7, 9, 6, 9, 7, 0,11, 0, 7, 0,11, 1, 6, 1,11 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 190:  ,1,2,3,4,5, ,7:  0, 5, 8, 6, 8, 5, 8, 6, 3,10, 3, 6, 3,10, 0, 5, 0,10 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion5[48][8] = {
    /*   7: 0,1,2, , , , , :  2, 8, 3, 2,10, 8,10, 9, 8 */ {0, 0, 0, 1, 1, 1, 1, 1},
    /*  11: 0,1, ,3, , , , :  1,11, 2, 1, 9,11, 9, 8,11 */ {0, 0, 1, 0, 1, 1, 1, 1},
    /*  19: 0,1, , ,4, , , :  4, 1, 9, 4, 7, 1, 7, 3, 1 */ {0, 0, 1, 1, 0, 1, 1, 1},
    /*  35: 0,1, , , ,5, , :  8, 5, 4, 8, 3, 5, 3, 1, 5 */ {0, 0, 1, 1, 1, 0, 1, 1},
    /*  13: 0, ,2,3, , , , :  0,10, 1, 0, 8,10, 8,11,10 */ {0, 1, 0, 0, 1, 1, 1, 1},
    /*  25: 0, , ,3,4, , , : 11, 4, 7,11, 2, 4, 2, 0, 4 */ {0, 1, 1, 0, 0, 1, 1, 1},
    /* 137: 0, , ,3, , , ,7:  7, 0, 8, 7, 6, 0, 6, 2, 0 */ {0, 1, 1, 0, 1, 1, 1, 0},
    /*  49: 0, , , ,4,5, , :  9, 3, 0, 9, 5, 3, 5, 7, 3 */ {0, 1, 1, 1, 0, 0, 1, 1},
    /* 145: 0, , , ,4, , ,7:  3, 6,11, 3, 0, 6, 0, 4, 6 */ {0, 1, 1, 1, 0, 1, 1, 0},
    /*  14:  ,1,2,3, , , , :  3, 9, 0, 3,11, 9,11,10, 9 */ {0, 1, 1, 1, 0, 0, 0, 0},
    /*  38:  ,1,2, , ,5, , :  5, 2,10, 5, 4, 2, 4, 0, 2 */ {0, 1, 1, 0, 0, 1, 0, 0},
    /*  70:  ,1,2, , , ,6, :  9, 6, 5, 9, 0, 6, 0, 2, 6 */ {0, 1, 1, 0, 0, 0, 1, 0},
    /*  50:  ,1, , ,4,5, , :  0, 7, 8, 0, 1, 7, 1, 5, 7 */ {0, 1, 0, 0, 1, 1, 0, 0},
    /*  98:  ,1, , , ,5,6, : 10, 0, 1,10, 6, 0, 6, 4, 0 */ {0, 1, 0, 0, 0, 1, 1, 0},
    /*  76:  , ,2,3, , ,6, :  6, 3,11, 6, 5, 3, 5, 1, 3 */ {0, 0, 1, 1, 0, 0, 1, 0},
    /* 140:  , ,2,3, , , ,7: 10, 7, 6,10, 1, 7, 1, 3, 7 */ {0, 0, 1, 1, 0, 0, 0, 1},
    /* 100:  , ,2, , ,5,6, :  1, 4, 9, 1, 2, 4, 2, 6, 4 */ {0, 0, 1, 0, 0, 1, 1, 0},
    /* 196:  , ,2, , , ,6,7: 11, 1, 2,11, 7, 1, 7, 5, 1 */ {0, 0, 1, 0, 0, 0, 1, 1},
    /* 152:  , , ,3,4, , ,7:  8, 2, 3, 8, 4, 2, 4, 6, 2 */ {0, 0, 0, 1, 1, 0, 0, 1},
    /* 200:  , , ,3, , ,6,7:  2, 5,10, 2, 3, 5, 3, 7, 5 */ {0, 0, 0, 1, 0, 0, 1, 1},
    /* 112:  , , , ,4,5,6, :  7,10, 6, 7, 8,10, 8, 9,10 */ {0, 0, 0, 0, 1, 1, 1, 0},
    /* 176:  , , , ,4,5, ,7:  6, 9, 5, 6,11, 9,11, 8, 9 */ {0, 0, 0, 0, 1, 1, 0, 1},
    /* 208:  , , , ,4, ,6,7:  5, 8, 4, 5,10, 8,10,11, 8 */ {0, 0, 0, 0, 1, 0, 1, 1},
    /* 224:  , , , , ,5,6,7:  4,11, 7, 4, 9,11, 9,10,11 */ {0, 0, 0, 0, 0, 1, 1, 1},
    /*  31: 0,1,2,3,4, , , :  4, 7,11, 4,11, 9, 9,11,10 */ {0, 0, 0, 0, 0, 1, 1, 1},
    /*  47: 0,1,2,3, ,5, , :  5, 4, 8, 5, 8,10,10, 8,11 */ {0, 0, 0, 0, 1, 0, 1, 1},
    /*  79: 0,1,2,3, , ,6, :  6, 5, 9, 6, 9,11,11, 9, 8 */ {0, 0, 0, 0, 1, 1, 0, 1},
    /* 143: 0,1,2,3, , , ,7:  7, 6,10, 7,10, 8, 8,10, 9 */ {0, 0, 0, 0, 1, 1, 1, 0},
    /*  55: 0,1,2, ,4,5, , :  2,10, 5, 2, 5, 3, 3, 5, 7 */ {0, 0, 0, 1, 0, 0, 1, 1},
    /* 103: 0,1,2, , ,5,6, :  8, 3, 2, 8, 2, 4, 4, 2, 6 */ {0, 0, 0, 1, 1, 0, 0, 1},
    /*  59: 0,1, ,3,4,5, , : 11, 2, 1,11, 1, 7, 7, 1, 5 */ {0, 0, 1, 0, 0, 0, 1, 1},
    /* 155: 0,1, ,3,4, , ,7:  1, 9, 4, 1, 4, 2, 2, 4, 6 */ {0, 0, 1, 0, 0, 1, 1, 0},
    /* 115: 0,1, , ,4,5,6, : 10, 6, 7,10, 7, 1, 1, 7, 3 */ {0, 0, 1, 1, 0, 0, 0, 1},
    /* 179: 0,1, , ,4,5, ,7:  6,11, 3, 6, 3, 5, 5, 3, 1 */ {0, 0, 1, 1, 0, 0, 1, 0},
    /* 157: 0, ,2,3,4, , ,7: 10, 1, 0,10, 0, 6, 6, 0, 4 */ {0, 1, 0, 0, 0, 1, 1, 0},
    /* 205: 0, ,2,3, , ,6,7:  0, 8, 7, 0, 7, 1, 1, 7, 5 */ {0, 1, 0, 0, 1, 1, 0, 0},
    /* 185: 0, , ,3,4,5, ,7:  9, 5, 6, 9, 6, 0, 0, 6, 2 */ {0, 1, 1, 0, 0, 0, 1, 0},
    /* 217: 0, , ,3,4, ,6,7:  5,10, 2, 5, 2, 4, 4, 2, 0 */ {0, 1, 1, 0, 0, 1, 0, 0},
    /* 241: 0, , , ,4,5,6,7:  3, 0, 9, 3, 9,11,11, 9,10 */ {0, 1, 1, 1, 0, 0, 0, 0},
    /* 110:  ,1,2,3, ,5,6, :  3,11, 6, 3, 6, 0, 0, 6, 4 */ {0, 1, 1, 1, 0, 1, 1, 0},
    /* 206:  ,1,2,3, , ,6,7:  9, 0, 3, 9, 3, 5, 5, 3, 7 */ {0, 1, 1, 1, 0, 0, 1, 1},
    /* 118:  ,1,2, ,4,5,6, :  7, 8, 0, 7, 0, 6, 6, 0, 2 */ {0, 1, 1, 0, 1, 1, 1, 0},
    /* 230:  ,1,2, , ,5,6,7: 11, 7, 4,11, 4, 2, 2, 4, 0 */ {0, 1, 1, 0, 0, 1, 1, 1},
    /* 242:  ,1, , ,4,5,6,7:  0, 1,10, 0,10, 8, 8,10,11 */ {0, 1, 0, 0, 1, 1, 1, 1},
    /* 220:  , ,2,3,4, ,6,7:  8, 4, 5, 8, 5, 3, 3, 5, 1 */ {0, 0, 1, 1, 1, 0, 1, 1},
    /* 236:  , ,2,3, ,5,6,7:  4, 9, 1, 4, 1, 7, 7, 1, 3 */ {0, 0, 1, 1, 0, 1, 1, 1},
    /* 244:  , ,2, ,4,5,6,7:  1, 2,11, 1,11, 9, 9,11, 8 */ {0, 0, 1, 0, 1, 1, 1, 1},
    /* 248:  , , ,3,4,5,6,7:  2, 3, 8, 2, 8,10,10, 8, 9 */ {0, 0, 0, 1, 1, 1, 1, 1},
};
static const signed char triangleRegion5[48][6] = {
    /*   7: 0,1,2, , , , , :  2, 8, 3, 2,10, 8,10, 9, 8 */ {0, 1, 0, 1, 0, 1},
    /*  11: 0,1, ,3, , , , :  1,11, 2, 1, 9,11, 9, 8,11 */ {0, 1, 0, 1, 0, 1},
    /*  19: 0,1, , ,4, , , :  4, 1, 9, 4, 7, 1, 7, 3, 1 */ {0, 1, 0, 1, 0, 1},
    /*  35: 0,1, , , ,5, , :  8, 5, 4, 8, 3, 5, 3, 1, 5 */ {0, 1, 0, 1, 0, 1},
    /*  13: 0, ,2,3, , , , :  0,10, 1, 0, 8,10, 8,11,10 */ {0, 1, 0, 1, 0, 1},
    /*  25: 0, , ,3,4, , , : 11, 4, 7,11, 2, 4, 2, 0, 4 */ {0, 1, 0, 1, 0, 1},
    /* 137: 0, , ,3, , , ,7:  7, 0, 8, 7, 6, 0, 6, 2, 0 */ {0, 1, 0, 1, 0, 1},
    /*  49: 0, , , ,4,5, , :  9, 3, 0, 9, 5, 3, 5, 7, 3 */ {0, 1, 0, 1, 0, 1},
    /* 145: 0, , , ,4, , ,7:  3, 6,11, 3, 0, 6, 0, 4, 6 */ {0, 1, 0, 1, 0, 1},
    /*  14:  ,1,2,3, , , , :  3, 9, 0, 3,11, 9,11,10, 9 */ {1, 0, 1, 0, 1, 0},
    /*  38:  ,1,2, , ,5, , :  5, 2,10, 5, 4, 2, 4, 0, 2 */ {1, 0, 1, 0, 1, 0},
    /*  70:  ,1,2, , , ,6, :  9, 6, 5, 9, 0, 6, 0, 2, 6 */ {1, 0, 1, 0, 1, 0},
    /*  50:  ,1, , ,4,5, , :  0, 7, 8, 0, 1, 7, 1, 5, 7 */ {1, 0, 1, 0, 1, 0},
    /*  98:  ,1, , , ,5,6, : 10, 0, 1,10, 6, 0, 6, 4, 0 */ {1, 0, 1, 0, 1, 0},
    /*  76:  , ,2,3, , ,6, :  6, 3,11, 6, 5, 3, 5, 1, 3 */ {1, 0, 1, 0, 1, 0},
    /* 140:  , ,2,3, , , ,7: 10, 7, 6,10, 1, 7, 1, 3, 7 */ {1, 0, 1, 0, 1, 0},
    /* 100:  , ,2, , ,5,6, :  1, 4, 9, 1, 2, 4, 2, 6, 4 */ {1, 0, 1, 0, 1, 0},
    /* 196:  , ,2, , , ,6,7: 11, 1, 2,11, 7, 1, 7, 5, 1 */ {1, 0, 1, 0, 1, 0},
    /* 152:  , , ,3,4, , ,7:  8, 2, 3, 8, 4, 2, 4, 6, 2 */ {1, 0, 1, 0, 1, 0},
    /* 200:  , , ,3, , ,6,7:  2, 5,10, 2, 3, 5, 3, 7, 5 */ {1, 0, 1, 0, 1, 0},
    /* 112:  , , , ,4,5,6, :  7,10, 6, 7, 8,10, 8, 9,10 */ {1, 0, 1, 0, 1, 0},
    /* 176:  , , , ,4,5, ,7:  6, 9, 5, 6,11, 9,11, 8, 9 */ {1, 0, 1, 0, 1, 0},
    /* 208:  , , , ,4, ,6,7:  5, 8, 4, 5,10, 8,10,11, 8 */ {1, 0, 1, 0, 1, 0},
    /* 224:  , , , , ,5,6,7:  4,11, 7, 4, 9,11, 9,10,11 */ {1, 0, 1, 0, 1, 0},
    /*  31: 0,1,2,3,4, , , :  4, 7,11, 4,11, 9, 9,11,10 */ {0, 1, 0, 1, 0, 1},
    /*  47: 0,1,2,3, ,5, , :  5, 4, 8, 5, 8,10,10, 8,11 */ {0, 1, 0, 1, 0, 1},
    /*  79: 0,1,2,3, , ,6, :  6, 5, 9, 6, 9,11,11, 9, 8 */ {0, 1, 0, 1, 0, 1},
    /* 143: 0,1,2,3, , , ,7:  7, 6,10, 7,10, 8, 8,10, 9 */ {0, 1, 0, 1, 0, 1},
    /*  55: 0,1,2, ,4,5, , :  2,10, 5, 2, 5, 3, 3, 5, 7 */ {0, 1, 0, 1, 0, 1},
    /* 103: 0,1,2, , ,5,6, :  8, 3, 2, 8, 2, 4, 4, 2, 6 */ {0, 1, 0, 1, 0, 1},
    /*  59: 0,1, ,3,4,5, , : 11, 2, 1,11, 1, 7, 7, 1, 5 */ {0, 1, 0, 1, 0, 1},
    /* 155: 0,1, ,3,4, , ,7:  1, 9, 4, 1, 4, 2, 2, 4, 6 */ {0, 1, 0, 1, 0, 1},
    /* 115: 0,1, , ,4,5,6, : 10, 6, 7,10, 7, 1, 1, 7, 3 */ {0, 1, 0, 1, 0, 1},
    /* 179: 0,1, , ,4,5, ,7:  6,11, 3, 6, 3, 5, 5, 3, 1 */ {0, 1, 0, 1, 0, 1},
    /* 157: 0, ,2,3,4, , ,7: 10, 1, 0,10, 0, 6, 6, 0, 4 */ {0, 1, 0, 1, 0, 1},
    /* 205: 0, ,2,3, , ,6,7:  0, 8, 7, 0, 7, 1, 1, 7, 5 */ {0, 1, 0, 1, 0, 1},
    /* 185: 0, , ,3,4,5, ,7:  9, 5, 6, 9, 6, 0, 0, 6, 2 */ {0, 1, 0, 1, 0, 1},
    /* 217: 0, , ,3,4, ,6,7:  5,10, 2, 5, 2, 4, 4, 2, 0 */ {0, 1, 0, 1, 0, 1},
    /* 241: 0, , , ,4,5,6,7:  3, 0, 9, 3, 9,11,11, 9,10 */ {0, 1, 0, 1, 0, 1},
    /* 110:  ,1,2,3, ,5,6, :  3,11, 6, 3, 6, 0, 0, 6, 4 */ {1, 0, 1, 0, 1, 0},
    /* 206:  ,1,2,3, , ,6,7:  9, 0, 3, 9, 3, 5, 5, 3, 7 */ {1, 0, 1, 0, 1, 0},
    /* 118:  ,1,2, ,4,5,6, :  7, 8, 0, 7, 0, 6, 6, 0, 2 */ {1, 0, 1, 0, 1, 0},
    /* 230:  ,1,2, , ,5,6,7: 11, 7, 4,11, 4, 2, 2, 4, 0 */ {1, 0, 1, 0, 1, 0},
    /* 242:  ,1, , ,4,5,6,7:  0, 1,10, 0,10, 8, 8,10,11 */ {1, 0, 1, 0, 1, 0},
    /* 220:  , ,2,3,4, ,6,7:  8, 4, 5, 8, 5, 3, 3, 5, 1 */ {1, 0, 1, 0, 1, 0},
    /* 236:  , ,2,3, ,5,6,7:  4, 9, 1, 4, 1, 7, 7, 1, 3 */ {1, 0, 1, 0, 1, 0},
    /* 244:  , ,2, ,4,5,6,7:  1, 2,11, 1,11, 9, 9,11, 8 */ {1, 0, 1, 0, 1, 0},
    /* 248:  , , ,3,4,5,6,7:  2, 3, 8, 2, 8,10,10, 8, 9 */ {1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion6_1_1[48][8] = {
    /*  67: 0,1, , , , ,6, :  6, 5,10, 3, 1, 8, 9, 8, 1 */ {0, 0, 1, 1, 1, 1, 2, 1},
    /* 131: 0,1, , , , , ,7: 11, 7, 6, 9, 3, 1, 3, 9, 8 */ {0, 0, 1, 1, 1, 1, 1, 2},
    /*  21: 0, ,2, ,4, , , :  1, 2,10, 7, 0, 4, 0, 7, 3 */ {0, 1, 2, 1, 0, 1, 1, 1},
    /*  69: 0, ,2, , , ,6, :  3, 0, 8, 5, 2, 6, 2, 5, 1 */ {0, 1, 2, 1, 1, 1, 2, 1},
    /*  41: 0, , ,3, ,5, , :  5, 4, 9, 2, 0,11, 8,11, 0 */ {0, 1, 1, 0, 1, 2, 1, 1},
    /*  73: 0, , ,3, , ,6, : 10, 6, 5, 8, 2, 0, 2, 8,11 */ {0, 1, 1, 0, 1, 1, 2, 1},
    /*  81: 0, , , ,4, ,6, : 10, 6, 5, 0, 4, 3, 7, 3, 4 */ {0, 1, 1, 1, 0, 1, 2, 1},
    /*  97: 0, , , , ,5,6, :  3, 0, 8, 6, 4,10, 9,10, 4 */ {0, 1, 1, 1, 1, 2, 2, 1},
    /* 193: 0, , , , , ,6,7:  8, 3, 0,10, 7, 5, 7,10,11 */ {0, 1, 1, 1, 1, 1, 2, 2},
    /*  22:  ,1,2, ,4, , , :  8, 4, 7,10, 0, 2, 0,10, 9 */ {0, 1, 1, 0, 2, 0, 0, 0},
    /* 134:  ,1,2, , , , ,7:  7, 6,11, 0, 2, 9,10, 9, 2 */ {0, 1, 1, 0, 0, 0, 0, 2},
    /*  42:  ,1, ,3, ,5, , :  2, 3,11, 4, 1, 5, 1, 4, 0 */ {0, 1, 0, 2, 0, 1, 0, 0},
    /* 138:  ,1, ,3, , , ,7:  0, 1, 9, 6, 3, 7, 3, 6, 2 */ {0, 1, 0, 2, 0, 0, 0, 2},
    /* 146:  ,1, , ,4, , ,7:  9, 0, 1,11, 4, 6, 4,11, 8 */ {0, 1, 0, 0, 2, 0, 0, 2},
    /* 162:  ,1, , , ,5, ,7: 11, 7, 6, 1, 5, 0, 4, 0, 5 */ {0, 1, 0, 0, 0, 1, 0, 2},
    /* 194:  ,1, , , , ,6,7:  0, 1, 9, 7, 5,11,10,11, 5 */ {0, 1, 0, 0, 0, 0, 2, 2},
    /*  28:  , ,2,3,4, , , :  4, 7, 8, 1, 3,10,11,10, 3 */ {0, 0, 1, 1, 2, 0, 0, 0},
    /*  44:  , ,2,3, ,5, , :  9, 5, 4,11, 1, 3, 1,11,10 */ {0, 0, 1, 1, 0, 2, 0, 0},
    /*  52:  , ,2, ,4,5, , : 10, 1, 2, 8, 5, 7, 5, 8, 9 */ {0, 0, 1, 0, 2, 2, 0, 0},
    /*  84:  , ,2, ,4, ,6, :  8, 4, 7, 2, 6, 1, 5, 1, 6 */ {0, 0, 1, 0, 2, 0, 1, 0},
    /* 148:  , ,2, ,4, , ,7:  1, 2,10, 4, 6, 8,11, 8, 6 */ {0, 0, 1, 0, 2, 0, 0, 2},
    /*  56:  , , ,3,4,5, , :  2, 3,11, 5, 7, 9, 8, 9, 7 */ {0, 0, 0, 1, 2, 2, 0, 0},
    /* 104:  , , ,3, ,5,6, : 11, 2, 3, 9, 6, 4, 6, 9,10 */ {0, 0, 0, 1, 0, 2, 2, 0},
    /* 168:  , , ,3, ,5, ,7:  9, 5, 4, 3, 7, 2, 6, 2, 7 */ {0, 0, 0, 1, 0, 2, 0, 1},
    /*  87: 0,1,2, ,4, ,6, :  4, 5, 9, 2, 7, 3, 7, 2, 6 */ {0, 0, 0, 1, 0, 2, 0, 1},
    /* 151: 0,1,2, ,4, , ,7:  3, 2,11, 4, 6, 9,10, 9, 6 */ {0, 0, 0, 1, 0, 2, 2, 0},
    /* 199: 0,1,2, , , ,6,7: 11, 3, 2, 9, 7, 5, 7, 9, 8 */ {0, 0, 0, 1, 2, 2, 0, 0},
    /* 107: 0,1, ,3, ,5,6, : 10, 2, 1, 8, 6, 4, 6, 8,11 */ {0, 0, 1, 0, 2, 0, 0, 2},
    /* 171: 0,1, ,3, ,5, ,7:  7, 4, 8, 1, 6, 2, 6, 1, 5 */ {0, 0, 1, 0, 2, 0, 1, 0},
    /* 203: 0,1, ,3, , ,6,7:  2, 1,10, 7, 5, 8, 9, 8, 5 */ {0, 0, 1, 0, 2, 2, 0, 0},
    /* 211: 0,1, , ,4, ,6,7:  4, 5, 9, 3, 1,11,10,11, 1 */ {0, 0, 1, 1, 0, 2, 0, 0},
    /* 227: 0,1, , , ,5,6,7:  8, 7, 4,10, 3, 1, 3,10,11 */ {0, 0, 1, 1, 2, 0, 0, 0},
    /*  61: 0, ,2,3,4,5, , :  9, 1, 0,11, 5, 7, 5,11,10 */ {0, 1, 0, 0, 0, 0, 2, 2},
    /*  93: 0, ,2,3,4, ,6, :  6, 7,11, 0, 5, 1, 5, 0, 4 */ {0, 1, 0, 0, 0, 1, 0, 2},
    /* 109: 0, ,2,3, ,5,6, :  1, 0, 9, 6, 4,11, 8,11, 4 */ {0, 1, 0, 0, 2, 0, 0, 2},
    /* 117: 0, ,2, ,4,5,6, :  9, 1, 0, 7, 3, 6, 2, 6, 3 */ {0, 1, 0, 2, 0, 0, 0, 2},
    /* 213: 0, ,2, ,4, ,6,7: 11, 3, 2, 5, 1, 4, 0, 4, 1 */ {0, 1, 0, 2, 0, 1, 0, 0},
    /* 121: 0, , ,3,4,5,6, : 11, 6, 7, 9, 2, 0, 2, 9,10 */ {0, 1, 1, 0, 0, 0, 0, 2},
    /* 233: 0, , ,3, ,5,6,7:  7, 4, 8, 2, 0,10, 9,10, 0 */ {0, 1, 1, 0, 2, 0, 0, 0},
    /*  62:  ,1,2,3,4,5, , :  0, 3, 8, 5, 7,10,11,10, 7 */ {0, 1, 1, 1, 1, 1, 2, 2},
    /* 158:  ,1,2,3,4, , ,7:  8, 0, 3,10, 4, 6, 4,10, 9 */ {0, 1, 1, 1, 1, 2, 2, 1},
    /* 174:  ,1,2,3, ,5, ,7:  5, 6,10, 3, 4, 0, 4, 3, 7 */ {0, 1, 1, 1, 0, 1, 2, 1},
    /* 182:  ,1,2, ,4,5, ,7:  5, 6,10, 0, 2, 8,11, 8, 2 */ {0, 1, 1, 0, 1, 1, 2, 1},
    /* 214:  ,1,2, ,4, ,6,7:  9, 4, 5,11, 0, 2, 0,11, 8 */ {0, 1, 1, 0, 1, 2, 1, 1},
    /* 186:  ,1, ,3,4,5, ,7:  8, 0, 3, 6, 2, 5, 1, 5, 2 */ {0, 1, 2, 1, 1, 1, 2, 1},
    /* 234:  ,1, ,3, ,5,6,7: 10, 2, 1, 4, 0, 7, 3, 7, 0 */ {0, 1, 2, 1, 0, 1, 1, 1},
    /* 124:  , ,2,3,4,5,6, :  6, 7,11, 1, 3, 9, 8, 9, 3 */ {0, 0, 1, 1, 1, 1, 1, 2},
    /* 188:  , ,2,3,4,5, ,7: 10, 5, 6, 8, 1, 3, 1, 8, 9 */ {0, 0, 1, 1, 1, 1, 2, 1},
};
static const signed char triangleRegion6_1_1[48][6] = {
    /*  67: 0,1, , , , ,6, :  6, 5,10, 3, 1, 8, 9, 8, 1 */ {2, 1, 0, 1, 0, 1},
    /* 131: 0,1, , , , , ,7: 11, 7, 6, 9, 3, 1, 3, 9, 8 */ {2, 1, 0, 1, 0, 1},
    /*  21: 0, ,2, ,4, , , :  1, 2,10, 7, 0, 4, 0, 7, 3 */ {2, 1, 0, 1, 0, 1},
    /*  69: 0, ,2, , , ,6, :  3, 0, 8, 5, 2, 6, 2, 5, 1 */ {0, 1, 2, 1, 2, 1},
    /*  41: 0, , ,3, ,5, , :  5, 4, 9, 2, 0,11, 8,11, 0 */ {2, 1, 0, 1, 0, 1},
    /*  73: 0, , ,3, , ,6, : 10, 6, 5, 8, 2, 0, 2, 8,11 */ {2, 1, 0, 1, 0, 1},
    /*  81: 0, , , ,4, ,6, : 10, 6, 5, 0, 4, 3, 7, 3, 4 */ {2, 1, 0, 1, 0, 1},
    /*  97: 0, , , , ,5,6, :  3, 0, 8, 6, 4,10, 9,10, 4 */ {0, 1, 2, 1, 2, 1},
    /* 193: 0, , , , , ,6,7:  8, 3, 0,10, 7, 5, 7,10,11 */ {0, 1, 2, 1, 2, 1},
    /*  22:  ,1,2, ,4, , , :  8, 4, 7,10, 0, 2, 0,10, 9 */ {2, 0, 1, 0, 1, 0},
    /* 134:  ,1,2, , , , ,7:  7, 6,11, 0, 2, 9,10, 9, 2 */ {2, 0, 1, 0, 1, 0},
    /*  42:  ,1, ,3, ,5, , :  2, 3,11, 4, 1, 5, 1, 4, 0 */ {2, 0, 1, 0, 1, 0},
    /* 138:  ,1, ,3, , , ,7:  0, 1, 9, 6, 3, 7, 3, 6, 2 */ {1, 0, 2, 0, 2, 0},
    /* 146:  ,1, , ,4, , ,7:  9, 0, 1,11, 4, 6, 4,11, 8 */ {1, 0, 2, 0, 2, 0},
    /* 162:  ,1, , , ,5, ,7: 11, 7, 6, 1, 5, 0, 4, 0, 5 */ {2, 0, 1, 0, 1, 0},
    /* 194:  ,1, , , , ,6,7:  0, 1, 9, 7, 5,11,10,11, 5 */ {1, 0, 2, 0, 2, 0},
    /*  28:  , ,2,3,4, , , :  4, 7, 8, 1, 3,10,11,10, 3 */ {2, 0, 1, 0, 1, 0},
    /*  44:  , ,2,3, ,5, , :  9, 5, 4,11, 1, 3, 1,11,10 */ {2, 0, 1, 0, 1, 0},
    /*  52:  , ,2, ,4,5, , : 10, 1, 2, 8, 5, 7, 5, 8, 9 */ {1, 0, 2, 0, 2, 0},
    /*  84:  , ,2, ,4, ,6, :  8, 4, 7, 2, 6, 1, 5, 1, 6 */ {2, 0, 1, 0, 1, 0},
    /* 148:  , ,2, ,4, , ,7:  1, 2,10, 4, 6, 8,11, 8, 6 */ {1, 0, 2, 0, 2, 0},
    /*  56:  , , ,3,4,5, , :  2, 3,11, 5, 7, 9, 8, 9, 7 */ {1, 0, 2, 0, 2, 0},
    /* 104:  , , ,3, ,5,6, : 11, 2, 3, 9, 6, 4, 6, 9,10 */ {1, 0, 2, 0, 2, 0},
    /* 168:  , , ,3, ,5, ,7:  9, 5, 4, 3, 7, 2, 6, 2, 7 */ {2, 0, 1, 0, 1, 0},
    /*  87: 0,1,2, ,4, ,6, :  4, 5, 9, 2, 7, 3, 7, 2, 6 */ {0, 2, 0, 1, 0, 1},
    /* 151: 0,1,2, ,4, , ,7:  3, 2,11, 4, 6, 9,10, 9, 6 */ {0, 1, 0, 2, 0, 2},
    /* 199: 0,1,2, , , ,6,7: 11, 3, 2, 9, 7, 5, 7, 9, 8 */ {0, 1, 0, 2, 0, 2},
    /* 107: 0,1, ,3, ,5,6, : 10, 2, 1, 8, 6, 4, 6, 8,11 */ {0, 1, 0, 2, 0, 2},
    /* 171: 0,1, ,3, ,5, ,7:  7, 4, 8, 1, 6, 2, 6, 1, 5 */ {0, 2, 0, 1, 0, 1},
    /* 203: 0,1, ,3, , ,6,7:  2, 1,10, 7, 5, 8, 9, 8, 5 */ {0, 1, 0, 2, 0, 2},
    /* 211: 0,1, , ,4, ,6,7:  4, 5, 9, 3, 1,11,10,11, 1 */ {0, 2, 0, 1, 0, 1},
    /* 227: 0,1, , , ,5,6,7:  8, 7, 4,10, 3, 1, 3,10,11 */ {0, 2, 0, 1, 0, 1},
    /*  61: 0, ,2,3,4,5, , :  9, 1, 0,11, 5, 7, 5,11,10 */ {0, 1, 0, 2, 0, 2},
    /*  93: 0, ,2,3,4, ,6, :  6, 7,11, 0, 5, 1, 5, 0, 4 */ {0, 2, 0, 1, 0, 1},
    /* 109: 0, ,2,3, ,5,6, :  1, 0, 9, 6, 4,11, 8,11, 4 */ {0, 1, 0, 2, 0, 2},
    /* 117: 0, ,2, ,4,5,6, :  9, 1, 0, 7, 3, 6, 2, 6, 3 */ {0, 1, 0, 2, 0, 2},
    /* 213: 0, ,2, ,4, ,6,7: 11, 3, 2, 5, 1, 4, 0, 4, 1 */ {0, 2, 0, 1, 0, 1},
    /* 121: 0, , ,3,4,5,6, : 11, 6, 7, 9, 2, 0, 2, 9,10 */ {0, 2, 0, 1, 0, 1},
    /* 233: 0, , ,3, ,5,6,7:  7, 4, 8, 2, 0,10, 9,10, 0 */ {0, 2, 0, 1, 0, 1},
    /*  62:  ,1,2,3,4,5, , :  0, 3, 8, 5, 7,10,11,10, 7 */ {1, 0, 1, 2, 1, 2},
    /* 158:  ,1,2,3,4, , ,7:  8, 0, 3,10, 4, 6, 4,10, 9 */ {1, 0, 1, 2, 1, 2},
    /* 174:  ,1,2,3, ,5, ,7:  5, 6,10, 3, 4, 0, 4, 3, 7 */ {1, 2, 1, 0, 1, 0},
    /* 182:  ,1,2, ,4,5, ,7:  5, 6,10, 0, 2, 8,11, 8, 2 */ {1, 2, 1, 0, 1, 0},
    /* 214:  ,1,2, ,4, ,6,7:  9, 4, 5,11, 0, 2, 0,11, 8 */ {1, 2, 1, 0, 1, 0},
    /* 186:  ,1, ,3,4,5, ,7:  8, 0, 3, 6, 2, 5, 1, 5, 2 */ {1, 0, 1, 2, 1, 2},
    /* 234:  ,1, ,3, ,5,6,7: 10, 2, 1, 4, 0, 7, 3, 7, 0 */ {1, 2, 1, 0, 1, 0},
    /* 124:  , ,2,3,4,5,6, :  6, 7,11, 1, 3, 9, 8, 9, 3 */ {1, 2, 1, 0, 1, 0},
    /* 188:  , ,2,3,4,5, ,7: 10, 5, 6, 8, 1, 3, 1, 8, 9 */ {1, 2, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion6_1_2[48][8] = {
    /*  67: 0,1, , , , ,6, :  1,12, 3,12,10, 3, 6, 3,10, 3, 6, 8, 5, 8, 6, 8, 5,12,12, 9, 8, 1, 9,12,12, 5,10 */
    {0, 0, 1, 1, 1, 1, 0, 1},
    /* 131: 0,1, , , , , ,7:  1,12, 3, 1,11,12,11, 1, 6, 9, 6, 1, 6, 9, 7,12, 7, 9, 9, 8,12,12, 8, 3,11, 7,12 */
    {0, 0, 1, 1, 1, 1, 1, 0},
    /*  21: 0, ,2, ,4, , , :  4,12, 0, 4, 1,12, 1, 4,10, 7,10, 4,10, 7, 2,12, 2, 7, 7, 3,12,12, 3, 0, 1, 2,12 */
    {0, 1, 0, 1, 0, 1, 1, 1},
    /*  69: 0, ,2, , , ,6, :  6,12, 2, 6, 3,12, 3, 6, 8, 5, 8, 6, 8, 5, 0,12, 0, 5, 5, 1,12,12, 1, 2, 3, 0,12 */
    {2, 1, 2, 1, 1, 1, 2, 1},
    /*  41: 0, , ,3, ,5, , :  0,12, 2,12, 9, 2, 5, 2, 9, 2, 5,11, 4,11, 5,11, 4,12,12, 8,11, 0, 8,12,12, 4, 9 */
    {0, 1, 1, 0, 1, 0, 1, 1},
    /*  73: 0, , ,3, , ,6, :  0,12, 2, 0,10,12,10, 0, 5, 8, 5, 0, 5, 8, 6,12, 6, 8, 8,11,12,12,11, 2,10, 6,12 */
    {0, 1, 1, 0, 1, 1, 0, 1},
    /*  81: 0, , , ,4, ,6, :  4,12, 0,12, 5, 0,10, 0, 5, 0,10, 3, 6, 3,10, 3, 6,12,12, 7, 3, 4, 7,12,12, 6, 5 */
    {0, 1, 1, 1, 0, 1, 0, 1},
    /*  97: 0, , , , ,5,6, :  4,12, 6,12, 8, 6, 3, 6, 8, 6, 3,10, 0,10, 3,10, 0,12,12, 9,10, 4, 9,12,12, 0, 8 */
    {2, 1, 1, 1, 1, 2, 2, 1},
    /* 193: 0, , , , , ,6,7:  5,12, 7, 5, 8,12, 8, 5, 0,10, 0, 5, 0,10, 3,12, 3,10,10,11,12,12,11, 7, 8, 3,12 */
    {2, 1, 1, 1, 1, 1, 2, 2},
    /*  22:  ,1,2, ,4, , , :  2,12, 0, 2, 8,12, 8, 2, 7,10, 7, 2, 7,10, 4,12, 4,10,10, 9,12,12, 9, 0, 8, 4,12 */
    {0, 1, 1, 0, 1, 0, 0, 0},
    /* 134:  ,1,2, , , , ,7:  2,12, 0,12,11, 0, 7, 0,11, 0, 7, 9, 6, 9, 7, 9, 6,12,12,10, 9, 2,10,12,12, 6,11 */
    {0, 1, 1, 0, 0, 0, 0, 1},
    /*  42:  ,1, ,3, ,5, , :  5,12, 1, 5, 2,12, 2, 5,11, 4,11, 5,11, 4, 3,12, 3, 4, 4, 0,12,12, 0, 1, 2, 3,12 */
    {0, 1, 0, 1, 0, 1, 0, 0},
    /* 138:  ,1, ,3, , , ,7:  7,12, 3, 7, 0,12, 0, 7, 9, 6, 9, 7, 9, 6, 1,12, 1, 6, 6, 2,12,12, 2, 3, 0, 1,12 */
    {0, 2, 0, 2, 0, 0, 0, 2},
    /* 146:  ,1, , ,4, , ,7:  6,12, 4, 6, 9,12, 9, 6, 1,11, 1, 6, 1,11, 0,12, 0,11,11, 8,12,12, 8, 4, 9, 0,12 */
    {0, 2, 0, 0, 2, 0, 0, 2},
    /* 162:  ,1, , , ,5, ,7:  5,12, 1,12, 6, 1,11, 1, 6, 1,11, 0, 7, 0,11, 0, 7,12,12, 4, 0, 5, 4,12,12, 7, 6 */
    {0, 1, 0, 0, 0, 1, 0, 1},
    /* 194:  ,1, , , , ,6,7:  5,12, 7,12, 9, 7, 0, 7, 9, 7, 0,11, 1,11, 0,11, 1,12,12,10,11, 5,10,12,12, 1, 9 */
    {0, 2, 0, 0, 0, 0, 2, 2},
    /*  28:  , ,2,3,4, , , :  3,12, 1,12, 8, 1, 4, 1, 8, 1, 4,10, 7,10, 4,10, 7,12,12,11,10, 3,11,12,12, 7, 8 */
    {0, 0, 1, 1, 1, 0, 0, 0},
    /*  44:  , ,2,3, ,5, , :  3,12, 1, 3, 9,12, 9, 3, 4,11, 4, 3, 4,11, 5,12, 5,11,11,10,12,12,10, 1, 9, 5,12 */
    {0, 0, 1, 1, 0, 1, 0, 0},
    /*  52:  , ,2, ,4,5, , :  7,12, 5, 7,10,12,10, 7, 2, 8, 2, 7, 2, 8, 1,12, 1, 8, 8, 9,12,12, 9, 5,10, 1,12 */
    {0, 0, 2, 0, 2, 2, 0, 0},
    /*  84:  , ,2, ,4, ,6, :  6,12, 2,12, 7, 2, 8, 2, 7, 2, 8, 1, 4, 1, 8, 1, 4,12,12, 5, 1, 6, 5,12,12, 4, 7 */
    {0, 0, 1, 0, 1, 0, 1, 0},
    /* 148:  , ,2, ,4, , ,7:  6,12, 4,12,10, 4, 1, 4,10, 4, 1, 8, 2, 8, 1, 8, 2,12,12,11, 8, 6,11,12,12, 2,10 */
    {0, 0, 2, 0, 2, 0, 0, 2},
    /*  56:  , , ,3,4,5, , :  7,12, 5,12,11, 5, 2, 5,11, 5, 2, 9, 3, 9, 2, 9, 3,12,12, 8, 9, 7, 8,12,12, 3,11 */
    {0, 0, 0, 2, 2, 2, 0, 0},
    /* 104:  , , ,3, ,5,6, :  4,12, 6, 4,11,12,11, 4, 3, 9, 3, 4, 3, 9, 2,12, 2, 9, 9,10,12,12,10, 6,11, 2,12 */
    {0, 0, 0, 2, 0, 2, 2, 0},
    /* 168:  , , ,3, ,5, ,7:  7,12, 3,12, 4, 3, 9, 3, 4, 3, 9, 2, 5, 2, 9, 2, 5,12,12, 6, 2, 7, 6,12,12, 5, 4 */
    {0, 0, 0, 1, 0, 1, 0, 1},
    /*  87: 0,1,2, ,4, ,6, :  3,12, 7, 3, 4,12, 4, 3, 9, 2, 9, 3, 9, 2, 5,12, 5, 2, 2, 6,12,12, 6, 7, 4, 5,12 */
    {0, 0, 0, 1, 0, 1, 0, 1},
    /* 151: 0,1,2, ,4, , ,7:  6,12, 4,12,11, 4, 3, 4,11, 4, 3, 9, 2, 9, 3, 9, 2,12,12,10, 9, 6,10,12,12, 2,11 */
    {0, 0, 0, 2, 0, 2, 2, 0},
    /* 199: 0,1,2, , , ,6,7:  5,12, 7, 5,11,12,11, 5, 2, 9, 2, 5, 2, 9, 3,12, 3, 9, 9, 8,12,12, 8, 7,11, 3,12 */
    {0, 0, 0, 2, 2, 2, 0, 0},
    /* 107: 0,1, ,3, ,5,6, :  4,12, 6, 4,10,12,10, 4, 1, 8, 1, 4, 1, 8, 2,12, 2, 8, 8,11,12,12,11, 6,10, 2,12 */
    {0, 0, 2, 0, 2, 0, 0, 2},
    /* 171: 0,1, ,3, ,5, ,7:  2,12, 6, 2, 7,12, 7, 2, 8, 1, 8, 2, 8, 1, 4,12, 4, 1, 1, 5,12,12, 5, 6, 7, 4,12 */
    {0, 0, 1, 0, 1, 0, 1, 0},
    /* 203: 0,1, ,3, , ,6,7:  5,12, 7,12,10, 7, 2, 7,10, 7, 2, 8, 1, 8, 2, 8, 1,12,12, 9, 8, 5, 9,12,12, 1,10 */
    {0, 0, 2, 0, 2, 2, 0, 0},
    /* 211: 0,1, , ,4, ,6,7:  1,12, 3,12, 9, 3, 4, 3, 9, 3, 4,11, 5,11, 4,11, 5,12,12,10,11, 1,10,12,12, 5, 9 */
    {0, 0, 1, 1, 0, 1, 0, 0},
    /* 227: 0,1, , , ,5,6,7:  1,12, 3, 1, 8,12, 8, 1, 4,10, 4, 1, 4,10, 7,12, 7,10,10,11,12,12,11, 3, 8, 7,12 */
    {0, 0, 1, 1, 1, 0, 0, 0},
    /*  61: 0, ,2,3,4,5, , :  7,12, 5, 7, 9,12, 9, 7, 0,11, 0, 7, 0,11, 1,12, 1,11,11,10,12,12,10, 5, 9, 1,12 */
    {0, 2, 0, 0, 0, 0, 2, 2},
    /*  93: 0, ,2,3,4, ,6, :  1,12, 5, 1, 6,12, 6, 1,11, 0,11, 1,11, 0, 7,12, 7, 0, 0, 4,12,12, 4, 5, 6, 7,12 */
    {0, 1, 0, 0, 0, 1, 0, 1},
    /* 109: 0, ,2,3, ,5,6, :  4,12, 6,12, 9, 6, 1, 6, 9, 6, 1,11, 0,11, 1,11, 0,12,12, 8,11, 4, 8,12,12, 0, 9 */
    {0, 2, 0, 0, 2, 0, 0, 2},
    /* 117: 0, ,2, ,4,5,6, :  3,12, 7,12, 0, 7, 9, 7, 0, 7, 9, 6, 1, 6, 9, 6, 1,12,12, 2, 6, 3, 2,12,12, 1, 0 */
    {0, 2, 0, 2, 0, 0, 0, 2},
    /* 213: 0, ,2, ,4, ,6,7:  1,12, 5,12, 2, 5,11, 5, 2, 5,11, 4, 3, 4,11, 4, 3,12,12, 0, 4, 1, 0,12,12, 3, 2 */
    {0, 1, 0, 1, 0, 1, 0, 0},
    /* 121: 0, , ,3,4,5,6, :  0,12, 2, 0,11,12,11, 0, 7, 9, 7, 0, 7, 9, 6,12, 6, 9, 9,10,12,12,10, 2,11, 6,12 */
    {0, 1, 1, 0, 0, 0, 0, 1},
    /* 233: 0, , ,3, ,5,6,7:  0,12, 2,12, 8, 2, 7, 2, 8, 2, 7,10, 4,10, 7,10, 4,12,12, 9,10, 0, 9,12,12, 4, 8 */
    {0, 1, 1, 0, 1, 0, 0, 0},
    /*  62:  ,1,2,3,4,5, , :  7,12, 5,12, 8, 5, 0, 5, 8, 5, 0,10, 3,10, 0,10, 3,12,12,11,10, 7,11,12,12, 3, 8 */
    {2, 1, 1, 1, 1, 1, 2, 2},
    /* 158:  ,1,2,3,4, , ,7:  6,12, 4, 6, 8,12, 8, 6, 3,10, 3, 6, 3,10, 0,12, 0,10,10, 9,12,12, 9, 4, 8, 0,12 */
    {2, 1, 1, 1, 1, 2, 2, 1},
    /* 174:  ,1,2,3, ,5, ,7:  0,12, 4, 0, 5,12, 5, 0,10, 3,10, 0,10, 3, 6,12, 6, 3, 3, 7,12,12, 7, 4, 5, 6,12 */
    {0, 1, 1, 1, 0, 1, 0, 1},
    /* 182:  ,1,2, ,4,5, ,7:  2,12, 0,12,10, 0, 5, 0,10, 0, 5, 8, 6, 8, 5, 8, 6,12,12,11, 8, 2,11,12,12, 6,10 */
    {0, 1, 1, 0, 1, 1, 0, 1},
    /* 214:  ,1,2, ,4, ,6,7:  2,12, 0, 2, 9,12, 9, 2, 5,11, 5, 2, 5,11, 4,12, 4,11,11, 8,12,12, 8, 0, 9, 4,12 */
    {0, 1, 1, 0, 1, 0, 1, 1},
    /* 186:  ,1, ,3,4,5, ,7:  2,12, 6,12, 3, 6, 8, 6, 3, 6, 8, 5, 0, 5, 8, 5, 0,12,12, 1, 5, 2, 1,12,12, 0, 3 */
    {2, 1, 2, 1, 1, 1, 2, 1},
    /* 234:  ,1, ,3, ,5,6,7:  0,12, 4,12, 1, 4,10, 4, 1, 4,10, 7, 2, 7,10, 7, 2,12,12, 3, 7, 0, 3,12,12, 2, 1 */
    {0, 1, 0, 1, 0, 1, 1, 1},
    /* 124:  , ,2,3,4,5,6, :  3,12, 1,12,11, 1, 6, 1,11, 1, 6, 9, 7, 9, 6, 9, 7,12,12, 8, 9, 3, 8,12,12, 7,11 */
    {0, 0, 1, 1, 1, 1, 1, 0},
    /* 188:  , ,2,3,4,5, ,7:  3,12, 1, 3,10,12,10, 3, 6, 8, 6, 3, 6, 8, 5,12, 5, 8, 8, 9,12,12, 9, 1,10, 5,12 */
    {0, 0, 1, 1, 1, 1, 0, 1},
};
static const signed char triangleRegion6_1_2[48][18] = {
    /*  67: 0,1, , , , ,6, :  1,12, 3,12,10, 3, 6, 3,10, 3, 6, 8, 5, 8, 6, 8, 5,12,12, 9, 8, 1, 9,12,12, 5,10 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 131: 0,1, , , , , ,7:  1,12, 3, 1,11,12,11, 1, 6, 9, 6, 1, 6, 9, 7,12, 7, 9, 9, 8,12,12, 8, 3,11, 7,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  21: 0, ,2, ,4, , , :  4,12, 0, 4, 1,12, 1, 4,10, 7,10, 4,10, 7, 2,12, 2, 7, 7, 3,12,12, 3, 0, 1, 2,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  69: 0, ,2, , , ,6, :  6,12, 2, 6, 3,12, 3, 6, 8, 5, 8, 6, 8, 5, 0,12, 0, 5, 5, 1,12,12, 1, 2, 3, 0,12 */
    {2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1},
    /*  41: 0, , ,3, ,5, , :  0,12, 2,12, 9, 2, 5, 2, 9, 2, 5,11, 4,11, 5,11, 4,12,12, 8,11, 0, 8,12,12, 4, 9 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  73: 0, , ,3, , ,6, :  0,12, 2, 0,10,12,10, 0, 5, 8, 5, 0, 5, 8, 6,12, 6, 8, 8,11,12,12,11, 2,10, 6,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  81: 0, , , ,4, ,6, :  4,12, 0,12, 5, 0,10, 0, 5, 0,10, 3, 6, 3,10, 3, 6,12,12, 7, 3, 4, 7,12,12, 6, 5 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  97: 0, , , , ,5,6, :  4,12, 6,12, 8, 6, 3, 6, 8, 6, 3,10, 0,10, 3,10, 0,12,12, 9,10, 4, 9,12,12, 0, 8 */
    {2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1},
    /* 193: 0, , , , , ,6,7:  5,12, 7, 5, 8,12, 8, 5, 0,10, 0, 5, 0,10, 3,12, 3,10,10,11,12,12,11, 7, 8, 3,12 */
    {2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1},
    /*  22:  ,1,2, ,4, , , :  2,12, 0, 2, 8,12, 8, 2, 7,10, 7, 2, 7,10, 4,12, 4,10,10, 9,12,12, 9, 0, 8, 4,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 134:  ,1,2, , , , ,7:  2,12, 0,12,11, 0, 7, 0,11, 0, 7, 9, 6, 9, 7, 9, 6,12,12,10, 9, 2,10,12,12, 6,11 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  42:  ,1, ,3, ,5, , :  5,12, 1, 5, 2,12, 2, 5,11, 4,11, 5,11, 4, 3,12, 3, 4, 4, 0,12,12, 0, 1, 2, 3,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 138:  ,1, ,3, , , ,7:  7,12, 3, 7, 0,12, 0, 7, 9, 6, 9, 7, 9, 6, 1,12, 1, 6, 6, 2,12,12, 2, 3, 0, 1,12 */
    {2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    /* 146:  ,1, , ,4, , ,7:  6,12, 4, 6, 9,12, 9, 6, 1,11, 1, 6, 1,11, 0,12, 0,11,11, 8,12,12, 8, 4, 9, 0,12 */
    {2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    /* 162:  ,1, , , ,5, ,7:  5,12, 1,12, 6, 1,11, 1, 6, 1,11, 0, 7, 0,11, 0, 7,12,12, 4, 0, 5, 4,12,12, 7, 6 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 194:  ,1, , , , ,6,7:  5,12, 7,12, 9, 7, 0, 7, 9, 7, 0,11, 1,11, 0,11, 1,12,12,10,11, 5,10,12,12, 1, 9 */
    {2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    /*  28:  , ,2,3,4, , , :  3,12, 1,12, 8, 1, 4, 1, 8, 1, 4,10, 7,10, 4,10, 7,12,12,11,10, 3,11,12,12, 7, 8 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  44:  , ,2,3, ,5, , :  3,12, 1, 3, 9,12, 9, 3, 4,11, 4, 3, 4,11, 5,12, 5,11,11,10,12,12,10, 1, 9, 5,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  52:  , ,2, ,4,5, , :  7,12, 5, 7,10,12,10, 7, 2, 8, 2, 7, 2, 8, 1,12, 1, 8, 8, 9,12,12, 9, 5,10, 1,12 */
    {2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    /*  84:  , ,2, ,4, ,6, :  6,12, 2,12, 7, 2, 8, 2, 7, 2, 8, 1, 4, 1, 8, 1, 4,12,12, 5, 1, 6, 5,12,12, 4, 7 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 148:  , ,2, ,4, , ,7:  6,12, 4,12,10, 4, 1, 4,10, 4, 1, 8, 2, 8, 1, 8, 2,12,12,11, 8, 6,11,12,12, 2,10 */
    {2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    /*  56:  , , ,3,4,5, , :  7,12, 5,12,11, 5, 2, 5,11, 5, 2, 9, 3, 9, 2, 9, 3,12,12, 8, 9, 7, 8,12,12, 3,11 */
    {2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    /* 104:  , , ,3, ,5,6, :  4,12, 6, 4,11,12,11, 4, 3, 9, 3, 4, 3, 9, 2,12, 2, 9, 9,10,12,12,10, 6,11, 2,12 */
    {2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    /* 168:  , , ,3, ,5, ,7:  7,12, 3,12, 4, 3, 9, 3, 4, 3, 9, 2, 5, 2, 9, 2, 5,12,12, 6, 2, 7, 6,12,12, 5, 4 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  87: 0,1,2, ,4, ,6, :  3,12, 7, 3, 4,12, 4, 3, 9, 2, 9, 3, 9, 2, 5,12, 5, 2, 2, 6,12,12, 6, 7, 4, 5,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 151: 0,1,2, ,4, , ,7:  6,12, 4,12,11, 4, 3, 4,11, 4, 3, 9, 2, 9, 3, 9, 2,12,12,10, 9, 6,10,12,12, 2,11 */
    {0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
    /* 199: 0,1,2, , , ,6,7:  5,12, 7, 5,11,12,11, 5, 2, 9, 2, 5, 2, 9, 3,12, 3, 9, 9, 8,12,12, 8, 7,11, 3,12 */
    {0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
    /* 107: 0,1, ,3, ,5,6, :  4,12, 6, 4,10,12,10, 4, 1, 8, 1, 4, 1, 8, 2,12, 2, 8, 8,11,12,12,11, 6,10, 2,12 */
    {0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
    /* 171: 0,1, ,3, ,5, ,7:  2,12, 6, 2, 7,12, 7, 2, 8, 1, 8, 2, 8, 1, 4,12, 4, 1, 1, 5,12,12, 5, 6, 7, 4,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 203: 0,1, ,3, , ,6,7:  5,12, 7,12,10, 7, 2, 7,10, 7, 2, 8, 1, 8, 2, 8, 1,12,12, 9, 8, 5, 9,12,12, 1,10 */
    {0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
    /* 211: 0,1, , ,4, ,6,7:  1,12, 3,12, 9, 3, 4, 3, 9, 3, 4,11, 5,11, 4,11, 5,12,12,10,11, 1,10,12,12, 5, 9 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 227: 0,1, , , ,5,6,7:  1,12, 3, 1, 8,12, 8, 1, 4,10, 4, 1, 4,10, 7,12, 7,10,10,11,12,12,11, 3, 8, 7,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  61: 0, ,2,3,4,5, , :  7,12, 5, 7, 9,12, 9, 7, 0,11, 0, 7, 0,11, 1,12, 1,11,11,10,12,12,10, 5, 9, 1,12 */
    {0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
    /*  93: 0, ,2,3,4, ,6, :  1,12, 5, 1, 6,12, 6, 1,11, 0,11, 1,11, 0, 7,12, 7, 0, 0, 4,12,12, 4, 5, 6, 7,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 109: 0, ,2,3, ,5,6, :  4,12, 6,12, 9, 6, 1, 6, 9, 6, 1,11, 0,11, 1,11, 0,12,12, 8,11, 4, 8,12,12, 0, 9 */
    {0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
    /* 117: 0, ,2, ,4,5,6, :  3,12, 7,12, 0, 7, 9, 7, 0, 7, 9, 6, 1, 6, 9, 6, 1,12,12, 2, 6, 3, 2,12,12, 1, 0 */
    {0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
    /* 213: 0, ,2, ,4, ,6,7:  1,12, 5,12, 2, 5,11, 5, 2, 5,11, 4, 3, 4,11, 4, 3,12,12, 0, 4, 1, 0,12,12, 3, 2 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 121: 0, , ,3,4,5,6, :  0,12, 2, 0,11,12,11, 0, 7, 9, 7, 0, 7, 9, 6,12, 6, 9, 9,10,12,12,10, 2,11, 6,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 233: 0, , ,3, ,5,6,7:  0,12, 2,12, 8, 2, 7, 2, 8, 2, 7,10, 4,10, 7,10, 4,12,12, 9,10, 0, 9,12,12, 4, 8 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  62:  ,1,2,3,4,5, , :  7,12, 5,12, 8, 5, 0, 5, 8, 5, 0,10, 3,10, 0,10, 3,12,12,11,10, 7,11,12,12, 3, 8 */
    {1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2},
    /* 158:  ,1,2,3,4, , ,7:  6,12, 4, 6, 8,12, 8, 6, 3,10, 3, 6, 3,10, 0,12, 0,10,10, 9,12,12, 9, 4, 8, 0,12 */
    {1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2},
    /* 174:  ,1,2,3, ,5, ,7:  0,12, 4, 0, 5,12, 5, 0,10, 3,10, 0,10, 3, 6,12, 6, 3, 3, 7,12,12, 7, 4, 5, 6,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 182:  ,1,2, ,4,5, ,7:  2,12, 0,12,10, 0, 5, 0,10, 0, 5, 8, 6, 8, 5, 8, 6,12,12,11, 8, 2,11,12,12, 6,10 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 214:  ,1,2, ,4, ,6,7:  2,12, 0, 2, 9,12, 9, 2, 5,11, 5, 2, 5,11, 4,12, 4,11,11, 8,12,12, 8, 0, 9, 4,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 186:  ,1, ,3,4,5, ,7:  2,12, 6,12, 3, 6, 8, 6, 3, 6, 8, 5, 0, 5, 8, 5, 0,12,12, 1, 5, 2, 1,12,12, 0, 3 */
    {1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2},
    /* 234:  ,1, ,3, ,5,6,7:  0,12, 4,12, 1, 4,10, 4, 1, 4,10, 7, 2, 7,10, 7, 2,12,12, 3, 7, 0, 3,12,12, 2, 1 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 124:  , ,2,3,4,5,6, :  3,12, 1,12,11, 1, 6, 1,11, 1, 6, 9, 7, 9, 6, 9, 7,12,12, 8, 9, 3, 8,12,12, 7,11 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 188:  , ,2,3,4,5, ,7:  3,12, 1, 3,10,12,10, 3, 6, 8, 6, 3, 6, 8, 5,12, 5, 8, 8, 9,12,12, 9, 1,10, 5,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion6_2[48][8] = {
    /*  67: 0,1, , , , ,6, :  1,10, 3, 6, 3,10, 3, 6, 8, 5, 8, 6, 8, 5, 9 */ {0, 0, 1, 1, 1, 1, 0, 1},
    /* 131: 0,1, , , , , ,7:  1,11, 3,11, 1, 6, 9, 6, 1, 6, 9, 7, 8, 7, 9 */ {0, 0, 1, 1, 1, 1, 1, 0},
    /*  21: 0, ,2, ,4, , , :  4, 1, 0, 1, 4,10, 7,10, 4,10, 7, 2, 3, 2, 7 */ {0, 1, 0, 1, 0, 1, 1, 1},
    /*  69: 0, ,2, , , ,6, :  6, 3, 2, 3, 6, 8, 5, 8, 6, 8, 5, 0, 1, 0, 5 */ {0, 1, 0, 1, 1, 1, 0, 1},
    /*  41: 0, , ,3, ,5, , :  0, 9, 2, 5, 2, 9, 2, 5,11, 4,11, 5,11, 4, 8 */ {0, 1, 1, 0, 1, 0, 1, 1},
    /*  73: 0, , ,3, , ,6, :  0,10, 2,10, 0, 5, 8, 5, 0, 5, 8, 6,11, 6, 8 */ {0, 1, 1, 0, 1, 1, 0, 1},
    /*  81: 0, , , ,4, ,6, :  4, 5, 0,10, 0, 5, 0,10, 3, 6, 3,10, 3, 6, 7 */ {0, 1, 1, 1, 0, 1, 0, 1},
    /*  97: 0, , , , ,5,6, :  4, 8, 6, 3, 6, 8, 6, 3,10, 0,10, 3,10, 0, 9 */ {0, 1, 1, 1, 1, 0, 0, 1},
    /* 193: 0, , , , , ,6,7:  5, 8, 7, 8, 5, 0,10, 0, 5, 0,10, 3,11, 3,10 */ {0, 1, 1, 1, 1, 1, 0, 0},
    /*  22:  ,1,2, ,4, , , :  2, 8, 0, 8, 2, 7,10, 7, 2, 7,10, 4, 9, 4,10 */ {0, 1, 1, 0, 1, 0, 0, 0},
    /* 134:  ,1,2, , , , ,7:  2,11, 0, 7, 0,11, 0, 7, 9, 6, 9, 7, 9, 6,10 */ {0, 1, 1, 0, 0, 0, 0, 1},
    /*  42:  ,1, ,3, ,5, , :  5, 2, 1, 2, 5,11, 4,11, 5,11, 4, 3, 0, 3, 4 */ {0, 1, 0, 1, 0, 1, 0, 0},
    /* 138:  ,1, ,3, , , ,7:  7, 0, 3, 0, 7, 9, 6, 9, 7, 9, 6, 1, 2, 1, 6 */ {0, 1, 0, 1, 0, 0, 0, 1},
    /* 146:  ,1, , ,4, , ,7:  6, 9, 4, 9, 6, 1,11, 1, 6, 1,11, 0, 8, 0,11 */ {0, 1, 0, 0, 1, 0, 0, 1},
    /* 162:  ,1, , , ,5, ,7:  5, 6, 1,11, 1, 6, 1,11, 0, 7, 0,11, 0, 7, 4 */ {0, 1, 0, 0, 0, 1, 0, 1},
    /* 194:  ,1, , , , ,6,7:  5, 9, 7, 0, 7, 9, 7, 0,11, 1,11, 0,11, 1,10 */ {0, 1, 0, 0, 0, 0, 1, 1},
    /*  28:  , ,2,3,4, , , :  3, 8, 1, 4, 1, 8, 1, 4,10, 7,10, 4,10, 7,11 */ {0, 0, 1, 1, 1, 0, 0, 0},
    /*  44:  , ,2,3, ,5, , :  3, 9, 1, 9, 3, 4,11, 4, 3, 4,11, 5,10, 5,11 */ {0, 0, 1, 1, 0, 1, 0, 0},
    /*  52:  , ,2, ,4,5, , :  7,10, 5,10, 7, 2, 8, 2, 7, 2, 8, 1, 9, 1, 8 */ {0, 0, 1, 0, 1, 1, 0, 0},
    /*  84:  , ,2, ,4, ,6, :  6, 7, 2, 8, 2, 7, 2, 8, 1, 4, 1, 8, 1, 4, 5 */ {0, 0, 1, 0, 1, 0, 1, 0},
    /* 148:  , ,2, ,4, , ,7:  6,10, 4, 1, 4,10, 4, 1, 8, 2, 8, 1, 8, 2,11 */ {0, 0, 1, 0, 1, 0, 0, 1},
    /*  56:  , , ,3,4,5, , :  7,11, 5, 2, 5,11, 5, 2, 9, 3, 9, 2, 9, 3, 8 */ {0, 0, 0, 1, 1, 1, 0, 0},
    /* 104:  , , ,3, ,5,6, :  4,11, 6,11, 4, 3, 9, 3, 4, 3, 9, 2,10, 2, 9 */ {0, 0, 0, 1, 0, 1, 1, 0},
    /* 168:  , , ,3, ,5, ,7:  7, 4, 3, 9, 3, 4, 3, 9, 2, 5, 2, 9, 2, 5, 6 */ {0, 0, 0, 1, 0, 1, 0, 1},
    /*  87: 0,1,2, ,4, ,6, :  3, 4, 7, 4, 3, 9, 2, 9, 3, 9, 2, 5, 6, 5, 2 */ {0, 0, 0, 1, 0, 1, 0, 1},
    /* 151: 0,1,2, ,4, , ,7:  6,11, 4, 3, 4,11, 4, 3, 9, 2, 9, 3, 9, 2,10 */ {0, 0, 0, 1, 0, 1, 1, 0},
    /* 199: 0,1,2, , , ,6,7:  5,11, 7,11, 5, 2, 9, 2, 5, 2, 9, 3, 8, 3, 9 */ {0, 0, 0, 1, 1, 1, 0, 0},
    /* 107: 0,1, ,3, ,5,6, :  4,10, 6,10, 4, 1, 8, 1, 4, 1, 8, 2,11, 2, 8 */ {0, 0, 1, 0, 1, 0, 0, 1},
    /* 171: 0,1, ,3, ,5, ,7:  2, 7, 6, 7, 2, 8, 1, 8, 2, 8, 1, 4, 5, 4, 1 */ {0, 0, 1, 0, 1, 0, 1, 0},
    /* 203: 0,1, ,3, , ,6,7:  5,10, 7, 2, 7,10, 7, 2, 8, 1, 8, 2, 8, 1, 9 */ {0, 0, 1, 0, 1, 1, 0, 0},
    /* 211: 0,1, , ,4, ,6,7:  1, 9, 3, 4, 3, 9, 3, 4,11, 5,11, 4,11, 5,10 */ {0, 0, 1, 1, 0, 1, 0, 0},
    /* 227: 0,1, , , ,5,6,7:  1, 8, 3, 8, 1, 4,10, 4, 1, 4,10, 7,11, 7,10 */ {0, 0, 1, 1, 1, 0, 0, 0},
    /*  61: 0, ,2,3,4,5, , :  7, 9, 5, 9, 7, 0,11, 0, 7, 0,11, 1,10, 1,11 */ {0, 1, 0, 0, 0, 0, 1, 1},
    /*  93: 0, ,2,3,4, ,6, :  1, 6, 5, 6, 1,11, 0,11, 1,11, 0, 7, 4, 7, 0 */ {0, 1, 0, 0, 0, 1, 0, 1},
    /* 109: 0, ,2,3, ,5,6, :  4, 9, 6, 1, 6, 9, 6, 1,11, 0,11, 1,11, 0, 8 */ {0, 1, 0, 0, 1, 0, 0, 1},
    /* 117: 0, ,2, ,4,5,6, :  3, 0, 7, 9, 7, 0, 7, 9, 6, 1, 6, 9, 6, 1, 2 */ {0, 1, 0, 1, 0, 0, 0, 1},
    /* 213: 0, ,2, ,4, ,6,7:  1, 2, 5,11, 5, 2, 5,11, 4, 3, 4,11, 4, 3, 0 */ {0, 1, 0, 1, 0, 1, 0, 0},
    /* 121: 0, , ,3,4,5,6, :  0,11, 2,11, 0, 7, 9, 7, 0, 7, 9, 6,10, 6, 9 */ {0, 1, 1, 0, 0, 0, 0, 1},
    /* 233: 0, , ,3, ,5,6,7:  0, 8, 2, 7, 2, 8, 2, 7,10, 4,10, 7,10, 4, 9 */ {0, 1, 1, 0, 1, 0, 0, 0},
    /*  62:  ,1,2,3,4,5, , :  7, 8, 5, 0, 5, 8, 5, 0,10, 3,10, 0,10, 3,11 */ {0, 1, 1, 1, 1, 1, 0, 0},
    /* 158:  ,1,2,3,4, , ,7:  6, 8, 4, 8, 6, 3,10, 3, 6, 3,10, 0, 9, 0,10 */ {0, 1, 1, 1, 1, 0, 0, 1},
    /* 174:  ,1,2,3, ,5, ,7:  0, 5, 4, 5, 0,10, 3,10, 0,10, 3, 6, 7, 6, 3 */ {0, 1, 1, 1, 0, 1, 0, 1},
    /* 182:  ,1,2, ,4,5, ,7:  2,10, 0, 5, 0,10, 0, 5, 8, 6, 8, 5, 8, 6,11 */ {0, 1, 1, 0, 1, 1, 0, 1},
    /* 214:  ,1,2, ,4, ,6,7:  2, 9, 0, 9, 2, 5,11, 5, 2, 5,11, 4, 8, 4,11 */ {0, 1, 1, 0, 1, 0, 1, 1},
    /* 186:  ,1, ,3,4,5, ,7:  2, 3, 6, 8, 6, 3, 6, 8, 5, 0, 5, 8, 5, 0, 1 */ {0, 1, 0, 1, 1, 1, 0, 1},
    /* 234:  ,1, ,3, ,5,6,7:  0, 1, 4,10, 4, 1, 4,10, 7, 2, 7,10, 7, 2, 3 */ {0, 1, 0, 1, 0, 1, 1, 1},
    /* 124:  , ,2,3,4,5,6, :  3,11, 1, 6, 1,11, 1, 6, 9, 7, 9, 6, 9, 7, 8 */ {0, 0, 1, 1, 1, 1, 1, 0},
    /* 188:  , ,2,3,4,5, ,7:  3,10, 1,10, 3, 6, 8, 6, 3, 6, 8, 5, 9, 5, 8 */ {0, 0, 1, 1, 1, 1, 0, 1},
};
static const signed char triangleRegion6_2[48][10] = {
    /*  67: 0,1, , , , ,6, :  1,10, 3, 6, 3,10, 3, 6, 8, 5, 8, 6, 8, 5, 9 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 131: 0,1, , , , , ,7:  1,11, 3,11, 1, 6, 9, 6, 1, 6, 9, 7, 8, 7, 9 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  21: 0, ,2, ,4, , , :  4, 1, 0, 1, 4,10, 7,10, 4,10, 7, 2, 3, 2, 7 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  69: 0, ,2, , , ,6, :  6, 3, 2, 3, 6, 8, 5, 8, 6, 8, 5, 0, 1, 0, 5 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  41: 0, , ,3, ,5, , :  0, 9, 2, 5, 2, 9, 2, 5,11, 4,11, 5,11, 4, 8 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  73: 0, , ,3, , ,6, :  0,10, 2,10, 0, 5, 8, 5, 0, 5, 8, 6,11, 6, 8 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  81: 0, , , ,4, ,6, :  4, 5, 0,10, 0, 5, 0,10, 3, 6, 3,10, 3, 6, 7 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  97: 0, , , , ,5,6, :  4, 8, 6, 3, 6, 8, 6, 3,10, 0,10, 3,10, 0, 9 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 193: 0, , , , , ,6,7:  5, 8, 7, 8, 5, 0,10, 0, 5, 0,10, 3,11, 3,10 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  22:  ,1,2, ,4, , , :  2, 8, 0, 8, 2, 7,10, 7, 2, 7,10, 4, 9, 4,10 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 134:  ,1,2, , , , ,7:  2,11, 0, 7, 0,11, 0, 7, 9, 6, 9, 7, 9, 6,10 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  42:  ,1, ,3, ,5, , :  5, 2, 1, 2, 5,11, 4,11, 5,11, 4, 3, 0, 3, 4 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 138:  ,1, ,3, , , ,7:  7, 0, 3, 0, 7, 9, 6, 9, 7, 9, 6, 1, 2, 1, 6 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 146:  ,1, , ,4, , ,7:  6, 9, 4, 9, 6, 1,11, 1, 6, 1,11, 0, 8, 0,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 162:  ,1, , , ,5, ,7:  5, 6, 1,11, 1, 6, 1,11, 0, 7, 0,11, 0, 7, 4 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 194:  ,1, , , , ,6,7:  5, 9, 7, 0, 7, 9, 7, 0,11, 1,11, 0,11, 1,10 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  28:  , ,2,3,4, , , :  3, 8, 1, 4, 1, 8, 1, 4,10, 7,10, 4,10, 7,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  44:  , ,2,3, ,5, , :  3, 9, 1, 9, 3, 4,11, 4, 3, 4,11, 5,10, 5,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  52:  , ,2, ,4,5, , :  7,10, 5,10, 7, 2, 8, 2, 7, 2, 8, 1, 9, 1, 8 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  84:  , ,2, ,4, ,6, :  6, 7, 2, 8, 2, 7, 2, 8, 1, 4, 1, 8, 1, 4, 5 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 148:  , ,2, ,4, , ,7:  6,10, 4, 1, 4,10, 4, 1, 8, 2, 8, 1, 8, 2,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  56:  , , ,3,4,5, , :  7,11, 5, 2, 5,11, 5, 2, 9, 3, 9, 2, 9, 3, 8 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 104:  , , ,3, ,5,6, :  4,11, 6,11, 4, 3, 9, 3, 4, 3, 9, 2,10, 2, 9 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 168:  , , ,3, ,5, ,7:  7, 4, 3, 9, 3, 4, 3, 9, 2, 5, 2, 9, 2, 5, 6 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  87: 0,1,2, ,4, ,6, :  3, 4, 7, 4, 3, 9, 2, 9, 3, 9, 2, 5, 6, 5, 2 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 151: 0,1,2, ,4, , ,7:  6,11, 4, 3, 4,11, 4, 3, 9, 2, 9, 3, 9, 2,10 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 199: 0,1,2, , , ,6,7:  5,11, 7,11, 5, 2, 9, 2, 5, 2, 9, 3, 8, 3, 9 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 107: 0,1, ,3, ,5,6, :  4,10, 6,10, 4, 1, 8, 1, 4, 1, 8, 2,11, 2, 8 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 171: 0,1, ,3, ,5, ,7:  2, 7, 6, 7, 2, 8, 1, 8, 2, 8, 1, 4, 5, 4, 1 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 203: 0,1, ,3, , ,6,7:  5,10, 7, 2, 7,10, 7, 2, 8, 1, 8, 2, 8, 1, 9 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 211: 0,1, , ,4, ,6,7:  1, 9, 3, 4, 3, 9, 3, 4,11, 5,11, 4,11, 5,10 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 227: 0,1, , , ,5,6,7:  1, 8, 3, 8, 1, 4,10, 4, 1, 4,10, 7,11, 7,10 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  61: 0, ,2,3,4,5, , :  7, 9, 5, 9, 7, 0,11, 0, 7, 0,11, 1,10, 1,11 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  93: 0, ,2,3,4, ,6, :  1, 6, 5, 6, 1,11, 0,11, 1,11, 0, 7, 4, 7, 0 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 109: 0, ,2,3, ,5,6, :  4, 9, 6, 1, 6, 9, 6, 1,11, 0,11, 1,11, 0, 8 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 117: 0, ,2, ,4,5,6, :  3, 0, 7, 9, 7, 0, 7, 9, 6, 1, 6, 9, 6, 1, 2 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 213: 0, ,2, ,4, ,6,7:  1, 2, 5,11, 5, 2, 5,11, 4, 3, 4,11, 4, 3, 0 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 121: 0, , ,3,4,5,6, :  0,11, 2,11, 0, 7, 9, 7, 0, 7, 9, 6,10, 6, 9 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 233: 0, , ,3, ,5,6,7:  0, 8, 2, 7, 2, 8, 2, 7,10, 4,10, 7,10, 4, 9 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  62:  ,1,2,3,4,5, , :  7, 8, 5, 0, 5, 8, 5, 0,10, 3,10, 0,10, 3,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 158:  ,1,2,3,4, , ,7:  6, 8, 4, 8, 6, 3,10, 3, 6, 3,10, 0, 9, 0,10 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 174:  ,1,2,3, ,5, ,7:  0, 5, 4, 5, 0,10, 3,10, 0,10, 3, 6, 7, 6, 3 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 182:  ,1,2, ,4,5, ,7:  2,10, 0, 5, 0,10, 0, 5, 8, 6, 8, 5, 8, 6,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 214:  ,1,2, ,4, ,6,7:  2, 9, 0, 9, 2, 5,11, 5, 2, 5,11, 4, 8, 4,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 186:  ,1, ,3,4,5, ,7:  2, 3, 6, 8, 6, 3, 6, 8, 5, 0, 5, 8, 5, 0, 1 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 234:  ,1, ,3, ,5,6,7:  0, 1, 4,10, 4, 1, 4,10, 7, 2, 7,10, 7, 2, 3 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 124:  , ,2,3,4,5,6, :  3,11, 1, 6, 1,11, 1, 6, 9, 7, 9, 6, 9, 7, 8 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 188:  , ,2,3,4,5, ,7:  3,10, 1,10, 3, 6, 8, 6, 3, 6, 8, 5, 9, 5, 8 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion7_1[16][8] = {
    /*  37: 0, ,2, , ,5, , :  9, 5, 4,10, 1, 2, 8, 3, 0 */ {0, 1, 2, 1, 1, 3, 1, 1},
    /* 133: 0, ,2, , , , ,7: 11, 7, 6, 8, 3, 0,10, 1, 2 */ {0, 1, 2, 1, 1, 1, 1, 3},
    /* 161: 0, , , , ,5, ,7:  3, 0, 8, 5, 4, 9, 7, 6,11 */ {0, 1, 1, 1, 1, 2, 1, 3},
    /*  26:  ,1, ,3,4, , , :  8, 4, 7, 9, 0, 1,11, 2, 3 */ {0, 1, 0, 2, 3, 0, 0, 0},
    /*  74:  ,1, ,3, , ,6, : 10, 6, 5,11, 2, 3, 9, 0, 1 */ {0, 1, 0, 2, 0, 0, 3, 0},
    /*  82:  ,1, , ,4, ,6, :  0, 1, 9, 6, 5,10, 4, 7, 8 */ {0, 1, 0, 0, 2, 0, 3, 0},
    /* 164:  , ,2, , ,5, ,7:  1, 2,10, 7, 6,11, 5, 4, 9 */ {0, 0, 1, 0, 0, 2, 0, 3},
    /*  88:  , , ,3,4, ,6, :  2, 3,11, 4, 7, 8, 6, 5,10 */ {0, 0, 0, 1, 2, 0, 3, 0},
    /* 167: 0,1,2, , ,5, ,7: 11, 3, 2, 8, 7, 4,10, 5, 6 */ {0, 0, 0, 1, 2, 0, 3, 0},
    /*  91: 0,1, ,3,4, ,6, : 10, 2, 1,11, 6, 7, 9, 4, 5 */ {0, 0, 1, 0, 0, 2, 0, 3},
    /* 173: 0, ,2,3, ,5, ,7:  9, 1, 0,10, 5, 6, 8, 7, 4 */ {0, 1, 0, 0, 2, 0, 3, 0},
    /* 181: 0, ,2, ,4,5, ,7:  5, 6,10, 3, 2,11, 1, 0, 9 */ {0, 1, 0, 2, 0, 0, 3, 0},
    /* 229: 0, ,2, , ,5,6,7:  7, 4, 8, 1, 0, 9, 3, 2,11 */ {0, 1, 0, 2, 3, 0, 0, 0},
    /*  94:  ,1,2,3,4, ,6, :  8, 0, 3, 9, 4, 5,11, 6, 7 */ {0, 1, 1, 1, 1, 2, 1, 3},
    /* 122:  ,1, ,3,4,5,6, :  6, 7,11, 0, 3, 8, 2, 1,10 */ {0, 1, 2, 1, 1, 1, 1, 3},
    /* 218:  ,1, ,3,4, ,6,7:  4, 5, 9, 2, 1,10, 0, 3, 8 */ {0, 1, 2, 1, 1, 3, 1, 1},
};
static const signed char triangleRegion7_1[16][6] = {
    /*  37: 0, ,2, , ,5, , :  9, 5, 4,10, 1, 2, 8, 3, 0 */ {3, 1, 2, 1, 0, 1},
    /* 133: 0, ,2, , , , ,7: 11, 7, 6, 8, 3, 0,10, 1, 2 */ {3, 1, 0, 1, 2, 1},
    /* 161: 0, , , , ,5, ,7:  3, 0, 8, 5, 4, 9, 7, 6,11 */ {0, 1, 2, 1, 3, 1},
    /*  26:  ,1, ,3,4, , , :  8, 4, 7, 9, 0, 1,11, 2, 3 */ {3, 0, 1, 0, 2, 0},
    /*  74:  ,1, ,3, , ,6, : 10, 6, 5,11, 2, 3, 9, 0, 1 */ {3, 0, 2, 0, 1, 0},
    /*  82:  ,1, , ,4, ,6, :  0, 1, 9, 6, 5,10, 4, 7, 8 */ {1, 0, 3, 0, 2, 0},
    /* 164:  , ,2, , ,5, ,7:  1, 2,10, 7, 6,11, 5, 4, 9 */ {1, 0, 3, 0, 2, 0},
    /*  88:  , , ,3,4, ,6, :  2, 3,11, 4, 7, 8, 6, 5,10 */ {1, 0, 2, 0, 3, 0},
    /* 167: 0,1,2, , ,5, ,7: 11, 3, 2, 8, 7, 4,10, 5, 6 */ {0, 1, 0, 2, 0, 3},
    /*  91: 0,1, ,3,4, ,6, : 10, 2, 1,11, 6, 7, 9, 4, 5 */ {0, 1, 0, 3, 0, 2},
    /* 173: 0, ,2,3, ,5, ,7:  9, 1, 0,10, 5, 6, 8, 7, 4 */ {0, 1, 0, 3, 0, 2},
    /* 181: 0, ,2, ,4,5, ,7:  5, 6,10, 3, 2,11, 1, 0, 9 */ {0, 3, 0, 2, 0, 1},
    /* 229: 0, ,2, , ,5,6,7:  7, 4, 8, 1, 0, 9, 3, 2,11 */ {0, 3, 0, 1, 0, 2},
    /*  94:  ,1,2,3,4, ,6, :  8, 0, 3, 9, 4, 5,11, 6, 7 */ {1, 0, 1, 2, 1, 3},
    /* 122:  ,1, ,3,4,5,6, :  6, 7,11, 0, 3, 8, 2, 1,10 */ {1, 3, 1, 0, 1, 2},
    /* 218:  ,1, ,3,4, ,6,7:  4, 5, 9, 2, 1,10, 0, 3, 8 */ {1, 3, 1, 2, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion7_2[16][3][8] = {
    {
        /*  37: 0, ,2, , ,5, , :  1, 2,10, 3, 4, 8, 4, 3, 5, 0, 5, 3, 5, 0, 9 */ {0, 1, 2, 1, 1, 0, 1, 1},
        /*  37: 0, ,2, , ,5, , :  3, 0, 8, 9, 1, 4, 2, 4, 1, 4, 2, 5,10, 5, 2 */ {0, 1, 2, 1, 1, 2, 1, 1},
        /*  37: 0, ,2, , ,5, , :  9, 5, 4, 0,10, 1,10, 0, 8,10, 8, 2, 3, 2, 8 */ {0, 1, 0, 1, 1, 2, 1, 1},
    },
    {
        /* 133: 0, ,2, , , , ,7:  3, 0, 8, 1, 6,10, 6, 1, 7, 2, 7, 1, 7, 2,11 */ {0, 1, 2, 1, 1, 1, 1, 2},
        /* 133: 0, ,2, , , , ,7:  1, 2,10,11, 3, 6, 0, 6, 3, 6, 0, 7, 8, 7, 0 */ {0, 1, 2, 1, 1, 1, 1, 0},
        /* 133: 0, ,2, , , , ,7: 11, 7, 6, 2, 8, 3, 8, 2,10, 8,10, 0, 1, 0,10 */ {0, 1, 0, 1, 1, 1, 1, 2},
    },
    {
        /* 161: 0, , , , ,5, ,7:  9, 5, 4,11, 3, 6, 0, 6, 3, 6, 0, 7, 8, 7, 0 */ {0, 1, 1, 1, 1, 2, 1, 0},
        /* 161: 0, , , , ,5, ,7: 11, 7, 6, 3, 4, 8, 4, 3, 5, 0, 5, 3, 5, 0, 9 */ {0, 1, 1, 1, 1, 0, 1, 2},
        /* 161: 0, , , , ,5, ,7:  3, 0, 8, 4, 9, 7,11, 7, 9, 5,11, 9,11, 5, 6 */ {0, 1, 1, 1, 1, 2, 1, 2},
    },
    {
        /*  26:  ,1, ,3,4, , , :  0, 1, 9, 2, 7,11, 7, 2, 4, 3, 4, 2, 4, 3, 8 */ {0, 1, 0, 2, 2, 0, 0, 0},
        /*  26:  ,1, ,3,4, , , :  2, 3,11, 8, 0, 7, 1, 7, 0, 7, 1, 4, 9, 4, 1 */ {0, 1, 0, 2, 1, 0, 0, 0},
        /*  26:  ,1, ,3,4, , , :  8, 4, 7, 3, 9, 0, 9, 3,11, 9,11, 1, 2, 1,11 */ {0, 1, 0, 1, 2, 0, 0, 0},
    },
    {
        /*  74:  ,1, ,3, , ,6, :  2, 3,11, 0, 5, 9, 5, 0, 6, 1, 6, 0, 6, 1,10 */ {0, 1, 0, 2, 0, 0, 1, 0},
        /*  74:  ,1, ,3, , ,6, :  0, 1, 9,10, 2, 5, 3, 5, 2, 5, 3, 6,11, 6, 3 */ {0, 1, 0, 2, 0, 0, 2, 0},
        /*  74:  ,1, ,3, , ,6, :  6, 5,10, 1,11, 2,11, 1, 9,11, 9, 3, 0, 3, 9 */ {0, 1, 0, 1, 0, 0, 2, 0},
    },
    {
        /*  82:  ,1, , ,4, ,6, :  6, 5,10, 8, 0, 7, 1, 7, 0, 7, 1, 4, 9, 4, 1 */ {0, 1, 0, 0, 1, 0, 2, 0},
        /*  82:  ,1, , ,4, ,6, :  8, 4, 7, 0, 5, 9, 5, 0, 6, 1, 6, 0, 6, 1,10 */ {0, 1, 0, 0, 2, 0, 1, 0},
        /*  82:  ,1, , ,4, ,6, :  0, 1, 9, 5,10, 4, 8, 4,10, 6, 8,10, 8, 6, 7 */ {0, 1, 0, 0, 2, 0, 2, 0},
    },
    {
        /* 164:  , ,2, , ,5, ,7: 11, 7, 6, 9, 1, 4, 2, 4, 1, 4, 2, 5,10, 5, 2 */ {0, 0, 1, 0, 0, 1, 0, 2},
        /* 164:  , ,2, , ,5, ,7:  9, 5, 4, 1, 6,10, 6, 1, 7, 2, 7, 1, 7, 2,11 */ {0, 0, 1, 0, 0, 2, 0, 1},
        /* 164:  , ,2, , ,5, ,7:  1, 2,10, 6,11, 5, 9, 5,11, 7, 9,11, 9, 7, 4 */ {0, 0, 1, 0, 0, 2, 0, 2},
    },
    {
        /*  88:  , , ,3,4, ,6, :  8, 4, 7,10, 2, 5, 3, 5, 2, 5, 3, 6,11, 6, 3 */ {0, 0, 0, 1, 2, 0, 1, 0},
        /*  88:  , , ,3,4, ,6, :  6, 5,10, 2, 7,11, 7, 2, 4, 3, 4, 2, 4, 3, 8 */ {0, 0, 0, 1, 1, 0, 2, 0},
        /*  88:  , , ,3,4, ,6, :  2, 3,11, 7, 8, 6,10, 6, 8, 4,10, 8,10, 4, 5 */ {0, 0, 0, 1, 2, 0, 2, 0},
    },
    {
        /* 167: 0,1,2, , ,5, ,7:  7, 4, 8, 5, 2,10, 2, 5, 3, 6, 3, 5, 3, 6,11 */ {0, 0, 0, 1, 2, 0, 1, 0},
        /* 167: 0,1,2, , ,5, ,7: 10, 5, 6,11, 7, 2, 4, 2, 7, 2, 4, 3, 8, 3, 4 */ {0, 0, 0, 1, 1, 0, 2, 0},
        /* 167: 0,1,2, , ,5, ,7: 11, 3, 2, 6, 8, 7, 8, 6,10, 8,10, 4, 5, 4,10 */ {0, 0, 0, 1, 2, 0, 2, 0},
    },
    {
        /*  91: 0,1, ,3,4, ,6, :  6, 7,11, 4, 1, 9, 1, 4, 2, 5, 2, 4, 2, 5,10 */ {0, 0, 1, 0, 0, 1, 0, 2},
        /*  91: 0,1, ,3,4, ,6, :  4, 5, 9,10, 6, 1, 7, 1, 6, 1, 7, 2,11, 2, 7 */ {0, 0, 1, 0, 0, 2, 0, 1},
        /*  91: 0,1, ,3,4, ,6, : 10, 2, 1, 5,11, 6,11, 5, 9,11, 9, 7, 4, 7, 9 */ {0, 0, 1, 0, 0, 2, 0, 2},
    },
    {
        /* 173: 0, ,2,3, ,5, ,7: 10, 5, 6, 7, 0, 8, 0, 7, 1, 4, 1, 7, 1, 4, 9 */ {0, 1, 0, 0, 1, 0, 2, 0},
        /* 173: 0, ,2,3, ,5, ,7:  7, 4, 8, 9, 5, 0, 6, 0, 5, 0, 6, 1,10, 1, 6 */ {0, 1, 0, 0, 2, 0, 1, 0},
        /* 173: 0, ,2,3, ,5, ,7:  9, 1, 0, 4,10, 5,10, 4, 8,10, 8, 6, 7, 6, 8 */ {0, 1, 0, 0, 2, 0, 2, 0},
    },
    {
        /* 181: 0, ,2, ,4,5, ,7: 11, 3, 2, 9, 5, 0, 6, 0, 5, 0, 6, 1,10, 1, 6 */ {0, 1, 0, 2, 0, 0, 1, 0},
        /* 181: 0, ,2, ,4,5, ,7:  9, 1, 0, 5, 2,10, 2, 5, 3, 6, 3, 5, 3, 6,11 */ {0, 1, 0, 2, 0, 0, 2, 0},
        /* 181: 0, ,2, ,4,5, ,7: 10, 5, 6, 2,11, 1, 9, 1,11, 3, 9,11, 9, 3, 0 */ {0, 1, 0, 1, 0, 0, 2, 0},
    },
    {
        /* 229: 0, ,2, , ,5,6,7:  9, 1, 0,11, 7, 2, 4, 2, 7, 2, 4, 3, 8, 3, 4 */ {0, 1, 0, 2, 2, 0, 0, 0},
        /* 229: 0, ,2, , ,5,6,7: 11, 3, 2, 7, 0, 8, 0, 7, 1, 4, 1, 7, 1, 4, 9 */ {0, 1, 0, 2, 1, 0, 0, 0},
        /* 229: 0, ,2, , ,5,6,7:  7, 4, 8, 0, 9, 3,11, 3, 9, 1,11, 9,11, 1, 2 */ {0, 1, 0, 1, 2, 0, 0, 0},
    },
    {
        /*  94:  ,1,2,3,4, ,6, :  4, 5, 9, 6, 3,11, 3, 6, 0, 7, 0, 6, 0, 7, 8 */ {0, 1, 1, 1, 1, 2, 1, 0},
        /*  94:  ,1,2,3,4, ,6, :  6, 7,11, 8, 4, 3, 5, 3, 4, 3, 5, 0, 9, 0, 5 */ {0, 1, 1, 1, 1, 0, 1, 2},
        /*  94:  ,1,2,3,4, ,6, :  8, 0, 3, 7, 9, 4, 9, 7,11, 9,11, 5, 6, 5,11 */ {0, 1, 1, 1, 1, 2, 1, 2},
    },
    {
        /* 122:  ,1, ,3,4,5,6, :  8, 0, 3,10, 6, 1, 7, 1, 6, 1, 7, 2,11, 2, 7 */ {0, 1, 2, 1, 1, 1, 1, 2},
        /* 122:  ,1, ,3,4,5,6, : 10, 2, 1, 6, 3,11, 3, 6, 0, 7, 0, 6, 0, 7, 8 */ {0, 1, 2, 1, 1, 1, 1, 0},
        /* 122:  ,1, ,3,4,5,6, :  6, 7,11, 3, 8, 2,10, 2, 8, 0,10, 8,10, 0, 1 */ {0, 1, 0, 1, 1, 1, 1, 2},
    },
    {
        /* 218:  ,1, ,3,4, ,6,7: 10, 2, 1, 8, 4, 3, 5, 3, 4, 3, 5, 0, 9, 0, 5 */ {0, 1, 2, 1, 1, 0, 1, 1},
        /* 218:  ,1, ,3,4, ,6,7:  8, 0, 3, 4, 1, 9, 1, 4, 2, 5, 2, 4, 2, 5,10 */ {0, 1, 2, 1, 1, 2, 1, 1},
        /* 218:  ,1, ,3,4, ,6,7:  4, 5, 9, 1,10, 0, 8, 0,10, 2, 8,10, 8, 2, 3 */ {0, 1, 0, 1, 1, 2, 1, 1},
    },
};
static const signed char triangleRegion7_2[16][3][10] = {
    {
        /*  37: 0, ,2, , ,5, , :  1, 2,10, 3, 4, 8, 4, 3, 5, 0, 5, 3, 5, 0, 9 */ {2, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /*  37: 0, ,2, , ,5, , :  3, 0, 8, 9, 1, 4, 2, 4, 1, 4, 2, 5,10, 5, 2 */ {0, 1, 2, 1, 2, 1, 2, 1, 2, 1},
        /*  37: 0, ,2, , ,5, , :  9, 5, 4, 0,10, 1,10, 0, 8,10, 8, 2, 3, 2, 8 */ {2, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /* 133: 0, ,2, , , , ,7:  3, 0, 8, 1, 6,10, 6, 1, 7, 2, 7, 1, 7, 2,11 */ {0, 1, 2, 1, 2, 1, 2, 1, 2, 1},
        /* 133: 0, ,2, , , , ,7:  1, 2,10,11, 3, 6, 0, 6, 3, 6, 0, 7, 8, 7, 0 */ {2, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 133: 0, ,2, , , , ,7: 11, 7, 6, 2, 8, 3, 8, 2,10, 8,10, 0, 1, 0,10 */ {2, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /* 161: 0, , , , ,5, ,7:  9, 5, 4,11, 3, 6, 0, 6, 3, 6, 0, 7, 8, 7, 0 */ {2, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 161: 0, , , , ,5, ,7: 11, 7, 6, 3, 4, 8, 4, 3, 5, 0, 5, 3, 5, 0, 9 */ {2, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 161: 0, , , , ,5, ,7:  3, 0, 8, 4, 9, 7,11, 7, 9, 5,11, 9,11, 5, 6 */ {0, 1, 2, 1, 2, 1, 2, 1, 2, 1},
    },
    {
        /*  26:  ,1, ,3,4, , , :  0, 1, 9, 2, 7,11, 7, 2, 4, 3, 4, 2, 4, 3, 8 */ {1, 0, 2, 0, 2, 0, 2, 0, 2, 0},
        /*  26:  ,1, ,3,4, , , :  2, 3,11, 8, 0, 7, 1, 7, 0, 7, 1, 4, 9, 4, 1 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  26:  ,1, ,3,4, , , :  8, 4, 7, 3, 9, 0, 9, 3,11, 9,11, 1, 2, 1,11 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /*  74:  ,1, ,3, , ,6, :  2, 3,11, 0, 5, 9, 5, 0, 6, 1, 6, 0, 6, 1,10 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  74:  ,1, ,3, , ,6, :  0, 1, 9,10, 2, 5, 3, 5, 2, 5, 3, 6,11, 6, 3 */ {1, 0, 2, 0, 2, 0, 2, 0, 2, 0},
        /*  74:  ,1, ,3, , ,6, :  6, 5,10, 1,11, 2,11, 1, 9,11, 9, 3, 0, 3, 9 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /*  82:  ,1, , ,4, ,6, :  6, 5,10, 8, 0, 7, 1, 7, 0, 7, 1, 4, 9, 4, 1 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  82:  ,1, , ,4, ,6, :  8, 4, 7, 0, 5, 9, 5, 0, 6, 1, 6, 0, 6, 1,10 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  82:  ,1, , ,4, ,6, :  0, 1, 9, 5,10, 4, 8, 4,10, 6, 8,10, 8, 6, 7 */ {1, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    },
    {
        /* 164:  , ,2, , ,5, ,7: 11, 7, 6, 9, 1, 4, 2, 4, 1, 4, 2, 5,10, 5, 2 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 164:  , ,2, , ,5, ,7:  9, 5, 4, 1, 6,10, 6, 1, 7, 2, 7, 1, 7, 2,11 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 164:  , ,2, , ,5, ,7:  1, 2,10, 6,11, 5, 9, 5,11, 7, 9,11, 9, 7, 4 */ {1, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    },
    {
        /*  88:  , , ,3,4, ,6, :  8, 4, 7,10, 2, 5, 3, 5, 2, 5, 3, 6,11, 6, 3 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  88:  , , ,3,4, ,6, :  6, 5,10, 2, 7,11, 7, 2, 4, 3, 4, 2, 4, 3, 8 */ {2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  88:  , , ,3,4, ,6, :  2, 3,11, 7, 8, 6,10, 6, 8, 4,10, 8,10, 4, 5 */ {1, 0, 2, 0, 2, 0, 2, 0, 2, 0},
    },
    {
        /* 167: 0,1,2, , ,5, ,7:  7, 4, 8, 5, 2,10, 2, 5, 3, 6, 3, 5, 3, 6,11 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 167: 0,1,2, , ,5, ,7: 10, 5, 6,11, 7, 2, 4, 2, 7, 2, 4, 3, 8, 3, 4 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 167: 0,1,2, , ,5, ,7: 11, 3, 2, 6, 8, 7, 8, 6,10, 8,10, 4, 5, 4,10 */ {0, 1, 0, 2, 0, 2, 0, 2, 0, 2},
    },
    {
        /*  91: 0,1, ,3,4, ,6, :  6, 7,11, 4, 1, 9, 1, 4, 2, 5, 2, 4, 2, 5,10 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /*  91: 0,1, ,3,4, ,6, :  4, 5, 9,10, 6, 1, 7, 1, 6, 1, 7, 2,11, 2, 7 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /*  91: 0,1, ,3,4, ,6, : 10, 2, 1, 5,11, 6,11, 5, 9,11, 9, 7, 4, 7, 9 */ {0, 1, 0, 2, 0, 2, 0, 2, 0, 2},
    },
    {
        /* 173: 0, ,2,3, ,5, ,7: 10, 5, 6, 7, 0, 8, 0, 7, 1, 4, 1, 7, 1, 4, 9 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 173: 0, ,2,3, ,5, ,7:  7, 4, 8, 9, 5, 0, 6, 0, 5, 0, 6, 1,10, 1, 6 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 173: 0, ,2,3, ,5, ,7:  9, 1, 0, 4,10, 5,10, 4, 8,10, 8, 6, 7, 6, 8 */ {0, 1, 0, 2, 0, 2, 0, 2, 0, 2},
    },
    {
        /* 181: 0, ,2, ,4,5, ,7: 11, 3, 2, 9, 5, 0, 6, 0, 5, 0, 6, 1,10, 1, 6 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 181: 0, ,2, ,4,5, ,7:  9, 1, 0, 5, 2,10, 2, 5, 3, 6, 3, 5, 3, 6,11 */ {0, 1, 0, 2, 0, 2, 0, 2, 0, 2},
        /* 181: 0, ,2, ,4,5, ,7: 10, 5, 6, 2,11, 1, 9, 1,11, 3, 9,11, 9, 3, 0 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /* 229: 0, ,2, , ,5,6,7:  9, 1, 0,11, 7, 2, 4, 2, 7, 2, 4, 3, 8, 3, 4 */ {0, 1, 0, 2, 0, 2, 0, 2, 0, 2},
        /* 229: 0, ,2, , ,5,6,7: 11, 3, 2, 7, 0, 8, 0, 7, 1, 4, 1, 7, 1, 4, 9 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 229: 0, ,2, , ,5,6,7:  7, 4, 8, 0, 9, 3,11, 3, 9, 1,11, 9,11, 1, 2 */ {0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /*  94:  ,1,2,3,4, ,6, :  4, 5, 9, 6, 3,11, 3, 6, 0, 7, 0, 6, 0, 7, 8 */ {1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  94:  ,1,2,3,4, ,6, :  6, 7,11, 8, 4, 3, 5, 3, 4, 3, 5, 0, 9, 0, 5 */ {1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  94:  ,1,2,3,4, ,6, :  8, 0, 3, 7, 9, 4, 9, 7,11, 9,11, 5, 6, 5,11 */ {1, 0, 1, 2, 1, 2, 1, 2, 1, 2},
    },
    {
        /* 122:  ,1, ,3,4,5,6, :  8, 0, 3,10, 6, 1, 7, 1, 6, 1, 7, 2,11, 2, 7 */ {1, 0, 1, 2, 1, 2, 1, 2, 1, 2},
        /* 122:  ,1, ,3,4,5,6, : 10, 2, 1, 6, 3,11, 3, 6, 0, 7, 0, 6, 0, 7, 8 */ {1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 122:  ,1, ,3,4,5,6, :  6, 7,11, 3, 8, 2,10, 2, 8, 0,10, 8,10, 0, 1 */ {1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /* 218:  ,1, ,3,4, ,6,7: 10, 2, 1, 8, 4, 3, 5, 3, 4, 3, 5, 0, 9, 0, 5 */ {1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 218:  ,1, ,3,4, ,6,7:  8, 0, 3, 4, 1, 9, 1, 4, 2, 5, 2, 4, 2, 5,10 */ {1, 0, 1, 2, 1, 2, 1, 2, 1, 2},
        /* 218:  ,1, ,3,4, ,6,7:  4, 5, 9, 1,10, 0, 8, 0,10, 2, 8,10, 8, 2, 3 */ {1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion7_3[16][3][8] = {
    {
        /*  37: 0, ,2, , ,5, , : 12, 2,10,12,10, 5,12, 5, 4,12, 4, 8,12, 8, 3,12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2 */
        {0, 1, 0, 1, 1, 0, 1, 1},
        /*  37: 0, ,2, , ,5, , : 12, 5, 4,12, 4, 8,12, 8, 3,12, 3, 2,12, 2,10,12,10, 1,12, 1, 0,12, 0, 9,12, 9, 5 */
        {0, 1, 0, 1, 1, 0, 1, 1},
        /*  37: 0, ,2, , ,5, , :  5, 4,12,10, 5,12, 2,10,12, 3, 2,12, 8, 3,12, 0, 8,12, 1, 0,12, 9, 1,12, 4, 9,12 */
        {0, 1, 0, 1, 1, 0, 1, 1},
    },
    {
        /* 133: 0, ,2, , , , ,7: 12, 0, 8,12, 8, 7,12, 7, 6,12, 6,10,12,10, 1,12, 1, 2,12, 2,11,12,11, 3,12, 3, 0 */
        {0, 1, 0, 1, 1, 1, 1, 0},
        /* 133: 0, ,2, , , , ,7: 12, 7, 6,12, 6,10,12,10, 1,12, 1, 0,12, 0, 8,12, 8, 3,12, 3, 2,12, 2,11,12,11, 7 */
        {0, 1, 0, 1, 1, 1, 1, 0},
        /* 133: 0, ,2, , , , ,7:  7, 6,12, 8, 7,12, 0, 8,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12,11, 3,12, 6,11,12 */
        {0, 1, 0, 1, 1, 1, 1, 0},
    },
    {
        /* 161: 0, , , , ,5, ,7:  9, 5,12, 0, 9,12, 3, 0,12,11, 3,12, 6,11,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12 */
        {0, 1, 1, 1, 1, 0, 1, 0},
        /* 161: 0, , , , ,5, ,7:  3, 0,12,11, 3,12, 6,11,12, 5, 6,12, 9, 5,12, 4, 9,12, 7, 4,12, 8, 7,12, 0, 8,12 */
        {0, 1, 1, 1, 1, 0, 1, 0},
        /* 161: 0, , , , ,5, ,7: 12, 3, 0,12, 0, 9,12, 9, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3 */
        {0, 1, 1, 1, 1, 0, 1, 0},
    },
    {
        /*  26:  ,1, ,3,4, , , : 12, 1, 9,12, 9, 4,12, 4, 7,12, 7,11,12,11, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1 */
        {0, 1, 0, 1, 1, 0, 0, 0},
        /*  26:  ,1, ,3,4, , , : 12, 4, 7,12, 7,11,12,11, 2,12, 2, 1,12, 1, 9,12, 9, 0,12, 0, 3,12, 3, 8,12, 8, 4 */
        {0, 1, 0, 1, 1, 0, 0, 0},
        /*  26:  ,1, ,3,4, , , :  4, 7,12, 9, 4,12, 1, 9,12, 2, 1,12,11, 2,12, 3,11,12, 0, 3,12, 8, 0,12, 7, 8,12 */
        {0, 1, 0, 1, 1, 0, 0, 0},
    },
    {
        /*  74:  ,1, ,3, , ,6, : 12, 3,11,12,11, 6,12, 6, 5,12, 5, 9,12, 9, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3 */
        {0, 1, 0, 1, 0, 0, 1, 0},
        /*  74:  ,1, ,3, , ,6, : 12, 6, 5,12, 5, 9,12, 9, 0,12, 0, 3,12, 3,11,12,11, 2,12, 2, 1,12, 1,10,12,10, 6 */
        {0, 1, 0, 1, 0, 0, 1, 0},
        /*  74:  ,1, ,3, , ,6, :  6, 5,12,11, 6,12, 3,11,12, 0, 3,12, 9, 0,12, 1, 9,12, 2, 1,12,10, 2,12, 5,10,12 */
        {0, 1, 0, 1, 0, 0, 1, 0},
    },
    {
        /*  82:  ,1, , ,4, ,6, : 10, 6,12, 1,10,12, 0, 1,12, 8, 0,12, 7, 8,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12 */
        {0, 1, 0, 0, 1, 0, 1, 0},
        /*  82:  ,1, , ,4, ,6, :  0, 1,12, 8, 0,12, 7, 8,12, 6, 7,12,10, 6,12, 5,10,12, 4, 5,12, 9, 4,12, 1, 9,12 */
        {0, 1, 0, 0, 1, 0, 1, 0},
        /*  82:  ,1, , ,4, ,6, : 12, 0, 1,12, 1,10,12,10, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 9,12, 9, 0 */
        {0, 1, 0, 0, 1, 0, 1, 0},
    },
    {
        /* 164:  , ,2, , ,5, ,7: 11, 7,12, 2,11,12, 1, 2,12, 9, 1,12, 4, 9,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12 */
        {0, 0, 1, 0, 0, 1, 0, 1},
        /* 164:  , ,2, , ,5, ,7:  1, 2,12, 9, 1,12, 4, 9,12, 7, 4,12,11, 7,12, 6,11,12, 5, 6,12,10, 5,12, 2,10,12 */
        {0, 0, 1, 0, 0, 1, 0, 1},
        /* 164:  , ,2, , ,5, ,7: 12, 1, 2,12, 2,11,12,11, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6,10,12,10, 1 */
        {0, 0, 1, 0, 0, 1, 0, 1},
    },
    {
        /*  88:  , , ,3,4, ,6, :  8, 4,12, 3, 8,12, 2, 3,12,10, 2,12, 5,10,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12 */
        {0, 0, 0, 1, 1, 0, 1, 0},
        /*  88:  , , ,3,4, ,6, :  2, 3,12,10, 2,12, 5,10,12, 4, 5,12, 8, 4,12, 7, 8,12, 6, 7,12,11, 6,12, 3,11,12 */
        {0, 0, 0, 1, 1, 0, 1, 0},
        /*  88:  , , ,3,4, ,6, : 12, 2, 3,12, 3, 8,12, 8, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7,11,12,11, 2 */
        {0, 0, 0, 1, 1, 0, 1, 0},
    },
    {
        /* 167: 0,1,2, , ,5, ,7: 12, 4, 8,12, 8, 3,12, 3, 2,12, 2,10,12,10, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4 */
        {0, 0, 0, 1, 1, 0, 1, 0},
        /* 167: 0,1,2, , ,5, ,7: 12, 3, 2,12, 2,10,12,10, 5,12, 5, 4,12, 4, 8,12, 8, 7,12, 7, 6,12, 6,11,12,11, 3 */
        {0, 0, 0, 1, 1, 0, 1, 0},
        /* 167: 0,1,2, , ,5, ,7:  3, 2,12, 8, 3,12, 4, 8,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12,11, 7,12, 2,11,12 */
        {0, 0, 0, 1, 1, 0, 1, 0},
    },
    {
        /*  91: 0,1, ,3,4, ,6, : 12, 7,11,12,11, 2,12, 2, 1,12, 1, 9,12, 9, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7 */
        {0, 0, 1, 0, 0, 1, 0, 1},
        /*  91: 0,1, ,3,4, ,6, : 12, 2, 1,12, 1, 9,12, 9, 4,12, 4, 7,12, 7,11,12,11, 6,12, 6, 5,12, 5,10,12,10, 2 */
        {0, 0, 1, 0, 0, 1, 0, 1},
        /*  91: 0,1, ,3,4, ,6, :  2, 1,12,11, 2,12, 7,11,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12,10, 6,12, 1,10,12 */
        {0, 0, 1, 0, 0, 1, 0, 1},
    },
    {
        /* 173: 0, ,2,3, ,5, ,7: 12, 6,10,12,10, 1,12, 1, 0,12, 0, 8,12, 8, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6 */
        {0, 1, 0, 0, 1, 0, 1, 0},
        /* 173: 0, ,2,3, ,5, ,7: 12, 1, 0,12, 0, 8,12, 8, 7,12, 7, 6,12, 6,10,12,10, 5,12, 5, 4,12, 4, 9,12, 9, 1 */
        {0, 1, 0, 0, 1, 0, 1, 0},
        /* 173: 0, ,2,3, ,5, ,7:  1, 0,12,10, 1,12, 6,10,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 9, 5,12, 0, 9,12 */
        {0, 1, 0, 0, 1, 0, 1, 0},
    },
    {
        /* 181: 0, ,2, ,4,5, ,7: 11, 3,12, 6,11,12, 5, 6,12, 9, 5,12, 0, 9,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12 */
        {0, 1, 0, 1, 0, 0, 1, 0},
        /* 181: 0, ,2, ,4,5, ,7:  5, 6,12, 9, 5,12, 0, 9,12, 3, 0,12,11, 3,12, 2,11,12, 1, 2,12,10, 1,12, 6,10,12 */
        {0, 1, 0, 1, 0, 0, 1, 0},
        /* 181: 0, ,2, ,4,5, ,7: 12, 5, 6,12, 6,11,12,11, 3,12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2,12, 2,10,12,10, 5 */
        {0, 1, 0, 1, 0, 0, 1, 0},
    },
    {
        /* 229: 0, ,2, , ,5,6,7:  9, 1,12, 4, 9,12, 7, 4,12,11, 7,12, 2,11,12, 3, 2,12, 8, 3,12, 0, 8,12, 1, 0,12 */
        {0, 1, 0, 1, 1, 0, 0, 0},
        /* 229: 0, ,2, , ,5,6,7:  7, 4,12,11, 7,12, 2,11,12, 1, 2,12, 9, 1,12, 0, 9,12, 3, 0,12, 8, 3,12, 4, 8,12 */
        {0, 1, 0, 1, 1, 0, 0, 0},
        /* 229: 0, ,2, , ,5,6,7: 12, 7, 4,12, 4, 9,12, 9, 1,12, 1, 2,12, 2,11,12,11, 3,12, 3, 0,12, 0, 8,12, 8, 7 */
        {0, 1, 0, 1, 1, 0, 0, 0},
    },
    {
        /*  94:  ,1,2,3,4, ,6, : 12, 5, 9,12, 9, 0,12, 0, 3,12, 3,11,12,11, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5 */
        {0, 1, 1, 1, 1, 0, 1, 0},
        /*  94:  ,1,2,3,4, ,6, : 12, 0, 3,12, 3,11,12,11, 6,12, 6, 5,12, 5, 9,12, 9, 4,12, 4, 7,12, 7, 8,12, 8, 0 */
        {0, 1, 1, 1, 1, 0, 1, 0},
        /*  94:  ,1,2,3,4, ,6, :  0, 3,12, 9, 0,12, 5, 9,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12 */
        {0, 1, 1, 1, 1, 0, 1, 0},
    },
    {
        /* 122:  ,1, ,3,4,5,6, :  8, 0,12, 7, 8,12, 6, 7,12,10, 6,12, 1,10,12, 2, 1,12,11, 2,12, 3,11,12, 0, 3,12 */
        {0, 1, 0, 1, 1, 1, 1, 0},
        /* 122:  ,1, ,3,4,5,6, :  6, 7,12,10, 6,12, 1,10,12, 0, 1,12, 8, 0,12, 3, 8,12, 2, 3,12,11, 2,12, 7,11,12 */
        {0, 1, 0, 1, 1, 1, 1, 0},
        /* 122:  ,1, ,3,4,5,6, : 12, 6, 7,12, 7, 8,12, 8, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3,12, 3,11,12,11, 6 */
        {0, 1, 0, 1, 1, 1, 1, 0},
    },
    {
        /* 218:  ,1, ,3,4, ,6,7: 10, 2,12, 5,10,12, 4, 5,12, 8, 4,12, 3, 8,12, 0, 3,12, 9, 0,12, 1, 9,12, 2, 1,12 */
        {0, 1, 0, 1, 1, 0, 1, 1},
        /* 218:  ,1, ,3,4, ,6,7:  4, 5,12, 8, 4,12, 3, 8,12, 2, 3,12,10, 2,12, 1,10,12, 0, 1,12, 9, 0,12, 5, 9,12 */
        {0, 1, 0, 1, 1, 0, 1, 1},
        /* 218:  ,1, ,3,4, ,6,7: 12, 4, 5,12, 5,10,12,10, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1,12, 1, 9,12, 9, 4 */
        {0, 1, 0, 1, 1, 0, 1, 1},
    },
};
static const signed char triangleRegion7_3[16][3][18] = {
    {
        /*  37: 0, ,2, , ,5, , : 12, 2,10,12,10, 5,12, 5, 4,12, 4, 8,12, 8, 3,12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /*  37: 0, ,2, , ,5, , : 12, 5, 4,12, 4, 8,12, 8, 3,12, 3, 2,12, 2,10,12,10, 1,12, 1, 0,12, 0, 9,12, 9, 5 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /*  37: 0, ,2, , ,5, , :  5, 4,12,10, 5,12, 2,10,12, 3, 2,12, 8, 3,12, 0, 8,12, 1, 0,12, 9, 1,12, 4, 9,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /* 133: 0, ,2, , , , ,7: 12, 0, 8,12, 8, 7,12, 7, 6,12, 6,10,12,10, 1,12, 1, 2,12, 2,11,12,11, 3,12, 3, 0 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 133: 0, ,2, , , , ,7: 12, 7, 6,12, 6,10,12,10, 1,12, 1, 0,12, 0, 8,12, 8, 3,12, 3, 2,12, 2,11,12,11, 7 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 133: 0, ,2, , , , ,7:  7, 6,12, 8, 7,12, 0, 8,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12,11, 3,12, 6,11,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /* 161: 0, , , , ,5, ,7:  9, 5,12, 0, 9,12, 3, 0,12,11, 3,12, 6,11,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 161: 0, , , , ,5, ,7:  3, 0,12,11, 3,12, 6,11,12, 5, 6,12, 9, 5,12, 4, 9,12, 7, 4,12, 8, 7,12, 0, 8,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 161: 0, , , , ,5, ,7: 12, 3, 0,12, 0, 9,12, 9, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /*  26:  ,1, ,3,4, , , : 12, 1, 9,12, 9, 4,12, 4, 7,12, 7,11,12,11, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  26:  ,1, ,3,4, , , : 12, 4, 7,12, 7,11,12,11, 2,12, 2, 1,12, 1, 9,12, 9, 0,12, 0, 3,12, 3, 8,12, 8, 4 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  26:  ,1, ,3,4, , , :  4, 7,12, 9, 4,12, 1, 9,12, 2, 1,12,11, 2,12, 3,11,12, 0, 3,12, 8, 0,12, 7, 8,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /*  74:  ,1, ,3, , ,6, : 12, 3,11,12,11, 6,12, 6, 5,12, 5, 9,12, 9, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  74:  ,1, ,3, , ,6, : 12, 6, 5,12, 5, 9,12, 9, 0,12, 0, 3,12, 3,11,12,11, 2,12, 2, 1,12, 1,10,12,10, 6 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  74:  ,1, ,3, , ,6, :  6, 5,12,11, 6,12, 3,11,12, 0, 3,12, 9, 0,12, 1, 9,12, 2, 1,12,10, 2,12, 5,10,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /*  82:  ,1, , ,4, ,6, : 10, 6,12, 1,10,12, 0, 1,12, 8, 0,12, 7, 8,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  82:  ,1, , ,4, ,6, :  0, 1,12, 8, 0,12, 7, 8,12, 6, 7,12,10, 6,12, 5,10,12, 4, 5,12, 9, 4,12, 1, 9,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  82:  ,1, , ,4, ,6, : 12, 0, 1,12, 1,10,12,10, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 9,12, 9, 0 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /* 164:  , ,2, , ,5, ,7: 11, 7,12, 2,11,12, 1, 2,12, 9, 1,12, 4, 9,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 164:  , ,2, , ,5, ,7:  1, 2,12, 9, 1,12, 4, 9,12, 7, 4,12,11, 7,12, 6,11,12, 5, 6,12,10, 5,12, 2,10,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 164:  , ,2, , ,5, ,7: 12, 1, 2,12, 2,11,12,11, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6,10,12,10, 1 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /*  88:  , , ,3,4, ,6, :  8, 4,12, 3, 8,12, 2, 3,12,10, 2,12, 5,10,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  88:  , , ,3,4, ,6, :  2, 3,12,10, 2,12, 5,10,12, 4, 5,12, 8, 4,12, 7, 8,12, 6, 7,12,11, 6,12, 3,11,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  88:  , , ,3,4, ,6, : 12, 2, 3,12, 3, 8,12, 8, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7,11,12,11, 2 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /* 167: 0,1,2, , ,5, ,7: 12, 4, 8,12, 8, 3,12, 3, 2,12, 2,10,12,10, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 167: 0,1,2, , ,5, ,7: 12, 3, 2,12, 2,10,12,10, 5,12, 5, 4,12, 4, 8,12, 8, 7,12, 7, 6,12, 6,11,12,11, 3 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 167: 0,1,2, , ,5, ,7:  3, 2,12, 8, 3,12, 4, 8,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12,11, 7,12, 2,11,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /*  91: 0,1, ,3,4, ,6, : 12, 7,11,12,11, 2,12, 2, 1,12, 1, 9,12, 9, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /*  91: 0,1, ,3,4, ,6, : 12, 2, 1,12, 1, 9,12, 9, 4,12, 4, 7,12, 7,11,12,11, 6,12, 6, 5,12, 5,10,12,10, 2 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /*  91: 0,1, ,3,4, ,6, :  2, 1,12,11, 2,12, 7,11,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12,10, 6,12, 1,10,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /* 173: 0, ,2,3, ,5, ,7: 12, 6,10,12,10, 1,12, 1, 0,12, 0, 8,12, 8, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 173: 0, ,2,3, ,5, ,7: 12, 1, 0,12, 0, 8,12, 8, 7,12, 7, 6,12, 6,10,12,10, 5,12, 5, 4,12, 4, 9,12, 9, 1 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 173: 0, ,2,3, ,5, ,7:  1, 0,12,10, 1,12, 6,10,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 9, 5,12, 0, 9,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /* 181: 0, ,2, ,4,5, ,7: 11, 3,12, 6,11,12, 5, 6,12, 9, 5,12, 0, 9,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 181: 0, ,2, ,4,5, ,7:  5, 6,12, 9, 5,12, 0, 9,12, 3, 0,12,11, 3,12, 2,11,12, 1, 2,12,10, 1,12, 6,10,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 181: 0, ,2, ,4,5, ,7: 12, 5, 6,12, 6,11,12,11, 3,12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2,12, 2,10,12,10, 5 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /* 229: 0, ,2, , ,5,6,7:  9, 1,12, 4, 9,12, 7, 4,12,11, 7,12, 2,11,12, 3, 2,12, 8, 3,12, 0, 8,12, 1, 0,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 229: 0, ,2, , ,5,6,7:  7, 4,12,11, 7,12, 2,11,12, 1, 2,12, 9, 1,12, 0, 9,12, 3, 0,12, 8, 3,12, 4, 8,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 229: 0, ,2, , ,5,6,7: 12, 7, 4,12, 4, 9,12, 9, 1,12, 1, 2,12, 2,11,12,11, 3,12, 3, 0,12, 0, 8,12, 8, 7 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /*  94:  ,1,2,3,4, ,6, : 12, 5, 9,12, 9, 0,12, 0, 3,12, 3,11,12,11, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  94:  ,1,2,3,4, ,6, : 12, 0, 3,12, 3,11,12,11, 6,12, 6, 5,12, 5, 9,12, 9, 4,12, 4, 7,12, 7, 8,12, 8, 0 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  94:  ,1,2,3,4, ,6, :  0, 3,12, 9, 0,12, 5, 9,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /* 122:  ,1, ,3,4,5,6, :  8, 0,12, 7, 8,12, 6, 7,12,10, 6,12, 1,10,12, 2, 1,12,11, 2,12, 3,11,12, 0, 3,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 122:  ,1, ,3,4,5,6, :  6, 7,12,10, 6,12, 1,10,12, 0, 1,12, 8, 0,12, 3, 8,12, 2, 3,12,11, 2,12, 7,11,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 122:  ,1, ,3,4,5,6, : 12, 6, 7,12, 7, 8,12, 8, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3,12, 3,11,12,11, 6 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
    {
        /* 218:  ,1, ,3,4, ,6,7: 10, 2,12, 5,10,12, 4, 5,12, 8, 4,12, 3, 8,12, 0, 3,12, 9, 0,12, 1, 9,12, 2, 1,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 218:  ,1, ,3,4, ,6,7:  4, 5,12, 8, 4,12, 3, 8,12, 2, 3,12,10, 2,12, 1,10,12, 0, 1,12, 9, 0,12, 5, 9,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /* 218:  ,1, ,3,4, ,6,7: 12, 4, 5,12, 5,10,12,10, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1,12, 1, 9,12, 9, 4 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion7_4_1[16][8] = {
    /*  37: 0, ,2, , ,5, , :  3, 4, 8, 4, 3,10, 2,10, 3, 4,10, 5, 9, 1, 0 */ {0, 1, 0, 2, 2, 0, 2, 2},
    /* 133: 0, ,2, , , , ,7:  1, 6,10, 6, 1, 8, 0, 8, 1, 6, 8, 7,11, 3, 2 */ {0, 1, 0, 2, 1, 1, 1, 0},
    /* 161: 0, , , , ,5, ,7: 11, 3, 6, 9, 6, 3, 6, 9, 5, 0, 9, 3, 7, 4, 8 */ {0, 1, 1, 1, 2, 0, 1, 0},
    /*  26:  ,1, ,3,4, , , :  2, 7,11, 7, 2, 9, 1, 9, 2, 7, 9, 4, 8, 0, 3 */ {0, 1, 2, 1, 1, 2, 2, 2},
    /*  74:  ,1, ,3, , ,6, :  0, 5, 9, 5, 0,11, 3,11, 0, 5,11, 6,10, 2, 1 */ {0, 1, 2, 1, 0, 0, 1, 0},
    /*  82:  ,1, , ,4, ,6, :  8, 0, 7,10, 7, 0, 7,10, 6, 1,10, 0, 4, 5, 9 */ {0, 1, 0, 0, 1, 2, 1, 0},
    /* 164:  , ,2, , ,5, ,7:  9, 1, 4,11, 4, 1, 4,11, 7, 2,11, 1, 5, 6,10 */ {0, 0, 1, 0, 0, 1, 2, 1},
    /*  88:  , , ,3,4, ,6, : 10, 2, 5, 8, 5, 2, 5, 8, 4, 3, 8, 2, 6, 7,11 */ {0, 0, 0, 1, 1, 0, 1, 2},
    /* 167: 0,1,2, , ,5, ,7:  5, 2,10, 2, 5, 8, 4, 8, 5, 2, 8, 3,11, 7, 6 */ {0, 0, 0, 1, 1, 0, 1, 2},
    /*  91: 0,1, ,3,4, ,6, :  4, 1, 9, 1, 4,11, 7,11, 4, 1,11, 2,10, 6, 5 */ {0, 0, 1, 0, 0, 1, 2, 1},
    /* 173: 0, ,2,3, ,5, ,7:  7, 0, 8, 0, 7,10, 6,10, 7, 0,10, 1, 9, 5, 4 */ {0, 1, 0, 0, 1, 2, 1, 0},
    /* 181: 0, ,2, ,4,5, ,7:  9, 5, 0,11, 0, 5, 0,11, 3, 6,11, 5, 1, 2,10 */ {0, 1, 2, 1, 0, 0, 1, 0},
    /* 229: 0, ,2, , ,5,6,7: 11, 7, 2, 9, 2, 7, 2, 9, 1, 4, 9, 7, 3, 0, 8 */ {0, 1, 2, 1, 1, 2, 2, 2},
    /*  94:  ,1,2,3,4, ,6, :  6, 3,11, 3, 6, 9, 5, 9, 6, 3, 9, 0, 8, 4, 7 */ {0, 1, 1, 1, 2, 0, 1, 0},
    /* 122:  ,1, ,3,4,5,6, : 10, 6, 1, 8, 1, 6, 1, 8, 0, 7, 8, 6, 2, 3,11 */ {0, 1, 0, 2, 1, 1, 1, 0},
    /* 218:  ,1, ,3,4, ,6,7:  8, 4, 3,10, 3, 4, 3,10, 2, 5,10, 4, 0, 1, 9 */ {0, 1, 0, 2, 2, 0, 2, 2},
};
static const signed char triangleRegion7_4_1[16][10] = {
    /*  37: 0, ,2, , ,5, , :  3, 4, 8, 4, 3,10, 2,10, 3, 4,10, 5, 9, 1, 0 */ {0, 2, 0, 2, 0, 2, 0, 2, 0, 1},
    /* 133: 0, ,2, , , , ,7:  1, 6,10, 6, 1, 8, 0, 8, 1, 6, 8, 7,11, 3, 2 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 2},
    /* 161: 0, , , , ,5, ,7: 11, 3, 6, 9, 6, 3, 6, 9, 5, 0, 9, 3, 7, 4, 8 */ {0, 1, 0, 1, 0, 1, 0, 1, 0, 2},
    /*  26:  ,1, ,3,4, , , :  2, 7,11, 7, 2, 9, 1, 9, 2, 7, 9, 4, 8, 0, 3 */ {1, 2, 1, 2, 1, 2, 1, 2, 1, 0},
    /*  74:  ,1, ,3, , ,6, :  0, 5, 9, 5, 0,11, 3,11, 0, 5,11, 6,10, 2, 1 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 2},
    /*  82:  ,1, , ,4, ,6, :  8, 0, 7,10, 7, 0, 7,10, 6, 1,10, 0, 4, 5, 9 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 2},
    /* 164:  , ,2, , ,5, ,7:  9, 1, 4,11, 4, 1, 4,11, 7, 2,11, 1, 5, 6,10 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 2},
    /*  88:  , , ,3,4, ,6, : 10, 2, 5, 8, 5, 2, 5, 8, 4, 3, 8, 2, 6, 7,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 1, 2},
    /* 167: 0,1,2, , ,5, ,7:  5, 2,10, 2, 5, 8, 4, 8, 5, 2, 8, 3,11, 7, 6 */ {0, 1, 0, 1, 0, 1, 0, 1, 2, 1},
    /*  91: 0,1, ,3,4, ,6, :  4, 1, 9, 1, 4,11, 7,11, 4, 1,11, 2,10, 6, 5 */ {0, 1, 0, 1, 0, 1, 0, 1, 2, 1},
    /* 173: 0, ,2,3, ,5, ,7:  7, 0, 8, 0, 7,10, 6,10, 7, 0,10, 1, 9, 5, 4 */ {0, 1, 0, 1, 0, 1, 0, 1, 2, 1},
    /* 181: 0, ,2, ,4,5, ,7:  9, 5, 0,11, 0, 5, 0,11, 3, 6,11, 5, 1, 2,10 */ {0, 1, 0, 1, 0, 1, 0, 1, 2, 1},
    /* 229: 0, ,2, , ,5,6,7: 11, 7, 2, 9, 2, 7, 2, 9, 1, 4, 9, 7, 3, 0, 8 */ {2, 1, 2, 1, 2, 1, 2, 1, 0, 1},
    /*  94:  ,1,2,3,4, ,6, :  6, 3,11, 3, 6, 9, 5, 9, 6, 3, 9, 0, 8, 4, 7 */ {1, 0, 1, 0, 1, 0, 1, 0, 2, 0},
    /* 122:  ,1, ,3,4,5,6, : 10, 6, 1, 8, 1, 6, 1, 8, 0, 7, 8, 6, 2, 3,11 */ {1, 0, 1, 0, 1, 0, 1, 0, 2, 0},
    /* 218:  ,1, ,3,4, ,6,7:  8, 4, 3,10, 3, 4, 3,10, 2, 5,10, 4, 0, 1, 9 */ {2, 0, 2, 0, 2, 0, 2, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion7_4_2[16][27] = {
    /*  37: 0, ,2, , ,5, , :  9, 4, 8, 4, 9, 5,10, 5, 9, 1,10, 9,10, 1, 2, 0, 2, 1, 2, 0, 3, 8, 3, 0, 9, 8, 0 */
    {0, 1, 2, 1, 1, 3, 1, 1},
    /* 133: 0, ,2, , , , ,7: 11, 6,10, 6,11, 7, 8, 7,11, 3, 8,11, 8, 3, 0, 2, 0, 3, 0, 2, 1,10, 1, 2,11,10, 2 */
    {0, 1, 2, 1, 1, 1, 1, 3},
    /* 161: 0, , , , ,5, ,7: 11, 3, 8, 0, 8, 3, 8, 0, 9, 8, 9, 4, 5, 4, 9, 4, 5, 7, 6, 7, 5, 7, 6,11, 7,11, 8 */
    {0, 1, 1, 1, 1, 2, 1, 3},
    /*  26:  ,1, ,3,4, , , :  8, 7,11, 7, 8, 4, 9, 4, 8, 0, 9, 8, 9, 0, 1, 3, 1, 0, 1, 3, 2,11, 2, 3, 8,11, 3 */
    {0, 1, 0, 2, 3, 0, 0, 0},
    /*  74:  ,1, ,3, , ,6, : 10, 5, 9, 5,10, 6,11, 6,10, 2,11,10,11, 2, 3, 1, 3, 2, 3, 1, 0, 9, 0, 1,10, 9, 1 */
    {0, 1, 0, 2, 0, 0, 3, 0},
    /*  82:  ,1, , ,4, ,6, :  8, 0, 9, 1, 9, 0, 9, 1,10, 9,10, 5, 6, 5,10, 5, 6, 4, 7, 4, 6, 4, 7, 8, 4, 8, 9 */
    {0, 1, 0, 0, 2, 0, 3, 0},
    /* 164:  , ,2, , ,5, ,7:  9, 1,10, 2,10, 1,10, 2,11,10,11, 6, 7, 6,11, 6, 7, 5, 4, 5, 7, 5, 4, 9, 5, 9,10 */
    {0, 0, 1, 0, 0, 2, 0, 3},
    /*  88:  , , ,3,4, ,6, : 10, 2,11, 3,11, 2,11, 3, 8,11, 8, 7, 4, 7, 8, 7, 4, 6, 5, 6, 4, 6, 5,10, 6,10,11 */
    {0, 0, 0, 1, 2, 0, 3, 0},
    /* 167: 0,1,2, , ,5, ,7: 11, 2,10, 2,11, 3, 8, 3,11, 7, 8,11, 8, 7, 4, 6, 4, 7, 4, 6, 5,10, 5, 6,11,10, 6 */
    {0, 0, 0, 1, 2, 0, 3, 0},
    /*  91: 0,1, ,3,4, ,6, : 10, 1, 9, 1,10, 2,11, 2,10, 6,11,10,11, 6, 7, 5, 7, 6, 7, 5, 4, 9, 4, 5,10, 9, 5 */
    {0, 0, 1, 0, 0, 2, 0, 3},
    /* 173: 0, ,2,3, ,5, ,7:  9, 0, 8, 0, 9, 1,10, 1, 9, 5,10, 9,10, 5, 6, 4, 6, 5, 6, 4, 7, 8, 7, 4, 9, 8, 4 */
    {0, 1, 0, 0, 2, 0, 3, 0},
    /* 181: 0, ,2, ,4,5, ,7:  9, 5,10, 6,10, 5,10, 6,11,10,11, 2, 3, 2,11, 2, 3, 1, 0, 1, 3, 1, 0, 9, 1, 9,10 */
    {0, 1, 0, 2, 0, 0, 3, 0},
    /* 229: 0, ,2, , ,5,6,7: 11, 7, 8, 4, 8, 7, 8, 4, 9, 8, 9, 0, 1, 0, 9, 0, 1, 3, 2, 3, 1, 3, 2,11, 3,11, 8 */
    {0, 1, 0, 2, 3, 0, 0, 0},
    /*  94:  ,1,2,3,4, ,6, :  8, 3,11, 3, 8, 0, 9, 0, 8, 4, 9, 8, 9, 4, 5, 7, 5, 4, 5, 7, 6,11, 6, 7, 8,11, 7 */
    {0, 1, 1, 1, 1, 2, 1, 3},
    /* 122:  ,1, ,3,4,5,6, : 10, 6,11, 7,11, 6,11, 7, 8,11, 8, 3, 0, 3, 8, 3, 0, 2, 1, 2, 0, 2, 1,10, 2,10,11 */
    {0, 1, 2, 1, 1, 1, 1, 3},
    /* 218:  ,1, ,3,4, ,6,7:  8, 4, 9, 5, 9, 4, 9, 5,10, 9,10, 1, 2, 1,10, 1, 2, 0, 3, 0, 2, 0, 3, 8, 0, 8, 9 */
    {0, 1, 2, 1, 1, 3, 1, 1},
};
static const signed char triangleRegion7_4_2[16][18] = {
    /*  37: 0, ,2, , ,5, , :  9, 4, 8, 4, 9, 5,10, 5, 9, 1,10, 9,10, 1, 2, 0, 2, 1, 2, 0, 3, 8, 3, 0, 9, 8, 0 */
    {1, 1, 3, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 0, 1, 1, 1},
    /* 133: 0, ,2, , , , ,7: 11, 6,10, 6,11, 7, 8, 7,11, 3, 8,11, 8, 3, 0, 2, 0, 3, 0, 2, 1,10, 1, 2,11,10, 2 */
    {1, 1, 3, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 2, 1, 1, 1},
    /* 161: 0, , , , ,5, ,7: 11, 3, 8, 0, 8, 3, 8, 0, 9, 8, 9, 4, 5, 4, 9, 4, 5, 7, 6, 7, 5, 7, 6,11, 7,11, 8 */
    {1, 1, 0, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1},
    /*  26:  ,1, ,3,4, , , :  8, 7,11, 7, 8, 4, 9, 4, 8, 0, 9, 8, 9, 0, 1, 3, 1, 0, 1, 3, 2,11, 2, 3, 8,11, 3 */
    {0, 0, 3, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 2, 0, 0, 0},
    /*  74:  ,1, ,3, , ,6, : 10, 5, 9, 5,10, 6,11, 6,10, 2,11,10,11, 2, 3, 1, 3, 2, 3, 1, 0, 9, 0, 1,10, 9, 1 */
    {0, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 1, 0, 0, 0},
    /*  82:  ,1, , ,4, ,6, :  8, 0, 9, 1, 9, 0, 9, 1,10, 9,10, 5, 6, 5,10, 5, 6, 4, 7, 4, 6, 4, 7, 8, 4, 8, 9 */
    {0, 0, 1, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0, 0},
    /* 164:  , ,2, , ,5, ,7:  9, 1,10, 2,10, 1,10, 2,11,10,11, 6, 7, 6,11, 6, 7, 5, 4, 5, 7, 5, 4, 9, 5, 9,10 */
    {0, 0, 1, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0, 0},
    /*  88:  , , ,3,4, ,6, : 10, 2,11, 3,11, 2,11, 3, 8,11, 8, 7, 4, 7, 8, 7, 4, 6, 5, 6, 4, 6, 5,10, 6,10,11 */
    {0, 0, 1, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 3, 0, 0, 0},
    /* 167: 0,1,2, , ,5, ,7: 11, 2,10, 2,11, 3, 8, 3,11, 7, 8,11, 8, 7, 4, 6, 4, 7, 4, 6, 5,10, 5, 6,11,10, 6 */
    {0, 0, 0, 1, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 3, 0, 0},
    /*  91: 0,1, ,3,4, ,6, : 10, 1, 9, 1,10, 2,11, 2,10, 6,11,10,11, 6, 7, 5, 7, 6, 7, 5, 4, 9, 4, 5,10, 9, 5 */
    {0, 0, 0, 1, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0},
    /* 173: 0, ,2,3, ,5, ,7:  9, 0, 8, 0, 9, 1,10, 1, 9, 5,10, 9,10, 5, 6, 4, 6, 5, 6, 4, 7, 8, 7, 4, 9, 8, 4 */
    {0, 0, 0, 1, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0},
    /* 181: 0, ,2, ,4,5, ,7:  9, 5,10, 6,10, 5,10, 6,11,10,11, 2, 3, 2,11, 2, 3, 1, 0, 1, 3, 1, 0, 9, 1, 9,10 */
    {0, 0, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 1, 0, 0},
    /* 229: 0, ,2, , ,5,6,7: 11, 7, 8, 4, 8, 7, 8, 4, 9, 8, 9, 0, 1, 0, 9, 0, 1, 3, 2, 3, 1, 3, 2,11, 3,11, 8 */
    {0, 0, 0, 3, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 2, 0, 0},
    /*  94:  ,1,2,3,4, ,6, :  8, 3,11, 3, 8, 0, 9, 0, 8, 4, 9, 8, 9, 4, 5, 7, 5, 4, 5, 7, 6,11, 6, 7, 8,11, 7 */
    {1, 1, 1, 0, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1},
    /* 122:  ,1, ,3,4,5,6, : 10, 6,11, 7,11, 6,11, 7, 8,11, 8, 3, 0, 3, 8, 3, 0, 2, 1, 2, 0, 2, 1,10, 2,10,11 */
    {1, 1, 1, 3, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 2, 1, 1},
    /* 218:  ,1, ,3,4, ,6,7:  8, 4, 9, 5, 9, 4, 9, 5,10, 9,10, 1, 2, 1,10, 1, 2, 0, 3, 0, 2, 0, 3, 8, 0, 8, 9 */
    {1, 1, 1, 3, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 0, 1, 1},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion8[6][8] = {
    /*  15: 0,1,2,3, , , , :  9, 8,10,10, 8,11 */ {0, 0, 0, 0, 1, 1, 1, 1},
    /*  51: 0,1, , ,4,5, , :  1, 5, 3, 3, 5, 7 */ {0, 0, 1, 1, 0, 0, 1, 1},
    /* 153: 0, , ,3,4, , ,7:  0, 4, 2, 4, 6, 2 */ {0, 1, 1, 0, 0, 1, 1, 0},
    /* 102:  ,1,2, , ,5,6, :  0, 2, 4, 4, 2, 6 */ {0, 1, 1, 0, 0, 1, 1, 0},
    /* 204:  , ,2,3, , ,6,7:  1, 3, 5, 3, 7, 5 */ {0, 0, 1, 1, 0, 0, 1, 1},
    /* 240:  , , , ,4,5,6,7:  9,10, 8,10,11, 8 */ {0, 0, 0, 0, 1, 1, 1, 1},
};
static const signed char triangleRegion8[6][4] = {
    /*  15: 0,1,2,3, , , , :  9, 8,10,10, 8,11 */ {0, 1, 0, 1},
    /*  51: 0,1, , ,4,5, , :  1, 5, 3, 3, 5, 7 */ {0, 1, 0, 1},
    /* 153: 0, , ,3,4, , ,7:  0, 4, 2, 4, 6, 2 */ {0, 1, 0, 1},
    /* 102:  ,1,2, , ,5,6, :  0, 2, 4, 4, 2, 6 */ {1, 0, 1, 0},
    /* 204:  , ,2,3, , ,6,7:  1, 3, 5, 3, 7, 5 */ {1, 0, 1, 0},
    /* 240:  , , , ,4,5,6,7:  9,10, 8,10,11, 8 */ {1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion9[8][8] = {
    /*  39: 0,1,2, , ,5, , :  2,10, 5, 3, 2, 5, 3, 5, 4, 3, 4, 8 */ {0, 0, 0, 1, 1, 0, 1, 1},
    /*  27: 0,1, ,3,4, , , :  4, 7,11, 9, 4,11, 9,11, 2, 9, 2, 1 */ {0, 0, 1, 0, 0, 1, 1, 1},
    /* 141: 0, ,2,3, , , ,7: 10, 7, 6, 1, 7,10, 1, 8, 7, 1, 0, 8 */ {0, 1, 0, 0, 1, 1, 1, 0},
    /* 177: 0, , , ,4,5, ,7:  3, 6,11, 0, 6, 3, 0, 5, 6, 0, 9, 5 */ {0, 1, 1, 1, 0, 0, 1, 0},
    /*  78:  ,1,2,3, , ,6, :  3,11, 6, 0, 3, 6, 0, 6, 5, 0, 5, 9 */ {0, 1, 1, 1, 0, 0, 1, 0},
    /* 114:  ,1, , ,4,5,6, : 10, 6, 7, 1,10, 7, 1, 7, 8, 1, 8, 0 */ {0, 1, 0, 0, 1, 1, 1, 0},
    /* 228:  , ,2, , ,5,6,7:  4,11, 7, 9,11, 4, 9, 2,11, 9, 1, 2 */ {0, 0, 1, 0, 0, 1, 1, 1},
    /* 216:  , , ,3,4, ,6,7:  2, 5,10, 3, 5, 2, 3, 4, 5, 3, 8, 4 */ {0, 0, 0, 1, 1, 0, 1, 1},
};
static const signed char triangleRegion9[8][8] = {
    /*  39: 0,1,2, , ,5, , :  2,10, 5, 3, 2, 5, 3, 5, 4, 3, 4, 8 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  27: 0,1, ,3,4, , , :  4, 7,11, 9, 4,11, 9,11, 2, 9, 2, 1 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 141: 0, ,2,3, , , ,7: 10, 7, 6, 1, 7,10, 1, 8, 7, 1, 0, 8 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 177: 0, , , ,4,5, ,7:  3, 6,11, 0, 6, 3, 0, 5, 6, 0, 9, 5 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  78:  ,1,2,3, , ,6, :  3,11, 6, 0, 3, 6, 0, 6, 5, 0, 5, 9 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 114:  ,1, , ,4,5,6, : 10, 6, 7, 1,10, 7, 1, 7, 8, 1, 8, 0 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 228:  , ,2, , ,5,6,7:  4,11, 7, 9,11, 4, 9, 2,11, 9, 1, 2 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 216:  , , ,3,4, ,6,7:  2, 5,10, 3, 5, 2, 3, 4, 5, 3, 8, 4 */ {1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion10_1_1[6][8] = {
    /* 195: 0,1, , , , ,6,7:  5,10, 7,11, 7,10, 8, 1, 9, 1, 8, 3 */ {0, 0, 1, 1, 1, 1, 2, 2},
    /*  85: 0, ,2, ,4, ,6, :  1, 2, 5, 6, 5, 2, 4, 3, 0, 3, 4, 7 */ {0, 1, 2, 1, 0, 1, 2, 1},
    /* 105: 0, , ,3, ,5,6, : 11, 0, 8, 0,11, 2, 4, 9, 6,10, 6, 9 */ {0, 1, 1, 0, 1, 2, 2, 1},
    /* 150:  ,1,2, ,4, , ,7:  9, 0,10, 2,10, 0, 6, 8, 4, 8, 6,11 */ {0, 1, 1, 0, 2, 0, 0, 2},
    /* 170:  ,1, ,3, ,5, ,7:  7, 2, 3, 2, 7, 6, 0, 1, 4, 5, 4, 1 */ {0, 1, 0, 2, 0, 1, 0, 2},
    /*  60:  , ,2,3,4,5, , :  7, 9, 5, 9, 7, 8,10, 1,11, 3,11, 1 */ {0, 0, 1, 1, 2, 2, 0, 0},
};
static const signed char triangleRegion10_1_1[6][8] = {
    /* 195: 0,1, , , , ,6,7:  5,10, 7,11, 7,10, 8, 1, 9, 1, 8, 3 */ {2, 1, 2, 1, 0, 1, 0, 1},
    /*  85: 0, ,2, ,4, ,6, :  1, 2, 5, 6, 5, 2, 4, 3, 0, 3, 4, 7 */ {2, 1, 2, 1, 0, 1, 0, 1},
    /* 105: 0, , ,3, ,5,6, : 11, 0, 8, 0,11, 2, 4, 9, 6,10, 6, 9 */ {0, 1, 0, 1, 2, 1, 2, 1},
    /* 150:  ,1,2, ,4, , ,7:  9, 0,10, 2,10, 0, 6, 8, 4, 8, 6,11 */ {1, 0, 1, 0, 2, 0, 2, 0},
    /* 170:  ,1, ,3, ,5, ,7:  7, 2, 3, 2, 7, 6, 0, 1, 4, 5, 4, 1 */ {2, 0, 2, 0, 1, 0, 1, 0},
    /*  60:  , ,2,3,4,5, , :  7, 9, 5, 9, 7, 8,10, 1,11, 3,11, 1 */ {2, 0, 2, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion10_1_1_[6][8] = {
    /* 195: 0,1, , , , ,6,7:  5, 9, 7, 8, 7, 9,11, 1,10, 1,11, 3 */ {0, 0, 1, 1, 2, 2, 0, 0},
    /*  85: 0, ,2, ,4, ,6, :  3, 2, 7, 6, 7, 2, 4, 1, 0, 1, 4, 5 */ {0, 1, 0, 2, 0, 1, 0, 2},
    /* 105: 0, , ,3, ,5,6, : 10, 0, 9, 0,10, 2, 4, 8, 6,11, 6, 8 */ {0, 1, 1, 0, 2, 0, 0, 2},
    /* 150:  ,1,2, ,4, , ,7:  8, 0,11, 2,11, 0, 6, 9, 4, 9, 6,10 */ {0, 1, 1, 0, 1, 2, 2, 1},
    /* 170:  ,1, ,3, ,5, ,7:  5, 2, 1, 2, 5, 6, 0, 3, 4, 7, 4, 3 */ {0, 1, 2, 1, 0, 1, 2, 1},
    /*  60:  , ,2,3,4,5, , :  7,10, 5,10, 7,11, 9, 1, 8, 3, 8, 1 */ {0, 0, 1, 1, 1, 1, 2, 2},
};
static const signed char triangleRegion10_1_1_[6][8] = {
    /* 195: 0,1, , , , ,6,7:  5, 9, 7, 8, 7, 9,11, 1,10, 1,11, 3 */ {0, 2, 0, 2, 0, 1, 0, 1},
    /*  85: 0, ,2, ,4, ,6, :  3, 2, 7, 6, 7, 2, 4, 1, 0, 1, 4, 5 */ {0, 2, 0, 2, 0, 1, 0, 1},
    /* 105: 0, , ,3, ,5,6, : 10, 0, 9, 0,10, 2, 4, 8, 6,11, 6, 8 */ {0, 1, 0, 1, 0, 2, 0, 2},
    /* 150:  ,1,2, ,4, , ,7:  8, 0,11, 2,11, 0, 6, 9, 4, 9, 6,10 */ {1, 0, 1, 0, 1, 2, 1, 2},
    /* 170:  ,1, ,3, ,5, ,7:  5, 2, 1, 2, 5, 6, 0, 3, 4, 7, 4, 3 */ {1, 2, 1, 2, 1, 0, 1, 0},
    /*  60:  , ,2,3,4,5, , :  7,10, 5,10, 7,11, 9, 1, 8, 3, 8, 1 */ {1, 2, 1, 2, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion10_1_2[6][8] = {
    /* 195: 0,1, , , , ,6,7:  3,11, 7, 3, 7, 8, 9, 8, 7, 5, 9, 7, 9, 5,10, 9,10, 1, 3, 1,10,11, 3,10 */ {0, 0, 1, 1, 2, 2, 0, 0},
    /*  85: 0, ,2, ,4, ,6, :  7, 6, 5, 7, 5, 4, 0, 4, 5, 1, 0, 5, 0, 1, 2, 0, 2, 3, 7, 3, 2, 6, 7, 2 */ {0, 1, 0, 2, 0, 1, 0, 2},
    /* 105: 0, , ,3, ,5,6, : 11, 2,10, 6,11,10,11, 6, 4,11, 4, 8, 0, 8, 4, 9, 0, 4, 0, 9,10, 0,10, 2 */ {0, 1, 1, 0, 2, 0, 0, 2},
    /* 150:  ,1,2, ,4, , ,7: 11, 2,10,11,10, 6, 4, 6,10, 9, 4,10, 4, 9, 0, 4, 0, 8,11, 8, 0, 2,11, 0 */ {0, 1, 1, 0, 1, 2, 2, 1},
    /* 170:  ,1, ,3, ,5, ,7:  7, 6, 5, 4, 7, 5, 7, 4, 0, 7, 0, 3, 2, 3, 0, 1, 2, 0, 2, 1, 5, 2, 5, 6 */ {0, 1, 2, 1, 0, 1, 2, 1},
    /*  60:  , ,2,3,4,5, , :  7, 8, 3,11, 7, 3, 7,11,10, 7,10, 5, 9, 5,10, 1, 9,10, 9, 1, 3, 9, 3, 8 */ {0, 0, 1, 1, 1, 1, 2, 2},
};
static const signed char triangleRegion10_1_2[6][16] = {
    /* 195: 0,1, , , , ,6,7:  3,11, 7, 3, 7, 8, 9, 8, 7, 5, 9, 7, 9, 5,10, 9,10, 1, 3, 1,10,11, 3,10 */
    {0, 0, 0, 0, 0, 2, 0, 2, 0, 0, 0, 0, 0, 1, 0, 1},
    /*  85: 0, ,2, ,4, ,6, :  7, 6, 5, 7, 5, 4, 0, 4, 5, 1, 0, 5, 0, 1, 2, 0, 2, 3, 7, 3, 2, 6, 7, 2 */
    {0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 2, 0, 2},
    /* 105: 0, , ,3, ,5,6, : 11, 2,10, 6,11,10,11, 6, 4,11, 4, 8, 0, 8, 4, 9, 0, 4, 0, 9,10, 0,10, 2 */
    {0, 0, 0, 0, 0, 2, 0, 2, 0, 0, 0, 0, 0, 1, 0, 1},
    /* 150:  ,1,2, ,4, , ,7: 11, 2,10,11,10, 6, 4, 6,10, 9, 4,10, 4, 9, 0, 4, 0, 8,11, 8, 0, 2,11, 0 */
    {1, 1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 0, 1, 0},
    /* 170:  ,1, ,3, ,5, ,7:  7, 6, 5, 4, 7, 5, 7, 4, 0, 7, 0, 3, 2, 3, 0, 1, 2, 0, 2, 1, 5, 2, 5, 6 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2, 1, 2},
    /*  60:  , ,2,3,4,5, , :  7, 8, 3,11, 7, 3, 7,11,10, 7,10, 5, 9, 5,10, 1, 9,10, 9, 1, 3, 9, 3, 8 */
    {1, 1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion10_2[6][8] = {
    /* 195: 0,1, , , , ,6,7: 12, 5, 9,12, 9, 8,12, 8, 3,12, 3, 1,12, 1,10,12,10,11,12,11, 7,12, 7, 5 */ {0, 0, 1, 1, 1, 1, 0, 0},
    /*  85: 0, ,2, ,4, ,6, : 12, 1, 0,12, 0, 4,12, 4, 7,12, 7, 3,12, 3, 2,12, 2, 6,12, 6, 5,12, 5, 1 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 105: 0, , ,3, ,5,6, :  4, 8,12, 6, 4,12,10, 6,12, 9,10,12, 0, 9,12, 2, 0,12,11, 2,12, 8,11,12 */ {0, 1, 1, 0, 1, 0, 0, 1},
    /* 150:  ,1,2, ,4, , ,7: 12, 9, 4,12, 4, 6,12, 6,11,12,11, 8,12, 8, 0,12, 0, 2,12, 2,10,12,10, 9 */ {0, 1, 1, 0, 1, 0, 0, 1},
    /* 170:  ,1, ,3, ,5, ,7:  0, 3,12, 4, 0,12, 5, 4,12, 1, 5,12, 2, 1,12, 6, 2,12, 7, 6,12, 3, 7,12 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  60:  , ,2,3,4,5, , : 10, 5,12,11,10,12, 3,11,12, 1, 3,12, 9, 1,12, 8, 9,12, 7, 8,12, 5, 7,12 */ {0, 0, 1, 1, 1, 1, 0, 0},
};
static const signed char triangleRegion10_2[6][16] = {
    /* 195: 0,1, , , , ,6,7: 12, 5, 9,12, 9, 8,12, 8, 3,12, 3, 1,12, 1,10,12,10,11,12,11, 7,12, 7, 5 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  85: 0, ,2, ,4, ,6, : 12, 1, 0,12, 0, 4,12, 4, 7,12, 7, 3,12, 3, 2,12, 2, 6,12, 6, 5,12, 5, 1 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 105: 0, , ,3, ,5,6, :  4, 8,12, 6, 4,12,10, 6,12, 9,10,12, 0, 9,12, 2, 0,12,11, 2,12, 8,11,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 150:  ,1,2, ,4, , ,7: 12, 9, 4,12, 4, 6,12, 6,11,12,11, 8,12, 8, 0,12, 0, 2,12, 2,10,12,10, 9 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 170:  ,1, ,3, ,5, ,7:  0, 3,12, 4, 0,12, 5, 4,12, 1, 5,12, 2, 1,12, 6, 2,12, 7, 6,12, 3, 7,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  60:  , ,2,3,4,5, , : 10, 5,12,11,10,12, 3,11,12, 1, 3,12, 9, 1,12, 8, 9,12, 7, 8,12, 5, 7,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion10_2_[6][8] = {
    /* 195: 0,1, , , , ,6,7:  8, 7,12, 9, 8,12, 1, 9,12, 3, 1,12,11, 3,12,10,11,12, 5,10,12, 7, 5,12 */ {0, 0, 1, 1, 1, 1, 0, 0},
    /*  85: 0, ,2, ,4, ,6, :  4, 5,12, 0, 4,12, 3, 0,12, 7, 3,12, 6, 7,12, 2, 6,12, 1, 2,12, 5, 1,12 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 105: 0, , ,3, ,5,6, : 12,11, 6,12, 6, 4,12, 4, 9,12, 9,10,12,10, 2,12, 2, 0,12, 0, 8,12, 8,11 */ {0, 1, 1, 0, 1, 0, 0, 1},
    /* 150:  ,1,2, ,4, , ,7:  6,10,12, 4, 6,12, 8, 4,12,11, 8,12, 2,11,12, 0, 2,12, 9, 0,12,10, 9,12 */ {0, 1, 1, 0, 1, 0, 0, 1},
    /* 170:  ,1, ,3, ,5, ,7: 12, 7, 4,12, 4, 0,12, 0, 1,12, 1, 5,12, 5, 6,12, 6, 2,12, 2, 3,12, 3, 7 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  60:  , ,2,3,4,5, , : 12, 7,11,12,11,10,12,10, 1,12, 1, 3,12, 3, 8,12, 8, 9,12, 9, 5,12, 5, 7 */ {0, 0, 1, 1, 1, 1, 0, 0},
};
static const signed char triangleRegion10_2_[6][16] = {
    /* 195: 0,1, , , , ,6,7:  8, 7,12, 9, 8,12, 1, 9,12, 3, 1,12,11, 3,12,10,11,12, 5,10,12, 7, 5,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  85: 0, ,2, ,4, ,6, :  4, 5,12, 0, 4,12, 3, 0,12, 7, 3,12, 6, 7,12, 2, 6,12, 1, 2,12, 5, 1,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 105: 0, , ,3, ,5,6, : 12,11, 6,12, 6, 4,12, 4, 9,12, 9,10,12,10, 2,12, 2, 0,12, 0, 8,12, 8,11 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 150:  ,1,2, ,4, , ,7:  6,10,12, 4, 6,12, 8, 4,12,11, 8,12, 2,11,12, 0, 2,12, 9, 0,12,10, 9,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 170:  ,1, ,3, ,5, ,7: 12, 7, 4,12, 4, 0,12, 0, 1,12, 1, 5,12, 5, 6,12, 6, 2,12, 2, 3,12, 3, 7 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  60:  , ,2,3,4,5, , : 12, 7,11,12,11,10,12,10, 1,12, 1, 3,12, 3, 8,12, 8, 9,12, 9, 5,12, 5, 7 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion11[12][8] = {
    /*  23: 0,1,2, ,4, , , :  2,10, 9, 2, 9, 7, 2, 7, 3, 7, 9, 4 */ {0, 0, 0, 1, 0, 1, 1, 1},
    /* 139: 0,1, ,3, , , ,7:  1, 6, 2, 1, 8, 6, 1, 9, 8, 8, 7, 6 */ {0, 0, 1, 0, 1, 1, 1, 0},
    /*  99: 0,1, , , ,5,6, :  8, 3, 1, 8, 1, 6, 8, 6, 4, 6, 1,10 */ {0, 0, 1, 1, 1, 0, 0, 1},
    /*  77: 0, ,2,3, , ,6, :  0, 8,11, 0,11, 5, 0, 5, 1, 5,11, 6 */ {0, 1, 0, 0, 1, 1, 0, 1},
    /*  57: 0, , ,3,4,5, , :  9, 5, 7, 9, 7, 2, 9, 2, 0, 2, 7,11 */ {0, 1, 1, 0, 0, 0, 1, 1},
    /* 209: 0, , , ,4, ,6,7:  5, 0, 4, 5,11, 0, 5,10,11,11, 3, 0 */ {0, 1, 1, 1, 0, 1, 0, 0},
    /*  46:  ,1,2,3, ,5, , :  5, 4, 0, 5, 0,11, 5,11,10,11, 0, 3 */ {0, 1, 1, 1, 0, 1, 0, 0},
    /* 198:  ,1,2, , , ,6,7:  9, 7, 5, 9, 2, 7, 9, 0, 2, 2,11, 7 */ {0, 1, 1, 0, 0, 0, 1, 1},
    /* 178:  ,1, , ,4,5, ,7:  0,11, 8, 0, 5,11, 0, 1, 5, 5, 6,11 */ {0, 1, 0, 0, 1, 1, 0, 1},
    /* 156:  , ,2,3,4, , ,7:  8, 1, 3, 8, 6, 1, 8, 4, 6, 6,10, 1 */ {0, 0, 1, 1, 1, 0, 0, 1},
    /* 116:  , ,2, ,4,5,6, :  1, 2, 6, 1, 6, 8, 1, 8, 9, 8, 6, 7 */ {0, 0, 1, 0, 1, 1, 1, 0},
    /* 232:  , , ,3, ,5,6,7:  2, 9,10, 2, 7, 9, 2, 3, 7, 7, 4, 9 */ {0, 0, 0, 1, 0, 1, 1, 1},
};
static const signed char triangleRegion11[12][8] = {
    /*  23: 0,1,2, ,4, , , :  2,10, 9, 2, 9, 7, 2, 7, 3, 7, 9, 4 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 139: 0,1, ,3, , , ,7:  1, 6, 2, 1, 8, 6, 1, 9, 8, 8, 7, 6 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  99: 0,1, , , ,5,6, :  8, 3, 1, 8, 1, 6, 8, 6, 4, 6, 1,10 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  77: 0, ,2,3, , ,6, :  0, 8,11, 0,11, 5, 0, 5, 1, 5,11, 6 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  57: 0, , ,3,4,5, , :  9, 5, 7, 9, 7, 2, 9, 2, 0, 2, 7,11 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 209: 0, , , ,4, ,6,7:  5, 0, 4, 5,11, 0, 5,10,11,11, 3, 0 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  46:  ,1,2,3, ,5, , :  5, 4, 0, 5, 0,11, 5,11,10,11, 0, 3 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 198:  ,1,2, , , ,6,7:  9, 7, 5, 9, 2, 7, 9, 0, 2, 2,11, 7 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 178:  ,1, , ,4,5, ,7:  0,11, 8, 0, 5,11, 0, 1, 5, 5, 6,11 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 156:  , ,2,3,4, , ,7:  8, 1, 3, 8, 6, 1, 8, 4, 6, 6,10, 1 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 116:  , ,2, ,4,5,6, :  1, 2, 6, 1, 6, 8, 1, 8, 9, 8, 6, 7 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 232:  , , ,3, ,5,6,7:  2, 9,10, 2, 7, 9, 2, 3, 7, 7, 4, 9 */ {1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion12_1_1[24][8] = {
    /* 135: 0,1,2, , , , ,7:  7, 6,11,10, 3, 2, 3,10, 8, 9, 8,10 */ {0, 0, 0, 1, 1, 1, 1, 2},
    /*  75: 0,1, ,3, , ,6, :  6, 5,10, 9, 2, 1, 2, 9,11, 8,11, 9 */ {0, 0, 1, 0, 1, 1, 2, 1},
    /*  83: 0,1, , ,4, ,6, : 10, 6, 5, 7, 9, 4, 9, 7, 1, 3, 1, 7 */ {0, 0, 1, 1, 0, 1, 2, 1},
    /* 163: 0,1, , , ,5, ,7:  7, 6,11, 4, 8, 5, 3, 5, 8, 5, 3, 1 */ {0, 0, 1, 1, 1, 0, 1, 2},
    /*  45: 0, ,2,3, ,5, , :  5, 4, 9, 8, 1, 0, 1, 8,10,11,10, 8 */ {0, 1, 0, 0, 1, 2, 1, 1},
    /*  53: 0, ,2, ,4,5, , :  1, 2,10, 0, 9, 3, 5, 3, 9, 3, 5, 7 */ {0, 1, 2, 1, 0, 0, 1, 1},
    /* 149: 0, ,2, ,4, , ,7: 10, 1, 2, 0,11, 3,11, 0, 6, 4, 6, 0 */ {0, 1, 2, 1, 0, 1, 1, 0},
    /* 101: 0, ,2, , ,5,6, :  8, 3, 0, 2, 9, 1, 9, 2, 4, 6, 4, 2 */ {0, 1, 2, 1, 1, 2, 2, 1},
    /* 197: 0, ,2, , , ,6,7:  3, 0, 8, 2,11, 1, 7, 1,11, 1, 7, 5 */ {0, 1, 2, 1, 1, 1, 2, 2},
    /*  89: 0, , ,3,4, ,6, :  6, 5,10, 7,11, 4, 2, 4,11, 4, 2, 0 */ {0, 1, 1, 0, 0, 1, 2, 1},
    /* 169: 0, , ,3, ,5, ,7:  9, 5, 4, 6, 8, 7, 8, 6, 0, 2, 0, 6 */ {0, 1, 1, 0, 1, 2, 1, 0},
    /* 225: 0, , , , ,5,6,7:  8, 3, 0, 7, 4,11, 9,11, 4,11, 9,10 */ {0, 1, 1, 1, 1, 2, 2, 2},
    /*  30:  ,1,2,3,4, , , :  4, 7, 8,11, 0, 3, 0,11, 9,10, 9,11 */ {0, 1, 1, 1, 2, 0, 0, 0},
    /*  86:  ,1,2, ,4, ,6, :  4, 7, 8, 5, 9, 6, 0, 6, 9, 6, 0, 2 */ {0, 1, 1, 0, 2, 0, 1, 0},
    /* 166:  ,1,2, , ,5, ,7: 11, 7, 6, 4,10, 5,10, 4, 2, 0, 2, 4 */ {0, 1, 1, 0, 0, 1, 0, 2},
    /*  58:  ,1, ,3,4,5, , : 11, 2, 3, 1, 8, 0, 8, 1, 7, 5, 7, 1 */ {0, 1, 0, 2, 1, 1, 0, 0},
    /* 154:  ,1, ,3,4, , ,7:  0, 1, 9, 3, 8, 2, 4, 2, 8, 2, 4, 6 */ {0, 1, 0, 2, 2, 0, 0, 2},
    /* 106:  ,1, ,3, ,5,6, :  2, 3,11, 1,10, 0, 6, 0,10, 0, 6, 4 */ {0, 1, 0, 2, 0, 1, 1, 0},
    /* 202:  ,1, ,3, , ,6,7:  9, 0, 1, 3,10, 2,10, 3, 5, 7, 5, 3 */ {0, 1, 0, 2, 0, 0, 2, 2},
    /* 210:  ,1, , ,4, ,6,7:  9, 0, 1, 4, 5, 8,10, 8, 5, 8,10,11 */ {0, 1, 0, 0, 2, 0, 2, 2},
    /*  92:  , ,2,3,4, ,6, :  8, 4, 7, 5,11, 6,11, 5, 3, 1, 3, 5 */ {0, 0, 1, 1, 2, 0, 1, 0},
    /* 172:  , ,2,3, ,5, ,7:  5, 4, 9, 6,10, 7, 1, 7,10, 7, 1, 3 */ {0, 0, 1, 1, 0, 2, 0, 1},
    /* 180:  , ,2, ,4,5, ,7: 10, 1, 2, 5, 6, 9,11, 9, 6, 9,11, 8 */ {0, 0, 1, 0, 2, 2, 0, 2},
    /* 120:  , , ,3,4,5,6, : 11, 2, 3, 6, 7,10, 8,10, 7,10, 8, 9 */ {0, 0, 0, 1, 2, 2, 2, 0},
};
static const signed char triangleRegion12_1_1[24][8] = {
    /* 135: 0,1,2, , , , ,7:  7, 6,11,10, 3, 2, 3,10, 8, 9, 8,10 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /*  75: 0,1, ,3, , ,6, :  6, 5,10, 9, 2, 1, 2, 9,11, 8,11, 9 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /*  83: 0,1, , ,4, ,6, : 10, 6, 5, 7, 9, 4, 9, 7, 1, 3, 1, 7 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /* 163: 0,1, , , ,5, ,7:  7, 6,11, 4, 8, 5, 3, 5, 8, 5, 3, 1 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /*  45: 0, ,2,3, ,5, , :  5, 4, 9, 8, 1, 0, 1, 8,10,11,10, 8 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /*  53: 0, ,2, ,4,5, , :  1, 2,10, 0, 9, 3, 5, 3, 9, 3, 5, 7 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /* 149: 0, ,2, ,4, , ,7: 10, 1, 2, 0,11, 3,11, 0, 6, 4, 6, 0 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /* 101: 0, ,2, , ,5,6, :  8, 3, 0, 2, 9, 1, 9, 2, 4, 6, 4, 2 */ {0, 1, 2, 1, 2, 1, 2, 1},
    /* 197: 0, ,2, , , ,6,7:  3, 0, 8, 2,11, 1, 7, 1,11, 1, 7, 5 */ {0, 1, 2, 1, 2, 1, 2, 1},
    /*  89: 0, , ,3,4, ,6, :  6, 5,10, 7,11, 4, 2, 4,11, 4, 2, 0 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /* 169: 0, , ,3, ,5, ,7:  9, 5, 4, 6, 8, 7, 8, 6, 0, 2, 0, 6 */ {2, 1, 0, 1, 0, 1, 0, 1},
    /* 225: 0, , , , ,5,6,7:  8, 3, 0, 7, 4,11, 9,11, 4,11, 9,10 */ {0, 1, 2, 1, 2, 1, 2, 1},
    /*  30:  ,1,2,3,4, , , :  4, 7, 8,11, 0, 3, 0,11, 9,10, 9,11 */ {2, 0, 1, 0, 1, 0, 1, 0},
    /*  86:  ,1,2, ,4, ,6, :  4, 7, 8, 5, 9, 6, 0, 6, 9, 6, 0, 2 */ {2, 0, 1, 0, 1, 0, 1, 0},
    /* 166:  ,1,2, , ,5, ,7: 11, 7, 6, 4,10, 5,10, 4, 2, 0, 2, 4 */ {2, 0, 1, 0, 1, 0, 1, 0},
    /*  58:  ,1, ,3,4,5, , : 11, 2, 3, 1, 8, 0, 8, 1, 7, 5, 7, 1 */ {2, 0, 1, 0, 1, 0, 1, 0},
    /* 154:  ,1, ,3,4, , ,7:  0, 1, 9, 3, 8, 2, 4, 2, 8, 2, 4, 6 */ {1, 0, 2, 0, 2, 0, 2, 0},
    /* 106:  ,1, ,3, ,5,6, :  2, 3,11, 1,10, 0, 6, 0,10, 0, 6, 4 */ {2, 0, 1, 0, 1, 0, 1, 0},
    /* 202:  ,1, ,3, , ,6,7:  9, 0, 1, 3,10, 2,10, 3, 5, 7, 5, 3 */ {1, 0, 2, 0, 2, 0, 2, 0},
    /* 210:  ,1, , ,4, ,6,7:  9, 0, 1, 4, 5, 8,10, 8, 5, 8,10,11 */ {1, 0, 2, 0, 2, 0, 2, 0},
    /*  92:  , ,2,3,4, ,6, :  8, 4, 7, 5,11, 6,11, 5, 3, 1, 3, 5 */ {2, 0, 1, 0, 1, 0, 1, 0},
    /* 172:  , ,2,3, ,5, ,7:  5, 4, 9, 6,10, 7, 1, 7,10, 7, 1, 3 */ {2, 0, 1, 0, 1, 0, 1, 0},
    /* 180:  , ,2, ,4,5, ,7: 10, 1, 2, 5, 6, 9,11, 9, 6, 9,11, 8 */ {1, 0, 2, 0, 2, 0, 2, 0},
    /* 120:  , , ,3,4,5,6, : 11, 2, 3, 6, 7,10, 8,10, 7,10, 8, 9 */ {1, 0, 2, 0, 2, 0, 2, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion12_1_1_[24][8] = {
    /* 135: 0,1,2, , , , ,7:  3, 2,11,10, 7, 6, 7,10, 8, 9, 8,10 */ {0, 0, 0, 1, 2, 2, 2, 0},
    /*  75: 0,1, ,3, , ,6, :  2, 1,10, 9, 6, 5, 6, 9,11, 8,11, 9 */ {0, 0, 1, 0, 2, 2, 0, 2},
    /*  83: 0,1, , ,4, ,6, :  9, 4, 5, 7,10, 6,10, 7, 1, 3, 1, 7 */ {0, 0, 1, 1, 0, 2, 0, 1},
    /* 163: 0,1, , , ,5, ,7:  7, 4, 8, 6,11, 5, 3, 5,11, 5, 3, 1 */ {0, 0, 1, 1, 2, 0, 1, 0},
    /*  45: 0, ,2,3, ,5, , :  1, 0, 9, 8, 5, 4, 5, 8,10,11,10, 8 */ {0, 1, 0, 0, 2, 0, 2, 2},
    /*  53: 0, ,2, ,4,5, , :  1, 0, 9, 2,10, 3, 5, 3,10, 3, 5, 7 */ {0, 1, 0, 2, 0, 0, 2, 2},
    /* 149: 0, ,2, ,4, , ,7: 11, 3, 2, 0,10, 1,10, 0, 6, 4, 6, 0 */ {0, 1, 0, 2, 0, 1, 1, 0},
    /* 101: 0, ,2, , ,5,6, :  9, 1, 0, 2, 8, 3, 8, 2, 4, 6, 4, 2 */ {0, 1, 0, 2, 2, 0, 0, 2},
    /* 197: 0, ,2, , , ,6,7:  3, 2,11, 0, 8, 1, 7, 1, 8, 1, 7, 5 */ {0, 1, 0, 2, 1, 1, 0, 0},
    /*  89: 0, , ,3,4, ,6, :  6, 7,11, 5,10, 4, 2, 4,10, 4, 2, 0 */ {0, 1, 1, 0, 0, 1, 0, 2},
    /* 169: 0, , ,3, ,5, ,7:  8, 7, 4, 6, 9, 5, 9, 6, 0, 2, 0, 6 */ {0, 1, 1, 0, 2, 0, 1, 0},
    /* 225: 0, , , , ,5,6,7:  8, 7, 4, 3, 0,11, 9,11, 0,11, 9,10 */ {0, 1, 1, 1, 2, 0, 0, 0},
    /*  30:  ,1,2,3,4, , , :  0, 3, 8,11, 4, 7, 4,11, 9,10, 9,11 */ {0, 1, 1, 1, 1, 2, 2, 2},
    /*  86:  ,1,2, ,4, ,6, :  4, 5, 9, 7, 8, 6, 0, 6, 8, 6, 0, 2 */ {0, 1, 1, 0, 1, 2, 1, 0},
    /* 166:  ,1,2, , ,5, ,7: 10, 5, 6, 4,11, 7,11, 4, 2, 0, 2, 4 */ {0, 1, 1, 0, 0, 1, 2, 1},
    /*  58:  ,1, ,3,4,5, , :  8, 0, 3, 1,11, 2,11, 1, 7, 5, 7, 1 */ {0, 1, 2, 1, 1, 1, 2, 2},
    /* 154:  ,1, ,3,4, , ,7:  0, 3, 8, 1, 9, 2, 4, 2, 9, 2, 4, 6 */ {0, 1, 2, 1, 1, 2, 2, 1},
    /* 106:  ,1, ,3, ,5,6, :  2, 1,10, 3,11, 0, 6, 0,11, 0, 6, 4 */ {0, 1, 2, 1, 0, 1, 1, 0},
    /* 202:  ,1, ,3, , ,6,7: 10, 2, 1, 3, 9, 0, 9, 3, 5, 7, 5, 3 */ {0, 1, 2, 1, 0, 0, 1, 1},
    /* 210:  ,1, , ,4, ,6,7:  9, 4, 5, 0, 1, 8,10, 8, 1, 8,10,11 */ {0, 1, 0, 0, 1, 2, 1, 1},
    /*  92:  , ,2,3,4, ,6, : 11, 6, 7, 5, 8, 4, 8, 5, 3, 1, 3, 5 */ {0, 0, 1, 1, 1, 0, 1, 2},
    /* 172:  , ,2,3, ,5, ,7:  5, 6,10, 4, 9, 7, 1, 7, 9, 7, 1, 3 */ {0, 0, 1, 1, 0, 1, 2, 1},
    /* 180:  , ,2, ,4,5, ,7: 10, 5, 6, 1, 2, 9,11, 9, 2, 9,11, 8 */ {0, 0, 1, 0, 1, 1, 2, 1},
    /* 120:  , , ,3,4,5,6, : 11, 6, 7, 2, 3,10, 8,10, 3,10, 8, 9 */ {0, 0, 0, 1, 1, 1, 1, 2},
};
static const signed char triangleRegion12_1_1_[24][8] = {
    /* 135: 0,1,2, , , , ,7:  3, 2,11,10, 7, 6, 7,10, 8, 9, 8,10 */ {0, 1, 0, 2, 0, 2, 0, 2},
    /*  75: 0,1, ,3, , ,6, :  2, 1,10, 9, 6, 5, 6, 9,11, 8,11, 9 */ {0, 1, 0, 2, 0, 2, 0, 2},
    /*  83: 0,1, , ,4, ,6, :  9, 4, 5, 7,10, 6,10, 7, 1, 3, 1, 7 */ {0, 2, 0, 1, 0, 1, 0, 1},
    /* 163: 0,1, , , ,5, ,7:  7, 4, 8, 6,11, 5, 3, 5,11, 5, 3, 1 */ {0, 2, 0, 1, 0, 1, 0, 1},
    /*  45: 0, ,2,3, ,5, , :  1, 0, 9, 8, 5, 4, 5, 8,10,11,10, 8 */ {0, 1, 0, 2, 0, 2, 0, 2},
    /*  53: 0, ,2, ,4,5, , :  1, 0, 9, 2,10, 3, 5, 3,10, 3, 5, 7 */ {0, 1, 0, 2, 0, 2, 0, 2},
    /* 149: 0, ,2, ,4, , ,7: 11, 3, 2, 0,10, 1,10, 0, 6, 4, 6, 0 */ {0, 2, 0, 1, 0, 1, 0, 1},
    /* 101: 0, ,2, , ,5,6, :  9, 1, 0, 2, 8, 3, 8, 2, 4, 6, 4, 2 */ {0, 1, 0, 2, 0, 2, 0, 2},
    /* 197: 0, ,2, , , ,6,7:  3, 2,11, 0, 8, 1, 7, 1, 8, 1, 7, 5 */ {0, 2, 0, 1, 0, 1, 0, 1},
    /*  89: 0, , ,3,4, ,6, :  6, 7,11, 5,10, 4, 2, 4,10, 4, 2, 0 */ {0, 2, 0, 1, 0, 1, 0, 1},
    /* 169: 0, , ,3, ,5, ,7:  8, 7, 4, 6, 9, 5, 9, 6, 0, 2, 0, 6 */ {0, 2, 0, 1, 0, 1, 0, 1},
    /* 225: 0, , , , ,5,6,7:  8, 7, 4, 3, 0,11, 9,11, 0,11, 9,10 */ {0, 2, 0, 1, 0, 1, 0, 1},
    /*  30:  ,1,2,3,4, , , :  0, 3, 8,11, 4, 7, 4,11, 9,10, 9,11 */ {1, 0, 1, 2, 1, 2, 1, 2},
    /*  86:  ,1,2, ,4, ,6, :  4, 5, 9, 7, 8, 6, 0, 6, 8, 6, 0, 2 */ {1, 2, 1, 0, 1, 0, 1, 0},
    /* 166:  ,1,2, , ,5, ,7: 10, 5, 6, 4,11, 7,11, 4, 2, 0, 2, 4 */ {1, 2, 1, 0, 1, 0, 1, 0},
    /*  58:  ,1, ,3,4,5, , :  8, 0, 3, 1,11, 2,11, 1, 7, 5, 7, 1 */ {1, 0, 1, 2, 1, 2, 1, 2},
    /* 154:  ,1, ,3,4, , ,7:  0, 3, 8, 1, 9, 2, 4, 2, 9, 2, 4, 6 */ {1, 0, 1, 2, 1, 2, 1, 2},
    /* 106:  ,1, ,3, ,5,6, :  2, 1,10, 3,11, 0, 6, 0,11, 0, 6, 4 */ {1, 2, 1, 0, 1, 0, 1, 0},
    /* 202:  ,1, ,3, , ,6,7: 10, 2, 1, 3, 9, 0, 9, 3, 5, 7, 5, 3 */ {1, 2, 1, 0, 1, 0, 1, 0},
    /* 210:  ,1, , ,4, ,6,7:  9, 4, 5, 0, 1, 8,10, 8, 1, 8,10,11 */ {1, 2, 1, 0, 1, 0, 1, 0},
    /*  92:  , ,2,3,4, ,6, : 11, 6, 7, 5, 8, 4, 8, 5, 3, 1, 3, 5 */ {1, 2, 1, 0, 1, 0, 1, 0},
    /* 172:  , ,2,3, ,5, ,7:  5, 6,10, 4, 9, 7, 1, 7, 9, 7, 1, 3 */ {1, 2, 1, 0, 1, 0, 1, 0},
    /* 180:  , ,2, ,4,5, ,7: 10, 5, 6, 1, 2, 9,11, 9, 2, 9,11, 8 */ {1, 2, 1, 0, 1, 0, 1, 0},
    /* 120:  , , ,3,4,5,6, : 11, 6, 7, 2, 3,10, 8,10, 3,10, 8, 9 */ {1, 2, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion12_1_2[24][8] = {
    /* 135: 0,1,2, , , , ,7:  7, 3,11, 3, 7, 8, 9, 8, 7, 6, 9, 7, 9, 6,10, 2,10, 6,11, 2, 6, 2,11, 3 */ {0, 0, 0, 1, 2, 2, 2, 0},
    /*  75: 0,1, ,3, , ,6, :  6, 2,10, 2, 6,11, 8,11, 6, 5, 8, 6, 8, 5, 9, 1, 9, 5,10, 1, 5, 1,10, 2 */ {0, 0, 1, 0, 2, 2, 0, 2},
    /*  83: 0,1, , ,4, ,6, : 10, 9, 5, 9,10, 1, 3, 1,10, 6, 3,10, 3, 6, 7, 4, 7, 6, 5, 4, 6, 4, 5, 9 */ {0, 0, 1, 1, 0, 2, 0, 1},
    /* 163: 0,1, , , ,5, ,7:  7, 8,11, 3,11, 8,11, 3, 1,11, 1, 6, 5, 6, 1, 6, 5, 4, 6, 4, 7, 8, 7, 4 */ {0, 0, 1, 1, 2, 0, 1, 0},
    /*  45: 0, ,2,3, ,5, , :  5, 1, 9, 1, 5,10,11,10, 5, 4,11, 5,11, 4, 8, 0, 8, 4, 9, 0, 4, 0, 9, 1 */ {0, 1, 0, 0, 2, 0, 2, 2},
    /*  53: 0, ,2, ,4,5, , :  1, 9,10, 5,10, 9,10, 5, 7,10, 7, 2, 3, 2, 7, 2, 3, 0, 2, 0, 1, 9, 1, 0 */ {0, 1, 0, 2, 0, 0, 2, 2},
    /* 149: 0, ,2, ,4, , ,7: 10,11, 2,11,10, 6, 4, 6,10, 1, 4,10, 4, 1, 0, 3, 0, 1, 2, 3, 1, 3, 2,11 */ {0, 1, 0, 2, 0, 1, 1, 0},
    /* 101: 0, ,2, , ,5,6, :  8, 9, 0, 9, 8, 4, 6, 4, 8, 3, 6, 8, 6, 3, 2, 1, 2, 3, 0, 1, 3, 1, 0, 9 */ {0, 1, 0, 2, 2, 0, 0, 2},
    /* 197: 0, ,2, , , ,6,7:  3,11, 8, 7, 8,11, 8, 7, 5, 8, 5, 0, 1, 0, 5, 0, 1, 2, 0, 2, 3,11, 3, 2 */ {0, 1, 0, 2, 1, 1, 0, 0},
    /*  89: 0, , ,3,4, ,6, :  6,11,10, 2,10,11,10, 2, 0,10, 0, 5, 4, 5, 0, 5, 4, 7, 5, 7, 6,11, 6, 7 */ {0, 1, 1, 0, 0, 1, 0, 2},
    /* 169: 0, , ,3, ,5, ,7:  9, 8, 4, 8, 9, 0, 2, 0, 9, 5, 2, 9, 2, 5, 6, 7, 6, 5, 4, 7, 5, 7, 4, 8 */ {0, 1, 1, 0, 2, 0, 1, 0},
    /* 225: 0, , , , ,5,6,7:  8, 4, 0, 9, 0, 4, 0, 9,10, 0,10, 3,11, 3,10, 3,11, 7, 3, 7, 8, 4, 8, 7 */ {0, 1, 1, 1, 2, 0, 0, 0},
    /*  30:  ,1,2,3,4, , , :  4, 0, 8, 0, 4, 9,10, 9, 4, 7,10, 4,10, 7,11, 3,11, 7, 8, 3, 7, 3, 8, 0 */ {0, 1, 1, 1, 1, 2, 2, 2},
    /*  86:  ,1,2, ,4, ,6, :  4, 9, 8, 0, 8, 9, 8, 0, 2, 8, 2, 7, 6, 7, 2, 7, 6, 5, 7, 5, 4, 9, 4, 5 */ {0, 1, 1, 0, 1, 2, 1, 0},
    /* 166:  ,1,2, , ,5, ,7: 11,10, 6,10,11, 2, 0, 2,11, 7, 0,11, 0, 7, 4, 5, 4, 7, 6, 5, 7, 5, 6,10 */ {0, 1, 1, 0, 0, 1, 2, 1},
    /*  58:  ,1, ,3,4,5, , : 11, 8, 3, 8,11, 7, 5, 7,11, 2, 5,11, 5, 2, 1, 0, 1, 2, 3, 0, 2, 0, 3, 8 */ {0, 1, 2, 1, 1, 1, 2, 2},
    /* 154:  ,1, ,3,4, , ,7:  0, 8, 9, 4, 9, 8, 9, 4, 6, 9, 6, 1, 2, 1, 6, 1, 2, 3, 1, 3, 0, 8, 0, 3 */ {0, 1, 2, 1, 1, 2, 2, 1},
    /* 106:  ,1, ,3, ,5,6, :  2,10,11, 6,11,10,11, 6, 4,11, 4, 3, 0, 3, 4, 3, 0, 1, 3, 1, 2,10, 2, 1 */ {0, 1, 2, 1, 0, 1, 1, 0},
    /* 202:  ,1, ,3, , ,6,7:  9,10, 1,10, 9, 5, 7, 5, 9, 0, 7, 9, 7, 0, 3, 2, 3, 0, 1, 2, 0, 2, 1,10 */ {0, 1, 2, 1, 0, 0, 1, 1},
    /* 210:  ,1, , ,4, ,6,7:  9, 5, 1,10, 1, 5, 1,10,11, 1,11, 0, 8, 0,11, 0, 8, 4, 0, 4, 9, 5, 9, 4 */ {0, 1, 0, 0, 1, 2, 1, 1},
    /*  92:  , ,2,3,4, ,6, :  8,11, 7,11, 8, 3, 1, 3, 8, 4, 1, 8, 1, 4, 5, 6, 5, 4, 7, 6, 4, 6, 7,11 */ {0, 0, 1, 1, 1, 0, 1, 2},
    /* 172:  , ,2,3, ,5, ,7:  5,10, 9, 1, 9,10, 9, 1, 3, 9, 3, 4, 7, 4, 3, 4, 7, 6, 4, 6, 5,10, 5, 6 */ {0, 0, 1, 1, 0, 1, 2, 1},
    /* 180:  , ,2, ,4,5, ,7: 10, 6, 2,11, 2, 6, 2,11, 8, 2, 8, 1, 9, 1, 8, 1, 9, 5, 1, 5,10, 6,10, 5 */ {0, 0, 1, 0, 1, 1, 2, 1},
    /* 120:  , , ,3,4,5,6, : 11, 7, 3, 8, 3, 7, 3, 8, 9, 3, 9, 2,10, 2, 9, 2,10, 6, 2, 6,11, 7,11, 6 */ {0, 0, 0, 1, 1, 1, 1, 2},
};
static const signed char triangleRegion12_1_2[24][16] = {
    /* 135: 0,1,2, , , , ,7:  7, 3,11, 3, 7, 8, 9, 8, 7, 6, 9, 7, 9, 6,10, 2,10, 6,11, 2, 6, 2,11, 3 */
    {0, 0, 0, 0, 0, 2, 0, 2, 0, 2, 0, 0, 0, 0, 0, 1},
    /*  75: 0,1, ,3, , ,6, :  6, 2,10, 2, 6,11, 8,11, 6, 5, 8, 6, 8, 5, 9, 1, 9, 5,10, 1, 5, 1,10, 2 */
    {0, 0, 0, 0, 0, 2, 0, 2, 0, 2, 0, 0, 0, 0, 0, 1},
    /*  83: 0,1, , ,4, ,6, : 10, 9, 5, 9,10, 1, 3, 1,10, 6, 3,10, 3, 6, 7, 4, 7, 6, 5, 4, 6, 4, 5, 9 */
    {0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 2},
    /* 163: 0,1, , , ,5, ,7:  7, 8,11, 3,11, 8,11, 3, 1,11, 1, 6, 5, 6, 1, 6, 5, 4, 6, 4, 7, 8, 7, 4 */
    {0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 2},
    /*  45: 0, ,2,3, ,5, , :  5, 1, 9, 1, 5,10,11,10, 5, 4,11, 5,11, 4, 8, 0, 8, 4, 9, 0, 4, 0, 9, 1 */
    {0, 0, 0, 0, 0, 2, 0, 2, 0, 2, 0, 0, 0, 0, 0, 1},
    /*  53: 0, ,2, ,4,5, , :  1, 9,10, 5,10, 9,10, 5, 7,10, 7, 2, 3, 2, 7, 2, 3, 0, 2, 0, 1, 9, 1, 0 */
    {0, 0, 0, 0, 0, 2, 0, 2, 0, 2, 0, 0, 0, 0, 0, 1},
    /* 149: 0, ,2, ,4, , ,7: 10,11, 2,11,10, 6, 4, 6,10, 1, 4,10, 4, 1, 0, 3, 0, 1, 2, 3, 1, 3, 2,11 */
    {0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 2},
    /* 101: 0, ,2, , ,5,6, :  8, 9, 0, 9, 8, 4, 6, 4, 8, 3, 6, 8, 6, 3, 2, 1, 2, 3, 0, 1, 3, 1, 0, 9 */
    {0, 0, 0, 0, 0, 2, 0, 2, 0, 2, 0, 0, 0, 0, 0, 1},
    /* 197: 0, ,2, , , ,6,7:  3,11, 8, 7, 8,11, 8, 7, 5, 8, 5, 0, 1, 0, 5, 0, 1, 2, 0, 2, 3,11, 3, 2 */
    {0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 2},
    /*  89: 0, , ,3,4, ,6, :  6,11,10, 2,10,11,10, 2, 0,10, 0, 5, 4, 5, 0, 5, 4, 7, 5, 7, 6,11, 6, 7 */
    {0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 2},
    /* 169: 0, , ,3, ,5, ,7:  9, 8, 4, 8, 9, 0, 2, 0, 9, 5, 2, 9, 2, 5, 6, 7, 6, 5, 4, 7, 5, 7, 4, 8 */
    {0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 2},
    /* 225: 0, , , , ,5,6,7:  8, 4, 0, 9, 0, 4, 0, 9,10, 0,10, 3,11, 3,10, 3,11, 7, 3, 7, 8, 4, 8, 7 */
    {0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 2},
    /*  30:  ,1,2,3,4, , , :  4, 0, 8, 0, 4, 9,10, 9, 4, 7,10, 4,10, 7,11, 3,11, 7, 8, 3, 7, 3, 8, 0 */
    {1, 1, 1, 1, 1, 2, 1, 2, 1, 2, 1, 1, 1, 1, 1, 0},
    /*  86:  ,1,2, ,4, ,6, :  4, 9, 8, 0, 8, 9, 8, 0, 2, 8, 2, 7, 6, 7, 2, 7, 6, 5, 7, 5, 4, 9, 4, 5 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
    /* 166:  ,1,2, , ,5, ,7: 11,10, 6,10,11, 2, 0, 2,11, 7, 0,11, 0, 7, 4, 5, 4, 7, 6, 5, 7, 5, 6,10 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
    /*  58:  ,1, ,3,4,5, , : 11, 8, 3, 8,11, 7, 5, 7,11, 2, 5,11, 5, 2, 1, 0, 1, 2, 3, 0, 2, 0, 3, 8 */
    {1, 1, 1, 1, 1, 2, 1, 2, 1, 2, 1, 1, 1, 1, 1, 0},
    /* 154:  ,1, ,3,4, , ,7:  0, 8, 9, 4, 9, 8, 9, 4, 6, 9, 6, 1, 2, 1, 6, 1, 2, 3, 1, 3, 0, 8, 0, 3 */
    {1, 1, 1, 1, 1, 2, 1, 2, 1, 2, 1, 1, 1, 1, 1, 0},
    /* 106:  ,1, ,3, ,5,6, :  2,10,11, 6,11,10,11, 6, 4,11, 4, 3, 0, 3, 4, 3, 0, 1, 3, 1, 2,10, 2, 1 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
    /* 202:  ,1, ,3, , ,6,7:  9,10, 1,10, 9, 5, 7, 5, 9, 0, 7, 9, 7, 0, 3, 2, 3, 0, 1, 2, 0, 2, 1,10 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
    /* 210:  ,1, , ,4, ,6,7:  9, 5, 1,10, 1, 5, 1,10,11, 1,11, 0, 8, 0,11, 0, 8, 4, 0, 4, 9, 5, 9, 4 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
    /*  92:  , ,2,3,4, ,6, :  8,11, 7,11, 8, 3, 1, 3, 8, 4, 1, 8, 1, 4, 5, 6, 5, 4, 7, 6, 4, 6, 7,11 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
    /* 172:  , ,2,3, ,5, ,7:  5,10, 9, 1, 9,10, 9, 1, 3, 9, 3, 4, 7, 4, 3, 4, 7, 6, 4, 6, 5,10, 5, 6 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
    /* 180:  , ,2, ,4,5, ,7: 10, 6, 2,11, 2, 6, 2,11, 8, 2, 8, 1, 9, 1, 8, 1, 9, 5, 1, 5,10, 6,10, 5 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
    /* 120:  , , ,3,4,5,6, : 11, 7, 3, 8, 3, 7, 3, 8, 9, 3, 9, 2,10, 2, 9, 2,10, 6, 2, 6,11, 7,11, 6 */
    {1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 2},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion12_2[24][8] = {
    /* 135: 0,1,2, , , , ,7:  9, 8,12,10, 9,12, 2,10,12, 3, 2,12,11, 3,12, 6,11,12, 7, 6,12, 8, 7,12 */ {0, 0, 0, 1, 1, 1, 1, 0},
    /*  75: 0,1, ,3, , ,6, :  8,11,12, 9, 8,12, 1, 9,12, 2, 1,12,10, 2,12, 5,10,12, 6, 5,12,11, 6,12 */ {0, 0, 1, 0, 1, 1, 0, 1},
    /*  83: 0,1, , ,4, ,6, :  3, 1,12, 7, 3,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12,10, 6,12, 1,10,12 */ {0, 0, 1, 1, 0, 1, 0, 1},
    /* 163: 0,1, , , ,5, ,7: 12, 3, 1,12, 1, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3 */ {0, 0, 1, 1, 1, 0, 1, 0},
    /*  45: 0, ,2,3, ,5, , : 11,10,12, 8,11,12, 0, 8,12, 1, 0,12, 9, 1,12, 4, 9,12, 5, 4,12,10, 5,12 */ {0, 1, 0, 0, 1, 0, 1, 1},
    /*  53: 0, ,2, ,4,5, , : 12, 5, 7,12, 7, 3,12, 3, 2,12, 2,10,12,10, 1,12, 1, 0,12, 0, 9,12, 9, 5 */ {0, 1, 0, 1, 0, 0, 1, 1},
    /* 149: 0, ,2, ,4, , ,7:  4, 6,12, 0, 4,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12,11, 3,12, 6,11,12 */ {0, 1, 0, 1, 0, 1, 1, 0},
    /* 101: 0, ,2, , ,5,6, :  6, 4,12, 2, 6,12, 3, 2,12, 8, 3,12, 0, 8,12, 1, 0,12, 9, 1,12, 4, 9,12 */ {0, 1, 0, 1, 1, 0, 0, 1},
    /* 197: 0, ,2, , , ,6,7: 12, 7, 5,12, 5, 1,12, 1, 0,12, 0, 8,12, 8, 3,12, 3, 2,12, 2,11,12,11, 7 */ {0, 1, 0, 1, 1, 1, 0, 0},
    /*  89: 0, , ,3,4, ,6, : 12, 2, 0,12, 0, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7,11,12,11, 2 */ {0, 1, 1, 0, 0, 1, 0, 1},
    /* 169: 0, , ,3, ,5, ,7:  2, 0,12, 6, 2,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 9, 5,12, 0, 9,12 */ {0, 1, 1, 0, 1, 0, 1, 0},
    /* 225: 0, , , , ,5,6,7: 12, 9,10,12,10,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3,12, 3, 0,12, 0, 9 */ {0, 1, 1, 1, 1, 0, 0, 0},
    /*  30:  ,1,2,3,4, , , : 10, 9,12,11,10,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12, 0, 3,12, 9, 0,12 */ {0, 1, 1, 1, 1, 0, 0, 0},
    /*  86:  ,1,2, ,4, ,6, : 12, 0, 2,12, 2, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 9,12, 9, 0 */ {0, 1, 1, 0, 1, 0, 1, 0},
    /* 166:  ,1,2, , ,5, ,7:  0, 2,12, 4, 0,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12,11, 7,12, 2,11,12 */ {0, 1, 1, 0, 0, 1, 0, 1},
    /*  58:  ,1, ,3,4,5, , :  5, 7,12, 1, 5,12, 0, 1,12, 8, 0,12, 3, 8,12, 2, 3,12,11, 2,12, 7,11,12 */ {0, 1, 0, 1, 1, 1, 0, 0},
    /* 154:  ,1, ,3,4, , ,7: 12, 4, 6,12, 6, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1,12, 1, 9,12, 9, 4 */ {0, 1, 0, 1, 1, 0, 0, 1},
    /* 106:  ,1, ,3, ,5,6, : 12, 6, 4,12, 4, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3,12, 3,11,12,11, 6 */ {0, 1, 0, 1, 0, 1, 1, 0},
    /* 202:  ,1, ,3, , ,6,7:  7, 5,12, 3, 7,12, 2, 3,12,10, 2,12, 1,10,12, 0, 1,12, 9, 0,12, 5, 9,12 */ {0, 1, 0, 1, 0, 0, 1, 1},
    /* 210:  ,1, , ,4, ,6,7: 12,10,11,12,11, 8,12, 8, 0,12, 0, 1,12, 1, 9,12, 9, 4,12, 4, 5,12, 5,10 */ {0, 1, 0, 0, 1, 0, 1, 1},
    /*  92:  , ,2,3,4, ,6, :  1, 3,12, 5, 1,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12 */ {0, 0, 1, 1, 1, 0, 1, 0},
    /* 172:  , ,2,3, ,5, ,7: 12, 1, 3,12, 3, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6,10,12,10, 1 */ {0, 0, 1, 1, 0, 1, 0, 1},
    /* 180:  , ,2, ,4,5, ,7: 12,11, 8,12, 8, 9,12, 9, 1,12, 1, 2,12, 2,10,12,10, 5,12, 5, 6,12, 6,11 */ {0, 0, 1, 0, 1, 1, 0, 1},
    /* 120:  , , ,3,4,5,6, : 12, 8, 9,12, 9,10,12,10, 2,12, 2, 3,12, 3,11,12,11, 6,12, 6, 7,12, 7, 8 */ {0, 0, 0, 1, 1, 1, 1, 0},
};
static const signed char triangleRegion12_2[24][16] = {
    /* 135: 0,1,2, , , , ,7:  9, 8,12,10, 9,12, 2,10,12, 3, 2,12,11, 3,12, 6,11,12, 7, 6,12, 8, 7,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  75: 0,1, ,3, , ,6, :  8,11,12, 9, 8,12, 1, 9,12, 2, 1,12,10, 2,12, 5,10,12, 6, 5,12,11, 6,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  83: 0,1, , ,4, ,6, :  3, 1,12, 7, 3,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12,10, 6,12, 1,10,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 163: 0,1, , , ,5, ,7: 12, 3, 1,12, 1, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  45: 0, ,2,3, ,5, , : 11,10,12, 8,11,12, 0, 8,12, 1, 0,12, 9, 1,12, 4, 9,12, 5, 4,12,10, 5,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  53: 0, ,2, ,4,5, , : 12, 5, 7,12, 7, 3,12, 3, 2,12, 2,10,12,10, 1,12, 1, 0,12, 0, 9,12, 9, 5 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 149: 0, ,2, ,4, , ,7:  4, 6,12, 0, 4,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12,11, 3,12, 6,11,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 101: 0, ,2, , ,5,6, :  6, 4,12, 2, 6,12, 3, 2,12, 8, 3,12, 0, 8,12, 1, 0,12, 9, 1,12, 4, 9,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 197: 0, ,2, , , ,6,7: 12, 7, 5,12, 5, 1,12, 1, 0,12, 0, 8,12, 8, 3,12, 3, 2,12, 2,11,12,11, 7 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  89: 0, , ,3,4, ,6, : 12, 2, 0,12, 0, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7,11,12,11, 2 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 169: 0, , ,3, ,5, ,7:  2, 0,12, 6, 2,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 9, 5,12, 0, 9,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 225: 0, , , , ,5,6,7: 12, 9,10,12,10,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3,12, 3, 0,12, 0, 9 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  30:  ,1,2,3,4, , , : 10, 9,12,11,10,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12, 0, 3,12, 9, 0,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  86:  ,1,2, ,4, ,6, : 12, 0, 2,12, 2, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 9,12, 9, 0 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 166:  ,1,2, , ,5, ,7:  0, 2,12, 4, 0,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12,11, 7,12, 2,11,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  58:  ,1, ,3,4,5, , :  5, 7,12, 1, 5,12, 0, 1,12, 8, 0,12, 3, 8,12, 2, 3,12,11, 2,12, 7,11,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 154:  ,1, ,3,4, , ,7: 12, 4, 6,12, 6, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1,12, 1, 9,12, 9, 4 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 106:  ,1, ,3, ,5,6, : 12, 6, 4,12, 4, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3,12, 3,11,12,11, 6 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 202:  ,1, ,3, , ,6,7:  7, 5,12, 3, 7,12, 2, 3,12,10, 2,12, 1,10,12, 0, 1,12, 9, 0,12, 5, 9,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 210:  ,1, , ,4, ,6,7: 12,10,11,12,11, 8,12, 8, 0,12, 0, 1,12, 1, 9,12, 9, 4,12, 4, 5,12, 5,10 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  92:  , ,2,3,4, ,6, :  1, 3,12, 5, 1,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 172:  , ,2,3, ,5, ,7: 12, 1, 3,12, 3, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6,10,12,10, 1 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 180:  , ,2, ,4,5, ,7: 12,11, 8,12, 8, 9,12, 9, 1,12, 1, 2,12, 2,10,12,10, 5,12, 5, 6,12, 6,11 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 120:  , , ,3,4,5,6, : 12, 8, 9,12, 9,10,12,10, 2,12, 2, 3,12, 3,11,12,11, 6,12, 6, 7,12, 7, 8 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion12_2_[24][8] = {
    /* 135: 0,1,2, , , , ,7: 12, 2,11,12,11, 7,12, 7, 6,12, 6,10,12,10, 9,12, 9, 8,12, 8, 3,12, 3, 2 */ {0, 0, 0, 1, 1, 1, 1, 0},
    /*  75: 0,1, ,3, , ,6, : 12, 1,10,12,10, 6,12, 6, 5,12, 5, 9,12, 9, 8,12, 8,11,12,11, 2,12, 2, 1 */ {0, 0, 1, 0, 1, 1, 0, 1},
    /*  83: 0,1, , ,4, ,6, : 12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7, 3,12, 3, 1,12, 1, 9,12, 9, 4 */ {0, 0, 1, 1, 0, 1, 0, 1},
    /* 163: 0,1, , , ,5, ,7:  7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 1, 5,12, 3, 1,12,11, 3,12, 6,11,12 */ {0, 0, 1, 1, 1, 0, 1, 0},
    /*  45: 0, ,2,3, ,5, , : 12, 0, 9,12, 9, 5,12, 5, 4,12, 4, 8,12, 8,11,12,11,10,12,10, 1,12, 1, 0 */ {0, 1, 0, 0, 1, 0, 1, 1},
    /*  53: 0, ,2, ,4,5, , :  1, 2,12, 9, 1,12, 0, 9,12, 3, 0,12, 7, 3,12, 5, 7,12,10, 5,12, 2,10,12 */ {0, 1, 0, 1, 0, 0, 1, 1},
    /* 149: 0, ,2, ,4, , ,7: 12, 1, 2,12, 2,11,12,11, 3,12, 3, 0,12, 0, 4,12, 4, 6,12, 6,10,12,10, 1 */ {0, 1, 0, 1, 0, 1, 1, 0},
    /* 101: 0, ,2, , ,5,6, : 12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2,12, 2, 6,12, 6, 4,12, 4, 8,12, 8, 3 */ {0, 1, 0, 1, 1, 0, 0, 1},
    /* 197: 0, ,2, , , ,6,7:  3, 0,12,11, 3,12, 2,11,12, 1, 2,12, 5, 1,12, 7, 5,12, 8, 7,12, 0, 8,12 */ {0, 1, 0, 1, 1, 1, 0, 0},
    /*  89: 0, , ,3,4, ,6, :  6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 0, 4,12, 2, 0,12,10, 2,12, 5,10,12 */ {0, 1, 1, 0, 0, 1, 0, 1},
    /* 169: 0, , ,3, ,5, ,7: 12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6, 2,12, 2, 0,12, 0, 8,12, 8, 7 */ {0, 1, 1, 0, 1, 0, 1, 0},
    /* 225: 0, , , , ,5,6,7:  8, 7,12, 0, 8,12, 3, 0,12,11, 3,12,10,11,12, 9,10,12, 4, 9,12, 7, 4,12 */ {0, 1, 1, 1, 1, 0, 0, 0},
    /*  30:  ,1,2,3,4, , , : 12, 7, 8,12, 8, 0,12, 0, 3,12, 3,11,12,11,10,12,10, 9,12, 9, 4,12, 4, 7 */ {0, 1, 1, 1, 1, 0, 0, 0},
    /*  86:  ,1,2, ,4, ,6, :  4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12, 2, 6,12, 0, 2,12, 8, 0,12, 7, 8,12 */ {0, 1, 1, 0, 1, 0, 1, 0},
    /* 166:  ,1,2, , ,5, ,7: 12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 0,12, 0, 2,12, 2,10,12,10, 5 */ {0, 1, 1, 0, 0, 1, 0, 1},
    /*  58:  ,1, ,3,4,5, , : 12, 0, 3,12, 3,11,12,11, 2,12, 2, 1,12, 1, 5,12, 5, 7,12, 7, 8,12, 8, 0 */ {0, 1, 0, 1, 1, 1, 0, 0},
    /* 154:  ,1, ,3,4, , ,7:  0, 3,12, 9, 0,12, 1, 9,12, 2, 1,12, 6, 2,12, 4, 6,12, 8, 4,12, 3, 8,12 */ {0, 1, 0, 1, 1, 0, 0, 1},
    /* 106:  ,1, ,3, ,5,6, :  2, 1,12,11, 2,12, 3,11,12, 0, 3,12, 4, 0,12, 6, 4,12,10, 6,12, 1,10,12 */ {0, 1, 0, 1, 0, 1, 1, 0},
    /* 202:  ,1, ,3, , ,6,7: 12, 2, 1,12, 1, 9,12, 9, 0,12, 0, 3,12, 3, 7,12, 7, 5,12, 5,10,12,10, 2 */ {0, 1, 0, 1, 0, 0, 1, 1},
    /* 210:  ,1, , ,4, ,6,7:  9, 0,12, 5, 9,12, 4, 5,12, 8, 4,12,11, 8,12,10,11,12, 1,10,12, 0, 1,12 */ {0, 1, 0, 0, 1, 0, 1, 1},
    /*  92:  , ,2,3,4, ,6, : 12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 1,12, 1, 3,12, 3,11,12,11, 6 */ {0, 0, 1, 1, 1, 0, 1, 0},
    /* 172:  , ,2,3, ,5, ,7:  5, 4,12,10, 5,12, 6,10,12, 7, 6,12, 3, 7,12, 1, 3,12, 9, 1,12, 4, 9,12 */ {0, 0, 1, 1, 0, 1, 0, 1},
    /* 180:  , ,2, ,4,5, ,7: 10, 1,12, 6,10,12, 5, 6,12, 9, 5,12, 8, 9,12,11, 8,12, 2,11,12, 1, 2,12 */ {0, 0, 1, 0, 1, 1, 0, 1},
    /* 120:  , , ,3,4,5,6, : 11, 2,12, 7,11,12, 6, 7,12,10, 6,12, 9,10,12, 8, 9,12, 3, 8,12, 2, 3,12 */ {0, 0, 0, 1, 1, 1, 1, 0},
};
static const signed char triangleRegion12_2_[24][16] = {
    /* 135: 0,1,2, , , , ,7: 12, 2,11,12,11, 7,12, 7, 6,12, 6,10,12,10, 9,12, 9, 8,12, 8, 3,12, 3, 2 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  75: 0,1, ,3, , ,6, : 12, 1,10,12,10, 6,12, 6, 5,12, 5, 9,12, 9, 8,12, 8,11,12,11, 2,12, 2, 1 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  83: 0,1, , ,4, ,6, : 12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7, 3,12, 3, 1,12, 1, 9,12, 9, 4 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 163: 0,1, , , ,5, ,7:  7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 1, 5,12, 3, 1,12,11, 3,12, 6,11,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  45: 0, ,2,3, ,5, , : 12, 0, 9,12, 9, 5,12, 5, 4,12, 4, 8,12, 8,11,12,11,10,12,10, 1,12, 1, 0 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  53: 0, ,2, ,4,5, , :  1, 2,12, 9, 1,12, 0, 9,12, 3, 0,12, 7, 3,12, 5, 7,12,10, 5,12, 2,10,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 149: 0, ,2, ,4, , ,7: 12, 1, 2,12, 2,11,12,11, 3,12, 3, 0,12, 0, 4,12, 4, 6,12, 6,10,12,10, 1 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 101: 0, ,2, , ,5,6, : 12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2,12, 2, 6,12, 6, 4,12, 4, 8,12, 8, 3 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 197: 0, ,2, , , ,6,7:  3, 0,12,11, 3,12, 2,11,12, 1, 2,12, 5, 1,12, 7, 5,12, 8, 7,12, 0, 8,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  89: 0, , ,3,4, ,6, :  6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 0, 4,12, 2, 0,12,10, 2,12, 5,10,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 169: 0, , ,3, ,5, ,7: 12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6, 2,12, 2, 0,12, 0, 8,12, 8, 7 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /* 225: 0, , , , ,5,6,7:  8, 7,12, 0, 8,12, 3, 0,12,11, 3,12,10,11,12, 9,10,12, 4, 9,12, 7, 4,12 */
    {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    /*  30:  ,1,2,3,4, , , : 12, 7, 8,12, 8, 0,12, 0, 3,12, 3,11,12,11,10,12,10, 9,12, 9, 4,12, 4, 7 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  86:  ,1,2, ,4, ,6, :  4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12, 2, 6,12, 0, 2,12, 8, 0,12, 7, 8,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 166:  ,1,2, , ,5, ,7: 12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 0,12, 0, 2,12, 2,10,12,10, 5 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  58:  ,1, ,3,4,5, , : 12, 0, 3,12, 3,11,12,11, 2,12, 2, 1,12, 1, 5,12, 5, 7,12, 7, 8,12, 8, 0 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 154:  ,1, ,3,4, , ,7:  0, 3,12, 9, 0,12, 1, 9,12, 2, 1,12, 6, 2,12, 4, 6,12, 8, 4,12, 3, 8,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 106:  ,1, ,3, ,5,6, :  2, 1,12,11, 2,12, 3,11,12, 0, 3,12, 4, 0,12, 6, 4,12,10, 6,12, 1,10,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 202:  ,1, ,3, , ,6,7: 12, 2, 1,12, 1, 9,12, 9, 0,12, 0, 3,12, 3, 7,12, 7, 5,12, 5,10,12,10, 2 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 210:  ,1, , ,4, ,6,7:  9, 0,12, 5, 9,12, 4, 5,12, 8, 4,12,11, 8,12,10,11,12, 1,10,12, 0, 1,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /*  92:  , ,2,3,4, ,6, : 12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 1,12, 1, 3,12, 3,11,12,11, 6 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 172:  , ,2,3, ,5, ,7:  5, 4,12,10, 5,12, 6,10,12, 7, 6,12, 3, 7,12, 1, 3,12, 9, 1,12, 4, 9,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 180:  , ,2, ,4,5, ,7: 10, 1,12, 6,10,12, 5, 6,12, 9, 5,12, 8, 9,12,11, 8,12, 2,11,12, 1, 2,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    /* 120:  , , ,3,4,5,6, : 11, 2,12, 7,11,12, 6, 7,12,10, 6,12, 9,10,12, 8, 9,12, 3, 8,12, 2, 3,12 */
    {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_1[2][8] = {
    /* 165: 0, ,2, , ,5, ,7: 11, 7, 6, 1, 2,10, 8, 3, 0, 9, 5, 4 */ {0, 1, 2, 1, 1, 3, 1, 4},
    /*  90:  ,1, ,3,4, ,6, :  8, 4, 7, 2, 3,11, 9, 0, 1,10, 6, 5 */ {0, 1, 0, 2, 3, 0, 4, 0},
};
static const signed char triangleRegion13_1[2][8] = {
    /* 165: 0, ,2, , ,5, ,7: 11, 7, 6, 1, 2,10, 8, 3, 0, 9, 5, 4 */ {4, 1, 2, 1, 0, 1, 3, 1},
    /*  90:  ,1, ,3,4, ,6, :  8, 4, 7, 2, 3,11, 9, 0, 1,10, 6, 5 */ {3, 0, 2, 0, 1, 0, 4, 0},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_1_[2][8] = {
    /* 165: 0, ,2, , ,5, ,7:  7, 4, 8,11, 3, 2, 1, 0, 9, 5, 6,10 */ {0, 1, 0, 2, 3, 0, 4, 0},
    /*  90:  ,1, ,3,4, ,6, :  6, 7,11,10, 2, 1, 0, 3, 8, 4, 5, 9 */ {0, 1, 2, 1, 1, 3, 1, 4},
};
static const signed char triangleRegion13_1_[2][8] = {
    /* 165: 0, ,2, , ,5, ,7:  7, 4, 8,11, 3, 2, 1, 0, 9, 5, 6,10 */ {0, 3, 0, 2, 0, 1, 0, 4},
    /*  90:  ,1, ,3,4, ,6, :  6, 7,11,10, 2, 1, 0, 3, 8, 4, 5, 9 */ {1, 4, 1, 2, 1, 0, 1, 3},
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_2[2][6][8] = {
    {
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10,11, 7, 6, 3, 4, 8, 4, 3, 5, 0, 5, 3, 5, 0, 9 */ {0, 1, 2, 1, 1, 0, 1, 3},
        /* 165: 0, ,2, , ,5, ,7:  8, 3, 0,11, 7, 6, 9, 1, 4, 2, 4, 1, 4, 2, 5,10, 5, 2 */ {0, 1, 2, 1, 1, 2, 1, 3},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4, 8, 3, 0, 1, 6,10, 6, 1, 7, 2, 7, 1, 7, 2,11 */ {0, 1, 2, 1, 1, 3, 1, 2},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4, 1, 2,10,11, 3, 6, 0, 6, 3, 6, 0, 7, 8, 7, 0 */ {0, 1, 2, 1, 1, 3, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4,11, 7, 6, 0,10, 1,10, 0, 8,10, 8, 2, 3, 2, 8 */ {0, 1, 0, 1, 1, 2, 1, 3},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10, 3, 0, 8, 4, 9, 7,11, 7, 9, 5,11, 9,11, 5, 6 */ {0, 1, 2, 1, 1, 3, 1, 3},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11, 8, 4, 7, 0, 5, 9, 5, 0, 6, 1, 6, 0, 6, 1,10 */ {0, 1, 0, 2, 3, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  9, 0, 1, 8, 4, 7,10, 2, 5, 3, 5, 2, 5, 3, 6,11, 6, 3 */ {0, 1, 0, 2, 3, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 9, 0, 1, 2, 7,11, 7, 2, 4, 3, 4, 2, 4, 3, 8 */ {0, 1, 0, 2, 2, 0, 3, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 2, 3,11, 8, 0, 7, 1, 7, 0, 7, 1, 4, 9, 4, 1 */ {0, 1, 0, 2, 1, 0, 3, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 8, 4, 7, 1,11, 2,11, 1, 9,11, 9, 3, 0, 3, 9 */ {0, 1, 0, 1, 2, 0, 3, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11, 0, 1, 9, 5,10, 4, 8, 4,10, 6, 8,10, 8, 6, 7 */ {0, 1, 0, 2, 3, 0, 3, 0},
    },
};
static const signed char triangleRegion13_2[2][6][12] = {
    {
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10,11, 7, 6, 3, 4, 8, 4, 3, 5, 0, 5, 3, 5, 0, 9 */ {2, 1, 3, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  8, 3, 0,11, 7, 6, 9, 1, 4, 2, 4, 1, 4, 2, 5,10, 5, 2 */ {0, 1, 3, 1, 2, 1, 2, 1, 2, 1, 2, 1},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4, 8, 3, 0, 1, 6,10, 6, 1, 7, 2, 7, 1, 7, 2,11 */ {3, 1, 0, 1, 2, 1, 2, 1, 2, 1, 2, 1},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4, 1, 2,10,11, 3, 6, 0, 6, 3, 6, 0, 7, 8, 7, 0 */ {3, 1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4,11, 7, 6, 0,10, 1,10, 0, 8,10, 8, 2, 3, 2, 8 */ {2, 1, 3, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10, 3, 0, 8, 4, 9, 7,11, 7, 9, 5,11, 9,11, 5, 6 */ {2, 1, 0, 1, 3, 1, 3, 1, 3, 1, 3, 1},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11, 8, 4, 7, 0, 5, 9, 5, 0, 6, 1, 6, 0, 6, 1,10 */ {2, 0, 3, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  9, 0, 1, 8, 4, 7,10, 2, 5, 3, 5, 2, 5, 3, 6,11, 6, 3 */ {1, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 9, 0, 1, 2, 7,11, 7, 2, 4, 3, 4, 2, 4, 3, 8 */ {3, 0, 1, 0, 2, 0, 2, 0, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 2, 3,11, 8, 0, 7, 1, 7, 0, 7, 1, 4, 9, 4, 1 */ {3, 0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 8, 4, 7, 1,11, 2,11, 1, 9,11, 9, 3, 0, 3, 9 */ {3, 0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11, 0, 1, 9, 5,10, 4, 8, 4,10, 6, 8,10, 8, 6, 7 */ {2, 0, 1, 0, 3, 0, 3, 0, 3, 0, 3, 0},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_2_[2][6][8] = {
    {
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6,11, 3, 2, 7, 0, 8, 0, 7, 1, 4, 1, 7, 1, 4, 9 */ {0, 1, 0, 2, 1, 0, 3, 0},
        /* 165: 0, ,2, , ,5, ,7: 11, 3, 2, 7, 4, 8, 9, 5, 0, 6, 0, 5, 0, 6, 1,10, 1, 6 */ {0, 1, 0, 2, 3, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  1, 0, 9, 7, 4, 8, 5, 2,10, 2, 5, 3, 6, 3, 5, 3, 6,11 */ {0, 1, 0, 2, 3, 0, 2, 0},
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6, 1, 0, 9,11, 7, 2, 4, 2, 7, 2, 4, 3, 8, 3, 4 */ {0, 1, 0, 2, 2, 0, 3, 0},
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6, 7, 4, 8, 2,11, 1, 9, 1,11, 3, 9,11, 9, 3, 0 */ {0, 1, 0, 1, 2, 0, 3, 0},
        /* 165: 0, ,2, , ,5, ,7: 11, 3, 2, 9, 1, 0, 4,10, 5,10, 4, 8,10, 8, 6, 7, 6, 8 */ {0, 1, 0, 2, 3, 0, 3, 0},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11, 8, 0, 3, 4, 1, 9, 1, 4, 2, 5, 2, 4, 2, 5,10 */ {0, 1, 2, 1, 1, 2, 1, 3},
        /*  90:  ,1, ,3,4, ,6, :  8, 0, 3, 4, 5, 9,10, 6, 1, 7, 1, 6, 1, 7, 2,11, 2, 7 */ {0, 1, 2, 1, 1, 3, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  2, 1,10, 4, 5, 9, 6, 3,11, 3, 6, 0, 7, 0, 6, 0, 7, 8 */ {0, 1, 2, 1, 1, 3, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11, 2, 1,10, 8, 4, 3, 5, 3, 4, 3, 5, 0, 9, 0, 5 */ {0, 1, 2, 1, 1, 0, 1, 3},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11, 4, 5, 9, 3, 8, 2,10, 2, 8, 0,10, 8,10, 0, 1 */ {0, 1, 0, 1, 1, 2, 1, 3},
        /*  90:  ,1, ,3,4, ,6, :  8, 0, 3,10, 2, 1, 5,11, 6,11, 5, 9,11, 9, 7, 4, 7, 9 */ {0, 1, 2, 1, 1, 3, 1, 3},
    },
};
static const signed char triangleRegion13_2_[2][6][12] = {
    {
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6,11, 3, 2, 7, 0, 8, 0, 7, 1, 4, 1, 7, 1, 4, 9 */ {0, 3, 0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7: 11, 3, 2, 7, 4, 8, 9, 5, 0, 6, 0, 5, 0, 6, 1,10, 1, 6 */ {0, 2, 0, 3, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  1, 0, 9, 7, 4, 8, 5, 2,10, 2, 5, 3, 6, 3, 5, 3, 6,11 */ {0, 1, 0, 3, 0, 2, 0, 2, 0, 2, 0, 2},
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6, 1, 0, 9,11, 7, 2, 4, 2, 7, 2, 4, 3, 8, 3, 4 */ {0, 3, 0, 1, 0, 2, 0, 2, 0, 2, 0, 2},
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6, 7, 4, 8, 2,11, 1, 9, 1,11, 3, 9,11, 9, 3, 0 */ {0, 3, 0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7: 11, 3, 2, 9, 1, 0, 4,10, 5,10, 4, 8,10, 8, 6, 7, 6, 8 */ {0, 2, 0, 1, 0, 3, 0, 3, 0, 3, 0, 3},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11, 8, 0, 3, 4, 1, 9, 1, 4, 2, 5, 2, 4, 2, 5,10 */ {1, 3, 1, 0, 1, 2, 1, 2, 1, 2, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  8, 0, 3, 4, 5, 9,10, 6, 1, 7, 1, 6, 1, 7, 2,11, 2, 7 */ {1, 0, 1, 3, 1, 2, 1, 2, 1, 2, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  2, 1,10, 4, 5, 9, 6, 3,11, 3, 6, 0, 7, 0, 6, 0, 7, 8 */ {1, 2, 1, 3, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11, 2, 1,10, 8, 4, 3, 5, 3, 4, 3, 5, 0, 9, 0, 5 */ {1, 3, 1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11, 4, 5, 9, 3, 8, 2,10, 2, 8, 0,10, 8,10, 0, 1 */ {1, 3, 1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 0, 3,10, 2, 1, 5,11, 6,11, 5, 9,11, 9, 7, 4, 7, 9 */ {1, 0, 1, 2, 1, 3, 1, 3, 1, 3, 1, 3},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_3[2][12][8] = {
    {
        /* 165: 0, ,2, , ,5, ,7: 11, 7, 6,12, 2,10,12,10, 5,12, 5, 4,12, 4, 8,12, 8, 3,12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2 */
        {0, 1, 0, 1, 1, 0, 1, 2},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10, 9, 5,12, 0, 9,12, 3, 0,12,11, 3,12, 6,11,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12 */
        {0, 1, 2, 1, 1, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7: 11, 7, 6,12, 5, 4,12, 4, 8,12, 8, 3,12, 3, 2,12, 2,10,12,10, 1,12, 1, 0,12, 0, 9,12, 9, 5 */
        {0, 1, 0, 1, 1, 0, 1, 2},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10,12, 3, 0,12, 0, 9,12, 9, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3 */
        {0, 1, 2, 1, 1, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  8, 3, 0,11, 7,12, 2,11,12, 1, 2,12, 9, 1,12, 4, 9,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12 */
        {0, 1, 2, 1, 1, 2, 1, 2},
        /* 165: 0, ,2, , ,5, ,7: 11, 7, 6, 5, 4,12,10, 5,12, 2,10,12, 3, 2,12, 8, 3,12, 0, 8,12, 1, 0,12, 9, 1,12, 4, 9,12 */
        {0, 1, 0, 1, 1, 0, 1, 2},
        /* 165: 0, ,2, , ,5, ,7:  8, 3, 0, 1, 2,12, 9, 1,12, 4, 9,12, 7, 4,12,11, 7,12, 6,11,12, 5, 6,12,10, 5,12, 2,10,12 */
        {0, 1, 2, 1, 1, 2, 1, 2},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4,12, 0, 8,12, 8, 7,12, 7, 6,12, 6,10,12,10, 1,12, 1, 2,12, 2,11,12,11, 3,12, 3, 0 */
        {0, 1, 0, 1, 1, 2, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4,12, 7, 6,12, 6,10,12,10, 1,12, 1, 0,12, 0, 8,12, 8, 3,12, 3, 2,12, 2,11,12,11, 7 */
        {0, 1, 0, 1, 1, 2, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  8, 3, 0,12, 1, 2,12, 2,11,12,11, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6,10,12,10, 1 */
        {0, 1, 2, 1, 1, 2, 1, 2},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4, 7, 6,12, 8, 7,12, 0, 8,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12,11, 3,12, 6,11,12 */
        {0, 1, 0, 1, 1, 2, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10, 3, 0,12,11, 3,12, 6,11,12, 5, 6,12, 9, 5,12, 4, 9,12, 7, 4,12, 8, 7,12, 0, 8,12 */
        {0, 1, 2, 1, 1, 0, 1, 0},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  8, 4, 7,12, 3,11,12,11, 6,12, 6, 5,12, 5, 9,12, 9, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3 */
        {0, 1, 0, 1, 2, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11,10, 6,12, 1,10,12, 0, 1,12, 8, 0,12, 7, 8,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12 */
        {0, 1, 0, 2, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 4, 7,12, 6, 5,12, 5, 9,12, 9, 0,12, 0, 3,12, 3,11,12,11, 2,12, 2, 1,12, 1,10,12,10, 6 */
        {0, 1, 0, 1, 2, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11,12, 0, 1,12, 1,10,12,10, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 9,12, 9, 0 */
        {0, 1, 0, 2, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  0, 1, 9, 8, 4,12, 3, 8,12, 2, 3,12,10, 2,12, 5,10,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12 */
        {0, 1, 0, 2, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 4, 7, 6, 5,12,11, 6,12, 3,11,12, 0, 3,12, 9, 0,12, 1, 9,12, 2, 1,12,10, 2,12, 5,10,12 */
        {0, 1, 0, 1, 2, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  9, 0, 1, 2, 3,12,10, 2,12, 5,10,12, 4, 5,12, 8, 4,12, 7, 8,12, 6, 7,12,11, 6,12, 3,11,12 */
        {0, 1, 0, 2, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10,12, 1, 9,12, 9, 4,12, 4, 7,12, 7,11,12,11, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1 */
        {0, 1, 0, 1, 1, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10,12, 4, 7,12, 7,11,12,11, 2,12, 2, 1,12, 1, 9,12, 9, 0,12, 0, 3,12, 3, 8,12, 8, 4 */
        {0, 1, 0, 1, 1, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  9, 0, 1,12, 2, 3,12, 3, 8,12, 8, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7,11,12,11, 2 */
        {0, 1, 0, 2, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 4, 7,12, 9, 4,12, 1, 9,12, 2, 1,12,11, 2,12, 3,11,12, 0, 3,12, 8, 0,12, 7, 8,12 */
        {0, 1, 0, 1, 1, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11, 0, 1,12, 8, 0,12, 7, 8,12, 6, 7,12,10, 6,12, 5,10,12, 4, 5,12, 9, 4,12, 1, 9,12 */
        {0, 1, 0, 2, 1, 0, 1, 0},
    },
};
static const signed char triangleRegion13_3[2][12][20] = {
    {
        /* 165: 0, ,2, , ,5, ,7: 11, 7, 6,12, 2,10,12,10, 5,12, 5, 4,12, 4, 8,12, 8, 3,12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10, 9, 5,12, 0, 9,12, 3, 0,12,11, 3,12, 6,11,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7: 11, 7, 6,12, 5, 4,12, 4, 8,12, 8, 3,12, 3, 2,12, 2,10,12,10, 1,12, 1, 0,12, 0, 9,12, 9, 5 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10,12, 3, 0,12, 0, 9,12, 9, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  8, 3, 0,11, 7,12, 2,11,12, 1, 2,12, 9, 1,12, 4, 9,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12 */
        {0, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1},
        /* 165: 0, ,2, , ,5, ,7: 11, 7, 6, 5, 4,12,10, 5,12, 2,10,12, 3, 2,12, 8, 3,12, 0, 8,12, 1, 0,12, 9, 1,12, 4, 9,12 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  8, 3, 0, 1, 2,12, 9, 1,12, 4, 9,12, 7, 4,12,11, 7,12, 6,11,12, 5, 6,12,10, 5,12, 2,10,12 */
        {0, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4,12, 0, 8,12, 8, 7,12, 7, 6,12, 6,10,12,10, 1,12, 1, 2,12, 2,11,12,11, 3,12, 3, 0 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4,12, 7, 6,12, 6,10,12,10, 1,12, 1, 0,12, 0, 8,12, 8, 3,12, 3, 2,12, 2,11,12,11, 7 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  8, 3, 0,12, 1, 2,12, 2,11,12,11, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6,10,12,10, 1 */
        {0, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1},
        /* 165: 0, ,2, , ,5, ,7:  9, 5, 4, 7, 6,12, 8, 7,12, 0, 8,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12,11, 3,12, 6,11,12 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10, 3, 0,12,11, 3,12, 6,11,12, 5, 6,12, 9, 5,12, 4, 9,12, 7, 4,12, 8, 7,12, 0, 8,12 */
        {2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  8, 4, 7,12, 3,11,12,11, 6,12, 6, 5,12, 5, 9,12, 9, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11,10, 6,12, 1,10,12, 0, 1,12, 8, 0,12, 7, 8,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 4, 7,12, 6, 5,12, 5, 9,12, 9, 0,12, 0, 3,12, 3,11,12,11, 2,12, 2, 1,12, 1,10,12,10, 6 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11,12, 0, 1,12, 1,10,12,10, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 9,12, 9, 0 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  0, 1, 9, 8, 4,12, 3, 8,12, 2, 3,12,10, 2,12, 5,10,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12 */
        {1, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 4, 7, 6, 5,12,11, 6,12, 3,11,12, 0, 3,12, 9, 0,12, 1, 9,12, 2, 1,12,10, 2,12, 5,10,12 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  9, 0, 1, 2, 3,12,10, 2,12, 5,10,12, 4, 5,12, 8, 4,12, 7, 8,12, 6, 7,12,11, 6,12, 3,11,12 */
        {1, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10,12, 1, 9,12, 9, 4,12, 4, 7,12, 7,11,12,11, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10,12, 4, 7,12, 7,11,12,11, 2,12, 2, 1,12, 1, 9,12, 9, 0,12, 0, 3,12, 3, 8,12, 8, 4 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  9, 0, 1,12, 2, 3,12, 3, 8,12, 8, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7,11,12,11, 2 */
        {1, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 4, 7,12, 9, 4,12, 1, 9,12, 2, 1,12,11, 2,12, 3,11,12, 0, 3,12, 8, 0,12, 7, 8,12 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11, 0, 1,12, 8, 0,12, 7, 8,12, 6, 7,12,10, 6,12, 5,10,12, 4, 5,12, 9, 4,12, 1, 9,12 */
        {2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_3_[2][12][8] = {
    {
        /* 165: 0, ,2, , ,5, ,7:  3, 2,11, 8, 7,12, 0, 8,12, 1, 0,12,10, 1,12, 6,10,12, 5, 6,12, 9, 5,12, 4, 9,12, 7, 4,12 */
        {0, 1, 0, 2, 1, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  5, 6,10,12, 2,11,12,11, 7,12, 7, 4,12, 4, 9,12, 9, 1,12, 1, 0,12, 0, 8,12, 8, 3,12, 3, 2 */
        {0, 1, 0, 1, 1, 0, 2, 0},
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6,12, 7, 4,12, 4, 9,12, 9, 1,12, 1, 2,12, 2,11,12,11, 3,12, 3, 0,12, 0, 8,12, 8, 7 */
        {0, 1, 0, 1, 1, 0, 2, 0},
        /* 165: 0, ,2, , ,5, ,7: 11, 3, 2,12, 1, 0,12, 0, 8,12, 8, 7,12, 7, 6,12, 6,10,12,10, 5,12, 5, 4,12, 4, 9,12, 9, 1 */
        {0, 1, 0, 2, 1, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  7, 4, 8,11, 3,12, 6,11,12, 5, 6,12, 9, 5,12, 0, 9,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12 */
        {0, 1, 0, 1, 2, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  7, 4, 8, 5, 6,12, 9, 5,12, 0, 9,12, 3, 0,12,11, 3,12, 2,11,12, 1, 2,12,10, 1,12, 6,10,12 */
        {0, 1, 0, 1, 2, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7: 11, 3, 2, 1, 0,12,10, 1,12, 6,10,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 9, 5,12, 0, 9,12 */
        {0, 1, 0, 2, 1, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  1, 0, 9,12, 4, 8,12, 8, 3,12, 3, 2,12, 2,10,12,10, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4 */
        {0, 1, 0, 2, 2, 0, 2, 0},
        /* 165: 0, ,2, , ,5, ,7:  7, 4, 8,12, 5, 6,12, 6,11,12,11, 3,12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2,12, 2,10,12,10, 5 */
        {0, 1, 0, 1, 2, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  1, 0, 9,12, 3, 2,12, 2,10,12,10, 5,12, 5, 4,12, 4, 8,12, 8, 7,12, 7, 6,12, 6,11,12,11, 3 */
        {0, 1, 0, 2, 2, 0, 2, 0},
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6, 7, 4,12,11, 7,12, 2,11,12, 1, 2,12, 9, 1,12, 0, 9,12, 3, 0,12, 8, 3,12, 4, 8,12 */
        {0, 1, 0, 1, 1, 0, 2, 0},
        /* 165: 0, ,2, , ,5, ,7:  9, 1, 0, 3, 2,12, 8, 3,12, 4, 8,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12,11, 7,12, 2,11,12 */
        {0, 1, 0, 2, 2, 0, 2, 0},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  0, 3, 8, 9, 4,12, 1, 9,12, 2, 1,12,11, 2,12, 7,11,12, 6, 7,12,10, 6,12, 5,10,12, 4, 5,12 */
        {0, 1, 2, 1, 1, 2, 1, 2},
        /*  90:  ,1, ,3,4, ,6, : 11, 6, 7,12, 3, 8,12, 8, 4,12, 4, 5,12, 5,10,12,10, 2,12, 2, 1,12, 1, 9,12, 9, 0,12, 0, 3 */
        {0, 1, 0, 1, 1, 0, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11,12, 4, 5,12, 5,10,12,10, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1,12, 1, 9,12, 9, 4 */
        {0, 1, 0, 1, 1, 0, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  8, 0, 3,12, 2, 1,12, 1, 9,12, 9, 4,12, 4, 7,12, 7,11,12,11, 6,12, 6, 5,12, 5,10,12,10, 2 */
        {0, 1, 2, 1, 1, 2, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  4, 5, 9, 8, 0,12, 7, 8,12, 6, 7,12,10, 6,12, 1,10,12, 2, 1,12,11, 2,12, 3,11,12, 0, 3,12 */
        {0, 1, 0, 1, 1, 2, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  4, 5, 9, 6, 7,12,10, 6,12, 1,10,12, 0, 1,12, 8, 0,12, 3, 8,12, 2, 3,12,11, 2,12, 7,11,12 */
        {0, 1, 0, 1, 1, 2, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 0, 3, 2, 1,12,11, 2,12, 7,11,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12,10, 6,12, 1,10,12 */
        {0, 1, 2, 1, 1, 2, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  2, 1,10,12, 5, 9,12, 9, 0,12, 0, 3,12, 3,11,12,11, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5 */
        {0, 1, 2, 1, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  4, 5, 9,12, 6, 7,12, 7, 8,12, 8, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3,12, 3,11,12,11, 6 */
        {0, 1, 0, 1, 1, 2, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 1,10,12, 0, 3,12, 3,11,12,11, 6,12, 6, 5,12, 5, 9,12, 9, 4,12, 4, 7,12, 7, 8,12, 8, 0 */
        {0, 1, 2, 1, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11, 4, 5,12, 8, 4,12, 3, 8,12, 2, 3,12,10, 2,12, 1,10,12, 0, 1,12, 9, 0,12, 5, 9,12 */
        {0, 1, 0, 1, 1, 0, 1, 2},
        /*  90:  ,1, ,3,4, ,6, : 10, 2, 1, 0, 3,12, 9, 0,12, 5, 9,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12 */
        {0, 1, 2, 1, 1, 0, 1, 0},
    },
};
static const signed char triangleRegion13_3_[2][12][20] = {
    {
        /* 165: 0, ,2, , ,5, ,7:  3, 2,11, 8, 7,12, 0, 8,12, 1, 0,12,10, 1,12, 6,10,12, 5, 6,12, 9, 5,12, 4, 9,12, 7, 4,12 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  5, 6,10,12, 2,11,12,11, 7,12, 7, 4,12, 4, 9,12, 9, 1,12, 1, 0,12, 0, 8,12, 8, 3,12, 3, 2 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6,12, 7, 4,12, 4, 9,12, 9, 1,12, 1, 2,12, 2,11,12,11, 3,12, 3, 0,12, 0, 8,12, 8, 7 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7: 11, 3, 2,12, 1, 0,12, 0, 8,12, 8, 7,12, 7, 6,12, 6,10,12,10, 5,12, 5, 4,12, 4, 9,12, 9, 1 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  7, 4, 8,11, 3,12, 6,11,12, 5, 6,12, 9, 5,12, 0, 9,12, 1, 0,12,10, 1,12, 2,10,12, 3, 2,12 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  7, 4, 8, 5, 6,12, 9, 5,12, 0, 9,12, 3, 0,12,11, 3,12, 2,11,12, 1, 2,12,10, 1,12, 6,10,12 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7: 11, 3, 2, 1, 0,12,10, 1,12, 6,10,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 9, 5,12, 0, 9,12 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  1, 0, 9,12, 4, 8,12, 8, 3,12, 3, 2,12, 2,10,12,10, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4 */
        {0, 1, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
        /* 165: 0, ,2, , ,5, ,7:  7, 4, 8,12, 5, 6,12, 6,11,12,11, 3,12, 3, 0,12, 0, 9,12, 9, 1,12, 1, 2,12, 2,10,12,10, 5 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  1, 0, 9,12, 3, 2,12, 2,10,12,10, 5,12, 5, 4,12, 4, 8,12, 8, 7,12, 7, 6,12, 6,11,12,11, 3 */
        {0, 1, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
        /* 165: 0, ,2, , ,5, ,7: 10, 5, 6, 7, 4,12,11, 7,12, 2,11,12, 1, 2,12, 9, 1,12, 0, 9,12, 3, 0,12, 8, 3,12, 4, 8,12 */
        {0, 2, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  9, 1, 0, 3, 2,12, 8, 3,12, 4, 8,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12,11, 7,12, 2,11,12 */
        {0, 1, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2, 0, 2},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  0, 3, 8, 9, 4,12, 1, 9,12, 2, 1,12,11, 2,12, 7,11,12, 6, 7,12,10, 6,12, 5,10,12, 4, 5,12 */
        {1, 0, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2},
        /*  90:  ,1, ,3,4, ,6, : 11, 6, 7,12, 3, 8,12, 8, 4,12, 4, 5,12, 5,10,12,10, 2,12, 2, 1,12, 1, 9,12, 9, 0,12, 0, 3 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11,12, 4, 5,12, 5,10,12,10, 2,12, 2, 3,12, 3, 8,12, 8, 0,12, 0, 1,12, 1, 9,12, 9, 4 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 0, 3,12, 2, 1,12, 1, 9,12, 9, 4,12, 4, 7,12, 7,11,12,11, 6,12, 6, 5,12, 5,10,12,10, 2 */
        {1, 0, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  4, 5, 9, 8, 0,12, 7, 8,12, 6, 7,12,10, 6,12, 1,10,12, 2, 1,12,11, 2,12, 3,11,12, 0, 3,12 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  4, 5, 9, 6, 7,12,10, 6,12, 1,10,12, 0, 1,12, 8, 0,12, 3, 8,12, 2, 3,12,11, 2,12, 7,11,12 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 0, 3, 2, 1,12,11, 2,12, 7,11,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12,10, 6,12, 1,10,12 */
        {1, 0, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2},
        /*  90:  ,1, ,3,4, ,6, :  2, 1,10,12, 5, 9,12, 9, 0,12, 0, 3,12, 3,11,12,11, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  4, 5, 9,12, 6, 7,12, 7, 8,12, 8, 0,12, 0, 1,12, 1,10,12,10, 2,12, 2, 3,12, 3,11,12,11, 6 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 1,10,12, 0, 3,12, 3,11,12,11, 6,12, 6, 5,12, 5, 9,12, 9, 4,12, 4, 7,12, 7, 8,12, 8, 0 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11, 4, 5,12, 8, 4,12, 3, 8,12, 2, 3,12,10, 2,12, 1,10,12, 0, 1,12, 9, 0,12, 5, 9,12 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, : 10, 2, 1, 0, 3,12, 9, 0,12, 5, 9,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12 */
        {1, 2, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_4[2][4][8] = {
    {
        /* 165: 0, ,2, , ,5, ,7: 12, 2,10,12,10, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3,12, 3, 0,12, 0, 9,12, 9,
           1,12, 1, 2 */
        {0, 1, 0, 1, 1, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7: 11, 3,12, 6,11,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 9, 5,12, 0, 9,12, 1, 0,12,10, 1,12,
           2,10,12, 3, 2,12 */
        {0, 1, 0, 1, 1, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  9, 1,12, 4, 9,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12,11, 7,12, 2,11,12, 3, 2,12, 8, 3,12, 0,
           8,12, 1, 0,12 */
        {0, 1, 0, 1, 1, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7: 12, 0, 8,12, 8, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6,10,12,10, 1,12, 1, 2,12, 2,11,12,11,
           3,12, 3, 0 */
        {0, 1, 0, 1, 1, 0, 1, 0},
    },
    {
        /*  90:  ,1, ,3,4, ,6, : 12, 3,11,12,11, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 9,12, 9, 0,12, 0, 1,12, 1,10,12,10,
           2,12, 2, 3 */
        {0, 1, 0, 1, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 0,12, 7, 8,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12,10, 6,12, 1,10,12, 2, 1,12,11, 2,12,
           3,11,12, 0, 3,12 */
        {0, 1, 0, 1, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, : 10, 2,12, 5,10,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12, 0, 3,12, 9, 0,12, 1,
           9,12, 2, 1,12 */
        {0, 1, 0, 1, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, : 12, 1, 9,12, 9, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7,11,12,11, 2,12, 2, 3,12, 3, 8,12, 8,
           0,12, 0, 1 */
        {0, 1, 0, 1, 1, 0, 1, 0},
    },
};
static const signed char triangleRegion13_4[2][4][24] = {
    {
        /* 165: 0, ,2, , ,5, ,7: 12, 2,10,12,10, 5,12, 5, 6,12, 6,11,12,11, 7,12, 7, 4,12, 4, 8,12, 8, 3,12, 3, 0,12, 0, 9,12, 9,
           1,12, 1, 2 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7: 11, 3,12, 6,11,12, 7, 6,12, 8, 7,12, 4, 8,12, 5, 4,12, 9, 5,12, 0, 9,12, 1, 0,12,10, 1,12,
           2,10,12, 3, 2,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  9, 1,12, 4, 9,12, 5, 4,12,10, 5,12, 6,10,12, 7, 6,12,11, 7,12, 2,11,12, 3, 2,12, 8, 3,12, 0,
           8,12, 1, 0,12 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7: 12, 0, 8,12, 8, 7,12, 7, 4,12, 4, 9,12, 9, 5,12, 5, 6,12, 6,10,12,10, 1,12, 1, 2,12, 2,11,12,11,
           3,12, 3, 0 */
        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /*  90:  ,1, ,3,4, ,6, : 12, 3,11,12,11, 6,12, 6, 7,12, 7, 8,12, 8, 4,12, 4, 5,12, 5, 9,12, 9, 0,12, 0, 1,12, 1,10,12,10,
           2,12, 2, 3 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  8, 0,12, 7, 8,12, 4, 7,12, 9, 4,12, 5, 9,12, 6, 5,12,10, 6,12, 1,10,12, 2, 1,12,11, 2,12,
           3,11,12, 0, 3,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, : 10, 2,12, 5,10,12, 6, 5,12,11, 6,12, 7,11,12, 4, 7,12, 8, 4,12, 3, 8,12, 0, 3,12, 9, 0,12, 1,
           9,12, 2, 1,12 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, : 12, 1, 9,12, 9, 4,12, 4, 5,12, 5,10,12,10, 6,12, 6, 7,12, 7,11,12,11, 2,12, 2, 3,12, 3, 8,12, 8,
           0,12, 0, 1 */
        {1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_5_1[2][4][8] = {
    {
        /* 165: 0, ,2, , ,5, ,7:  7, 6,11, 1, 0, 9,10, 3, 2, 3,10, 5, 3, 5, 8, 4, 8, 5 */ {0, 1, 0, 2, 2, 0, 2, 3},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10, 7, 4, 8, 3, 0,11, 6,11, 0, 9, 6, 0, 6, 9, 5 */ {0, 1, 2, 1, 3, 0, 1, 0},
        /* 165: 0, ,2, , ,5, ,7:  3, 0, 8, 5, 6,10, 1, 2, 9, 4, 9, 2,11, 4, 2, 4,11, 7 */ {0, 1, 2, 1, 1, 2, 3, 2},
        /* 165: 0, ,2, , ,5, ,7:  5, 4, 9, 3, 2,11, 8, 1, 0, 1, 8, 7, 1, 7,10, 6,10, 7 */ {0, 1, 0, 2, 1, 3, 1, 0},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  4, 7, 8, 2, 1,10,11, 0, 3, 0,11, 6, 0, 6, 9, 5, 9, 6 */ {0, 1, 2, 1, 3, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11, 4, 5, 9, 0, 1, 8, 7, 8, 1,10, 7, 1, 7,10, 6 */ {0, 1, 0, 2, 1, 3, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  0, 1, 9, 6, 7,11, 2, 3,10, 5,10, 3, 8, 5, 3, 5, 8, 4 */ {0, 1, 0, 2, 2, 0, 2, 3},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 0, 3, 8, 9, 2, 1, 2, 9, 4, 2, 4,11, 7,11, 4 */ {0, 1, 2, 1, 1, 2, 3, 2},
    },
};
static const signed char triangleRegion13_5_1[2][4][12] = {
    {
        /* 165: 0, ,2, , ,5, ,7:  7, 6,11, 1, 0, 9,10, 3, 2, 3,10, 5, 3, 5, 8, 4, 8, 5 */ {3, 2, 0, 1, 0, 2, 0, 2, 0, 2, 0, 2},
        /* 165: 0, ,2, , ,5, ,7:  1, 2,10, 7, 4, 8, 3, 0,11, 6,11, 0, 9, 6, 0, 6, 9, 5 */ {2, 1, 0, 3, 0, 1, 0, 1, 0, 1, 0, 1},
        /* 165: 0, ,2, , ,5, ,7:  3, 0, 8, 5, 6,10, 1, 2, 9, 4, 9, 2,11, 4, 2, 4,11, 7 */ {0, 1, 2, 3, 2, 1, 2, 1, 2, 1, 2, 1},
        /* 165: 0, ,2, , ,5, ,7:  5, 4, 9, 3, 2,11, 8, 1, 0, 1, 8, 7, 1, 7,10, 6,10, 7 */ {3, 1, 0, 2, 0, 1, 0, 1, 0, 1, 0, 1},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  4, 7, 8, 2, 1,10,11, 0, 3, 0,11, 6, 0, 6, 9, 5, 9, 6 */ {3, 0, 1, 2, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  2, 3,11, 4, 5, 9, 0, 1, 8, 7, 8, 1,10, 7, 1, 7,10, 6 */ {2, 0, 1, 3, 1, 0, 1, 0, 1, 0, 1, 0},
        /*  90:  ,1, ,3,4, ,6, :  0, 1, 9, 6, 7,11, 2, 3,10, 5,10, 3, 8, 5, 3, 5, 8, 4 */ {1, 0, 2, 3, 2, 0, 2, 0, 2, 0, 2, 0},
        /*  90:  ,1, ,3,4, ,6, :  6, 5,10, 0, 3, 8, 9, 2, 1, 2, 9, 4, 2, 4,11, 7,11, 4 */ {3, 2, 1, 0, 1, 2, 1, 2, 1, 2, 1, 2},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion13_5_2[2][4][8] = {
    {
        /* 165: 0, ,2, , ,5, ,7:  1, 0, 9, 7, 4, 8, 7, 8, 3, 7, 3,11, 2,11, 3,11, 2,10,11,10, 6, 5, 6,10, 6, 5, 7, 4, 7, 5 */
        {0, 1, 0, 2, 3, 0, 4, 0},
        /* 165: 0, ,2, , ,5, ,7:  7, 4, 8,11, 3, 2, 6,11, 2,10, 6, 2, 6,10, 5, 9, 5,10, 1, 9,10, 9, 1, 0, 2, 0, 1, 0, 2, 3 */
        {0, 1, 0, 2, 3, 0, 4, 0},
        /* 165: 0, ,2, , ,5, ,7:  5, 6,10, 9, 1, 0, 4, 9, 0, 8, 4, 0, 4, 8, 7,11, 7, 8, 3,11, 8,11, 3, 2, 0, 2, 3, 2, 0, 1 */
        {0, 1, 0, 2, 3, 0, 4, 0},
        /* 165: 0, ,2, , ,5, ,7:  3, 2,11, 5, 6,10, 5,10, 1, 5, 1, 9, 0, 9, 1, 9, 0, 8, 9, 8, 4, 4, 8, 7, 4, 7, 5, 6, 5, 7 */
        {0, 1, 0, 2, 3, 0, 4, 0},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  2, 1,10, 4, 5, 9, 4, 9, 0, 4, 0, 8, 3, 8, 0, 8, 3,11, 8,11, 7, 6, 7,11, 7, 6, 4, 5, 4, 6 */
        {0, 1, 2, 1, 1, 3, 1, 4},
        /*  90:  ,1, ,3,4, ,6, :  4, 5, 9, 8, 0, 3, 7, 8, 3,11, 7, 3, 7,11, 6,10, 6,11, 2,10,11,10, 2, 1, 3, 1, 2, 1, 3, 0 */
        {0, 1, 2, 1, 1, 3, 1, 4},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11,10, 2, 1, 5,10, 1, 9, 5, 1, 5, 9, 4, 8, 4, 9, 0, 8, 9, 8, 0, 3, 1, 3, 0, 3, 1, 2 */
        {0, 1, 2, 1, 1, 3, 1, 4},
        /*  90:  ,1, ,3,4, ,6, :  0, 3, 8, 6, 7,11, 6,11, 2, 6, 2,10, 1,10, 2,10, 1, 9,10, 9, 5, 5, 9, 4, 5, 4, 6, 7, 6, 4 */
        {0, 1, 2, 1, 1, 3, 1, 4},
    },
};
static const signed char triangleRegion13_5_2[2][4][20] = {
    {
        /* 165: 0, ,2, , ,5, ,7:  1, 0, 9, 7, 4, 8, 7, 8, 3, 7, 3,11, 2,11, 3,11, 2,10,11,10, 6, 5, 6,10, 6, 5, 7, 4, 7, 5 */
        {0, 1, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0},
        /* 165: 0, ,2, , ,5, ,7:  7, 4, 8,11, 3, 2, 6,11, 2,10, 6, 2, 6,10, 5, 9, 5,10, 1, 9,10, 9, 1, 0, 2, 0, 1, 0, 2, 3 */
        {0, 3, 0, 2, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0},
        /* 165: 0, ,2, , ,5, ,7:  5, 6,10, 9, 1, 0, 4, 9, 0, 8, 4, 0, 4, 8, 7,11, 7, 8, 3,11, 8,11, 3, 2, 0, 2, 3, 2, 0, 1 */
        {0, 4, 0, 1, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0},
        /* 165: 0, ,2, , ,5, ,7:  3, 2,11, 5, 6,10, 5,10, 1, 5, 1, 9, 0, 9, 1, 9, 0, 8, 9, 8, 4, 4, 8, 7, 4, 7, 5, 6, 5, 7 */
        {0, 2, 0, 4, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0},
    },
    {
        /*  90:  ,1, ,3,4, ,6, :  2, 1,10, 4, 5, 9, 4, 9, 0, 4, 0, 8, 3, 8, 0, 8, 3,11, 8,11, 7, 6, 7,11, 7, 6, 4, 5, 4, 6 */
        {1, 2, 1, 3, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 4, 1, 1, 1, 1},
        /*  90:  ,1, ,3,4, ,6, :  4, 5, 9, 8, 0, 3, 7, 8, 3,11, 7, 3, 7,11, 6,10, 6,11, 2,10,11,10, 2, 1, 3, 1, 2, 1, 3, 0 */
        {1, 3, 1, 0, 1, 1, 1, 1, 1, 4, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1},
        /*  90:  ,1, ,3,4, ,6, :  6, 7,11,10, 2, 1, 5,10, 1, 9, 5, 1, 5, 9, 4, 8, 4, 9, 0, 8, 9, 8, 0, 3, 1, 3, 0, 3, 1, 2 */
        {1, 4, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1},
        /*  90:  ,1, ,3,4, ,6, :  0, 3, 8, 6, 7,11, 6,11, 2, 6, 2,10, 1,10, 2,10, 1, 9,10, 9, 5, 5, 9, 4, 5, 4, 6, 7, 6, 4 */
        {1, 0, 1, 4, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1, 1},
    },
};
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
static const signed char vertexRegion14[12][8] = {
    /*  71: 0,1,2, , , ,6, :  5, 9, 8, 5, 8, 2, 5, 2, 6, 3, 2, 8 */ {0, 0, 0, 1, 1, 1, 0, 1},
    /*  43: 0,1, ,3, ,5, , :  2, 1, 5, 2, 5, 8, 2, 8,11, 4, 8, 5 */ {0, 0, 1, 0, 1, 0, 1, 1},
    /* 147: 0,1, , ,4, , ,7:  9, 4, 6, 9, 6, 3, 9, 3, 1,11, 3, 6 */ {0, 0, 1, 1, 0, 1, 1, 0},
    /*  29: 0, ,2,3,4, , , :  1,11,10, 1, 4,11, 1, 0, 4, 7,11, 4 */ {0, 1, 0, 0, 0, 1, 1, 1},
    /* 201: 0, , ,3, , ,6,7:  8, 2, 0, 8, 5, 2, 8, 7, 5,10, 2, 5 */ {0, 1, 1, 0, 1, 1, 0, 0},
    /* 113: 0, , , ,4,5,6, :  0, 7, 3, 0,10, 7, 0, 9,10, 6, 7,10 */ {0, 1, 1, 1, 0, 0, 0, 1},
    /* 142:  ,1,2,3, , , ,7:  0, 3, 7, 0, 7,10, 0,10, 9, 6,10, 7 */ {0, 1, 1, 1, 0, 0, 0, 1},
    /*  54:  ,1,2, ,4,5, , :  8, 0, 2, 8, 2, 5, 8, 5, 7,10, 5, 2 */ {0, 1, 1, 0, 1, 1, 0, 0},
    /* 226:  ,1, , , ,5,6,7:  1,10,11, 1,11, 4, 1, 4, 0, 7, 4,11 */ {0, 1, 0, 0, 0, 1, 1, 1},
    /* 108:  , ,2,3, ,5,6, :  9, 6, 4, 9, 3, 6, 9, 1, 3,11, 6, 3 */ {0, 0, 1, 1, 0, 1, 1, 0},
    /* 212:  , ,2, ,4, ,6,7:  2, 5, 1, 2, 8, 5, 2,11, 8, 4, 5, 8 */ {0, 0, 1, 0, 1, 0, 1, 1},
    /* 184:  , , ,3,4,5, ,7:  5, 8, 9, 5, 2, 8, 5, 6, 2, 3, 8, 2 */ {0, 0, 0, 1, 1, 1, 0, 1},
};
static const signed char triangleRegion14[12][8] = {
    /*  71: 0,1,2, , , ,6, :  5, 9, 8, 5, 8, 2, 5, 2, 6, 3, 2, 8 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  43: 0,1, ,3, ,5, , :  2, 1, 5, 2, 5, 8, 2, 8,11, 4, 8, 5 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 147: 0,1, , ,4, , ,7:  9, 4, 6, 9, 6, 3, 9, 3, 1,11, 3, 6 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /*  29: 0, ,2,3,4, , , :  1,11,10, 1, 4,11, 1, 0, 4, 7,11, 4 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 201: 0, , ,3, , ,6,7:  8, 2, 0, 8, 5, 2, 8, 7, 5,10, 2, 5 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 113: 0, , , ,4,5,6, :  0, 7, 3, 0,10, 7, 0, 9,10, 6, 7,10 */ {0, 1, 0, 1, 0, 1, 0, 1},
    /* 142:  ,1,2,3, , , ,7:  0, 3, 7, 0, 7,10, 0,10, 9, 6,10, 7 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /*  54:  ,1,2, ,4,5, , :  8, 0, 2, 8, 2, 5, 8, 5, 7,10, 5, 2 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 226:  ,1, , , ,5,6,7:  1,10,11, 1,11, 4, 1, 4, 0, 7, 4,11 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 108:  , ,2,3, ,5,6, :  9, 6, 4, 9, 3, 6, 9, 1, 3,11, 6, 3 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 212:  , ,2, ,4, ,6,7:  2, 5, 1, 2, 8, 5, 2,11, 8, 4, 5, 8 */ {1, 0, 1, 0, 1, 0, 1, 0},
    /* 184:  , , ,3,4,5, ,7:  5, 8, 9, 5, 2, 8, 5, 6, 2, 3, 8, 2 */ {1, 0, 1, 0, 1, 0, 1, 0},
};
//-----------------------------------------------------------------------------
